/*
 * gaia - opensource 3D interface to the planet
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "gefetch_internal.h"

/**
 * Convert (x, y, level) triple to string used in Google Earth's URLs
 *
 * @param buffer buffer to save zero-terminated string into
 * @param buffersize buffer length
 */
gefetch_error gefetch_coords_to_string(int x, int y, int level, char *buffer, size_t buffersize) {
	char *i = buffer;

	/* form index string */
	for (; level >= 0 && i < buffer + buffersize; level--) {
		int middle = 1 << level;

		if (x < middle && y < middle)
			*i++ = '0';
		else if (x >= middle && y < middle)
			*i++ = '1';
		else if (x < middle && y >= middle)
			*i++ = '3';
		else
			*i++ = '2';

		x %= middle;    
		y %= middle;
	}

	if (i > buffer + buffersize - 1)
		return GEFETCH_SMALL_BUFFER;

	*i = '\0';

	return GEFETCH_OK;
}

/**
 * Fetch arbitary URI from Google Earth's servers
 *
 * @param uri URI requested (usually starting with /)
 */
gefetch_error gefetch_fetch_uri(gefetch *handle, char *uri) {
	/* form full url */
	char urlbuf[1024];
	if (snprintf(urlbuf, sizeof(urlbuf), "%s%s", handle->url, uri) >= sizeof(urlbuf))
		return GEFETCH_SMALL_BUFFER;

	/* fetch */
	return gefetch_fetch(handle, urlbuf);
}

/**
 * Internal function to fetch data from Google Earth's servers
 *
 * Given full URL, does all required HTTP requests to server to
 * get requested data. Automatically fetches dbRoot and does
 * authentication if needed. Also decrypts data.
 *
 * @param URL full URL for request
 */
gefetch_error gefetch_fetch(gefetch *handle, char *url) {
	gefetch_error result;
	long code;

	/* ensure we have dbroot */
	if (handle->dbrootdata == 0) {
		char dbrooturlbuf[1024];
		if (snprintf(dbrooturlbuf, sizeof(dbrooturlbuf), "%s/dbRoot.v5", handle->url) >= sizeof(dbrooturlbuf))
			return GEFETCH_SMALL_BUFFER;

		/* if we have no dbRoot, there's not point to continue */
		if ((result = gefetch_do_http_request(handle, METHOD_GET, dbrooturlbuf, 0, 0, &code)) != GEFETCH_OK)
			return result;
		       
		if (code != 200)
			return GEFETCH_FETCH_FAILED;

		handle->dbrootsize = handle->currentsize;
		handle->dbrootdata = handle->currentdata;

		handle->currentsize = 0;
		handle->currentdata = 0;
	}

	/* try to get a file */
	if ((result = gefetch_do_http_request(handle, METHOD_GET, url, 0, 0, &code)) != GEFETCH_OK)
		return result;

	if (code == 403) {
		/* if auth is requested, authenticate and retry */
		if ((result = gefetch_authenticate(handle)) != GEFETCH_OK)
			return result;

		if ((result = gefetch_do_http_request(handle, METHOD_GET, url, 0, 0, &code)) != GEFETCH_OK)
			return result;
	}

	/* success */
	if (code == 200) {
		if ((result = gefetch_decipher_current_file(handle)) != GEFETCH_OK)
			return result;

		return GEFETCH_OK;
	}

	/* still unauthenticated */
	if (code == 403)
		return GEFETCH_AUTH_FAILED;

	/* not found */
	if (code == 404)
		return GEFETCH_NOT_FOUND;

	/* other result codes */
	return GEFETCH_FETCH_FAILED;
}
