/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "mmap.h"
#include "list_ext.h"


#ifndef MAP_FILE
# define MAP_OPTIONS MAP_SHARED
#else
# define MAP_OPTIONS MAP_SHARED | MAP_FILE
#endif


typedef struct {
	struct list_head base;
	
	dev_t  dev;
	ino_t  ino;
	size_t size;

	char  *mmap;

	int    use_count;
//	size_t size;
//	int    available;
//	int    times_used;
} cherokee_mmap_entry_t ;

#define MMAP_ENTRY(e) ((cherokee_mmap_entry_t *)(e))


ret_t 
cherokee_mmap_new  (cherokee_mmap_t **mmap)
{
	int i;
	CHEROKEE_NEW_STRUCT(n,mmap);

	/* Init
	 */
	for (i=0; i<MMAP_HASH_SIZE; i++) {
		INIT_LIST_HEAD(&n->hash[i]);
	}
	
	/* Return it
	 */
	*mmap = n;
	   
	return ret_ok;
}


static void
free_item (void *item)
{
	if (MMAP_ENTRY(item)->mmap != NULL) 
	{
		munmap (MMAP_ENTRY(item)->mmap, 
			MMAP_ENTRY(item)->size);

		free (item);
	}
}


ret_t 
cherokee_mmap_free (cherokee_mmap_t *cmmap)
{
	int i;

	for (i=0; i<MMAP_HASH_SIZE; i++) {
		cherokee_list_free (&cmmap->hash[i], free_item);
	}
	
	free (mmap);
	return ret_ok;
}

static void
maybe_clean_up_table (cherokee_mmap_t *cmmap)
{
	if (cmmap->items >= MAX_ITEMS_MMAP_TABLE) {
		list_t *a, *b;
		unsigned long i, average = 0;
		cherokee_mmap_entry_t *entry;

		/* Calculate the average
		 */
		for (i=0; i<MMAP_HASH_SIZE; i++) {
			list_for_each (a, &cmmap->hash[i]) {
				entry = MMAP_ENTRY(LIST_ITEM_INFO(a));
				average += entry->use_count;
			}
		}
		
		average = average / i;

                /* Delete olders
		 */
		for (i=0; i<MMAP_HASH_SIZE; i++) {
			list_for_each_safe (a,b, &cmmap->hash[i]) {
				entry = MMAP_ENTRY(LIST_ITEM_INFO(a));

				if (entry->use_count < average) {
					list_del ((list_t *)a);
					free(a);
					free_item(entry);
				} else {
					entry->use_count = 0;
				}
			}
		}
	}
}


static cherokee_mmap_entry_t *
find_hash (cherokee_mmap_t *cmmap, struct stat *s)
{
	list_t *i;

	list_for_each (i, &cmmap->hash[MMAP_HASH_FUNC(s)]) {

		if ((s->st_dev  == MMAP_ENTRY(i)->dev) &&
		    (s->st_ino  == MMAP_ENTRY(i)->ino) &&
		    (s->st_size == MMAP_ENTRY(i)->size))
		{
			MMAP_ENTRY(i)->use_count++;
			return MMAP_ENTRY(i);
		}
	}

	return NULL;
}

static cherokee_mmap_entry_t *
find (cherokee_mmap_t *cmmap, int fd, struct stat *s) 
{
	cherokee_mmap_entry_t *entry;
	void *area;

	entry = find_hash (cmmap, s);
	if (entry != NULL) {
		return entry;
	}

	area = mmap (NULL,           /* void   *start  */
		     s->st_size,     /* size_t  length */
		     PROT_READ,      /* int     prot   */
		     MAP_OPTIONS,    /* int     flag   */
		     fd,             /* int     fd     */
		     0);             /* off_t   offset */

	if (area == MAP_FAILED) {
		return NULL;
	}

	{
		CHEROKEE_NEW_STRUCT (n, mmap_entry);

		n->dev       = 0;
		n->ino       = 0;
		n->size      = 0;
		n->mmap      = NULL;
		n->use_count = 0;
		n->size      = s->st_size;
		
		maybe_clean_up_table (cmmap);

		cherokee_list_add (&cmmap->hash[MMAP_HASH_FUNC(s)], n);
		return n;
	}

	return NULL;
}


void * 
cherokee_mmap_get (cherokee_mmap_t *cmmap, const char *filename)
{
	int fd;
	struct stat s;
	cherokee_mmap_entry_t *entry;

	fd = open (filename, O_RDONLY);
	if (fd == -1) {
		return NULL;
	}

	fstat(fd, &s);
	if (S_ISDIR(s.st_mode)) {
		return NULL;
	}

	entry = find (cmmap, fd, &s);
	close(fd);

	return entry->mmap;
}


void * 
cherokee_mmap_get_fast (cherokee_mmap_t *cmmap, int fd, struct stat *s)
{
	cherokee_mmap_entry_t *entry;

	entry = find (cmmap, fd, s);
	return entry->mmap;
}
