%{
/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"

#include <pwd.h>
#include <grp.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>

#include "mime.h"
#include "server.h"
#include "table.h"
#include "virtual_server.h"
#include "handler_table_entry.h"
#include "encoder.h"
#include "logger_table.h"


/* Define the parameter name of the yyparse() argument
 */
#define YYPARSE_PARAM server

%}


%union {
	   int   number;
	   char *string;
	   void *ptr;

	   struct {
			 char *name;
			 void *ptr;
	   } name_ptr;

	   void *id_list;
}


%{
/* What is the right way to import this prototipe?
 */
extern int yylex(void);


extern char *yytext;
extern int   yylineno;

cherokee_virtual_server_t      *current_virtual_server      = NULL;
cherokee_handler_table_entry_t *current_handler_table_entry = NULL;
cherokee_encoder_table_entry_t *current_encoder_entry       = NULL;
cherokee_module_info_t         *current_module_info         = NULL;
list_t                         *current_list                = NULL;

struct {
	   char                          *handler_name;
	   cherokee_handler_table_entry_t *entry;
	   cherokee_virtual_server_t     *vserver;
	   char                          *document_root;
	   char                          *directory_name;
} directory_content_tmp;

typedef struct {
	   void *next;
	   char *string;
} linked_list_t;


#define auto_virtual_server ((current_virtual_server) ? current_virtual_server : SRV(server)->vserver_default)


static void
free_linked_list (linked_list_t *list, void (*free_func) (void *))
{
	   linked_list_t *i = list;

	   while (i != NULL) {
			 linked_list_t *prev = i;

			 if ((free_func) && (i->string)) {
				    free_func (i->string);
			 }

			 prev = i;
			 i = i->next;
			 free (prev);
	   }	   
}

static char *
make_finish_with_slash (char *string, int *len)
{
	   char *new;

	   if (string[(*len)-1] == '/') {
			 return string;
	   }

	   new = (char*) malloc ((*len)+2);
	   memcpy (new, string, *len);
	   new[(*len)] = '/';
	   new[(*len)+1] = '\0';
	   free (string);

	   (*len)++;

	   return new;
}


cherokee_handler_table_entry_t *
handler_table_entry_new (void)
{
	   cherokee_handler_table_entry_t *entry;

	   cherokee_handler_table_entry_new (&entry);
	   current_handler_table_entry = entry;

	   return entry;
}


char *
new_string_to_lowercase (const char *in)
{
	   int   i;
	   char *tmp;
	   
	   i = strlen(in);
	   tmp = (char *) malloc (i+1);
	   tmp[i] = '\0';

	   while (i--) {
			 tmp[i] = tolower(in[i]);
	   }

	   return tmp;
}


void
yyerror(char* msg)
{
        PRINT_ERROR ("Error parsing configuration: '%s', line %d, symbol '%s'\n", 
				 msg, yylineno, yytext);
}

%}



%token T_QUOTE T_DENY T_THREAD_NUM T_SSL_CERT_KEY_FILE T_SSL_CERT_FILE T_KEEPALIVE_MAX_REQUESTS T_ERROR_DOCUMENT
%token T_TIMEOUT T_KEEPALIVE T_DOCUMENT_ROOT T_LOG T_MIME_FILE T_DIRECTORY T_HANDLER T_USER T_GROUP
%token T_SERVER T_USERDIR T_URL T_PIDFILE T_LISTEN T_FILEDIR T_HIDE_VERSION T_HIDE_SERVER_NAME T_ENCODER T_ALLOW 
%token T_BGCOLOR T_TEXT T_LINK T_ALINK T_VLINK T_BACKGROUND T_DIRECTORYINDEX T_IPV6 T_SHOW T_CHROOT T_HEADER_FILE
%token T_ICONS T_AUTH T_NAME T_METHOD T_PASSWDFILE T_SSL_CA_LIST_FILE

%token <number> T_NUMBER T_PORT 
%token <string> T_QSTRING T_FULLDIR T_ID T_HTTP_URL T_HOSTNAME T_IP 

%type <name_ptr> directory_option handler
%type <string> host_name
%type <id_list> id_list

%%

conffile : /* Empty */
         | lines
         ; 

lines : line
	 | lines line
	 ;

server_lines : server_line
             | server_lines server_line
             ;

line : server_line
     | common_line
     ;

common_line : server
            | port
            | listen
            | hideversion
            | hideservername
            | mime
            | icons
            | timeout
            | keepalive
            | keepalive_max_requests
            | pidfile
            | user1 
            | user2
            | group1 
            | group2
            | userdir
            | chroot
            | thread_number
            | ipv6
            | directoryindex
            | errordocument
            ;

server_line : directory
            | document_root
            | log
            | encoder
            | ssl_file
            | ssl_key_file
            | ssl_ca_list_file
            ;

handler_options : 
                | handler_options handler_option
                ;

directory_options : 
                  | directory_options directory_option
                  ;

encoder_options :
                | encoder_options encoder_option
                ;

auth_options :
             | auth_options auth_option;

id_list : T_ID 
{
	   linked_list_t *n = (linked_list_t *) malloc(sizeof(linked_list_t));
	   n->next = NULL;
	   n->string = $1;

	   $$ = n;
};

id_list : T_ID ',' id_list
{
	   linked_list_t *n = (linked_list_t *) malloc(sizeof(linked_list_t));

	   n->next = $3;
	   n->string = $1;

	   $$ = n;
};

port : T_PORT T_NUMBER 
{
	   SRV(server)->port = $2;
};

listen : T_LISTEN host_name
{
	   SRV(server)->listen_to = $2;
};

document_root : T_DOCUMENT_ROOT T_FULLDIR
{
	   cherokee_virtual_server_t *vserver;

	   vserver = auto_virtual_server;
	   vserver->root = $2;
	   vserver->root_len = strlen ($2);

	   /* Check for the endding slash
	    */
	   vserver->root = make_finish_with_slash (vserver->root, &(vserver->root_len));
};


log : T_LOG T_ID
{
	   ret_t ret;

	   /* Maybe load the module
	    */
	   ret = cherokee_module_loader_load (SRV(server)->loader, $2);
	   if (ret < ret_ok) {
			 PRINT_ERROR ("ERROR: Can't load logger module '%s'\n", $2);
			 return 1;
	   }

	   cherokee_module_loader_get  (SRV(server)->loader, $2, &current_module_info);
}
log_optional
{
	   cherokee_virtual_server_t *vserver;
	   vserver = auto_virtual_server;

	   /* Instance the logger object
	    */
	   cherokee_logger_table_new_logger (SRV(server)->loggers, $2, current_module_info,
								  vserver->logger_props, &vserver->logger);
	   current_module_info = NULL;
};

log_optional : 
             | '{' tuple_list '}';

tuple_list : 
           | tuple_list tuple
           ;

tuple : T_ID T_FULLDIR
{
	   cherokee_virtual_server_t *vserver;
	   vserver = auto_virtual_server;

	   if (vserver->logger_props == NULL) {
			 cherokee_table_new (&vserver->logger_props);
	   }

	   cherokee_table_add (vserver->logger_props, $1, $2);
};


hideversion : T_HIDE_VERSION T_NUMBER
{
	   SRV(server)->hideversion = $2;
};

hideservername : T_HIDE_SERVER_NAME T_NUMBER
{
	   SRV(server)->hideservername = $2;
};

mime : T_MIME_FILE T_FULLDIR
{
	   cherokee_mime_t *mime;

	   cherokee_mime_init_with_file (&mime, (char *)$2);
	   free ($2);
};

icons : T_ICONS T_FULLDIR
{
	   SRV(server)->icons_file = $2;
};

timeout : T_TIMEOUT T_NUMBER
{
	   SRV(server)->timeout = $2;

	   cherokee_buffer_clean  (SRV(server)->timeout_header);
	   cherokee_buffer_add_va (SRV(server)->timeout_header, "Keep-Alive: timeout=%d"CRLF, $2);
};

keepalive : T_KEEPALIVE T_NUMBER
{
	   SRV(server)->keepalive = $2;
};

keepalive_max_requests : T_KEEPALIVE_MAX_REQUESTS T_NUMBER
{
	   SRV(server)->keepalive_max = $2;
};

ssl_file : T_SSL_CERT_FILE T_FULLDIR
{
#ifdef HAVE_TLS
	   cherokee_virtual_server_t *vsrv = auto_virtual_server;

	   if (vsrv->server_cert != NULL) {
			 PRINT_ERROR ("\"SSLCertificateFile\" overlap: '%s' <- '%s'\n", vsrv->server_cert, $2);
			 free (vsrv->server_cert);
	   }

	   vsrv->server_cert = $2;

#else
	   PRINT_ERROR ("Ignoring SSL configuration entry: \"SSLCertificateFile\"\n");
#endif
};

ssl_key_file : T_SSL_CERT_KEY_FILE T_FULLDIR
{
#ifdef HAVE_TLS
	   cherokee_virtual_server_t *vsrv = auto_virtual_server;

	   if (vsrv->server_key != NULL) {
			 PRINT_ERROR ("\"SSLCertificateKeyFile\" overlap: '%s' <- '%s'\n", vsrv->server_key, $2);
			 free (vsrv->server_key);
	   }

	   vsrv->server_key = $2;

#else
	   PRINT_ERROR ("Ignoring SSL configuration entry: \"SSLCertificateKeyFile\"\n");
#endif
};

ssl_ca_list_file : T_SSL_CA_LIST_FILE T_FULLDIR
{
#ifdef HAVE_TLS
	   cherokee_virtual_server_t *vsrv = auto_virtual_server;

	   if (vsrv->ca_cert != NULL) {
			 PRINT_ERROR ("\"SSLCAListFile\" overlap: '%s' <- '%s'\n", vsrv->ca_cert, $2);
			 free (vsrv->ca_cert);
	   }

	   vsrv->ca_cert = $2;

#else
	   PRINT_ERROR ("Ignoring SSL configuration entry: \"SSLCAListFile\"\n");
#endif
};


encoder : T_ENCODER T_ID
{
	   ret_t ret;

	   ret = cherokee_module_loader_load (SRV(server)->loader, $2);
	   if (ret < ret_ok) {
			 PRINT_ERROR ("ERROR: Can't load encoder module '%s'\n", $2);
			 return 1;
	   }

};

encoder : T_ENCODER T_ID '{'
{
	   ret_t ret;
	   cherokee_module_info_t *info;
	   cherokee_encoder_table_entry_t *enc;

	   /* Load the module
	    */
	   ret = cherokee_module_loader_load (SRV(server)->loader, $2);
	   if (ret < ret_ok) {
			 PRINT_ERROR ("ERROR: Can't load encoder module '%s'\n", $2);
			 return 1;
	   }

	   cherokee_module_loader_get  (SRV(server)->loader, $2, &info);

	   /* Set the info in the new entry
	    */
	   cherokee_encoder_table_entry_new (&enc);
	   cherokee_encoder_table_entry_get_info (enc, info);

	   /* Set in the encoders table
	    */
	   cherokee_encoder_table_set (SRV(server)->encoders, $2, enc);
	   current_encoder_entry = enc;
} 
encoder_options '}' 
{
	   current_encoder_entry = NULL;
};

encoder_option : T_ALLOW id_list
{
	   linked_list_t *i;
	   cherokee_matching_list_t *matching;

	   cherokee_matching_list_new (&matching);
	   cherokee_encoder_entry_set_matching_list (current_encoder_entry, matching);

	   i = $2;
	   while (i!=NULL) {
			 linked_list_t *prev;

			 cherokee_matching_list_add_allow (matching, i->string);

			 free(i->string);
			 prev = i;
			 i = i->next;
			 free(prev);
	   }
};

encoder_option : T_DENY id_list
{
	   linked_list_t *i;
	   cherokee_matching_list_t *matching;

	   cherokee_matching_list_new (&matching);
	   cherokee_encoder_entry_set_matching_list (current_encoder_entry, matching);

	   i = $2;
	   while (i!=NULL) {
			 linked_list_t *prev;

			 cherokee_matching_list_add_deny (matching, i->string);

			 free(i->string);
			 prev = i;
			 i = i->next;
			 free(prev);
	   }
};

pidfile : T_PIDFILE T_FULLDIR
{
	   FILE *file;

	   file = fopen ($2, "w");
	   if (file == NULL) {
			 PRINT_ERROR ("Can't write PID file '%s'\n", $2);
			 return 1;
	   }

	   snprintf (gbl_buffer, gbl_buffer_size, "%d\n", getpid());
	   fwrite (gbl_buffer, 1, strlen(gbl_buffer), file);
	   fclose (file);

	   free ($2);
};

chroot : T_CHROOT T_FULLDIR
{
	   SRV(server)->chroot = $2;
};

thread_number : T_THREAD_NUM T_NUMBER
{
#ifdef HAVE_PTHREAD
	   SRV(server)->ncpus = $2;
#endif
};

ipv6 : T_IPV6 T_NUMBER
{
	   SRV(server)->ipv6 = $2;
};

user1 : T_USER T_ID
{
	   struct passwd *pwd;
	   
	   pwd = (struct passwd *) getpwnam ($2);
	   if (pwd == NULL) {
			 PRINT_ERROR ("User '%s' not found in the system", $2);
			 return 1;
	   }

	   SRV(server)->user = pwd->pw_uid;

	   free ($2);
};

user2 : T_USER T_NUMBER
{
	   SRV(server)->user = $2;
};

group1 : T_GROUP T_ID
{
	   struct group *grp;

	   grp = (struct group *) getgrnam ($2);
	   if (grp == NULL) {
			 PRINT_ERROR ("Group '%s' not found in the system", $2);
			 return 1;
	   }

	   SRV(server)->group = grp->gr_gid;

	   free ($2);
};

group2 : T_GROUP T_NUMBER
{
	   SRV(server)->group = $2;
};

handler : T_HANDLER T_ID '{' handler_options '}' 
{
	   $$.name = $2;
	   $$.ptr = current_handler_table_entry;
};

handler : T_HANDLER T_ID 
{
	   $$.name = $2;
	   $$.ptr = current_handler_table_entry;
};


handler_option : T_URL T_HTTP_URL
{
	   cherokee_handler_table_entry_set (current_handler_table_entry, "url", $2);
};

handler_option : T_FILEDIR T_FULLDIR
{
	   cherokee_handler_table_entry_set (current_handler_table_entry, "filedir", $2);
};

handler_option : T_BGCOLOR T_ID
{ cherokee_handler_table_entry_set (current_handler_table_entry, "bgcolor", $2); };

handler_option : T_TEXT T_ID
{ cherokee_handler_table_entry_set (current_handler_table_entry, "text", $2); };

handler_option : T_LINK T_ID
{ cherokee_handler_table_entry_set (current_handler_table_entry, "link", $2); };

handler_option : T_VLINK T_ID
{ cherokee_handler_table_entry_set (current_handler_table_entry, "vlink", $2); };

handler_option : T_ALINK T_ID
{ cherokee_handler_table_entry_set (current_handler_table_entry, "alink", $2); };

handler_option : T_HEADER_FILE T_ID
{ cherokee_handler_table_entry_set (current_handler_table_entry, "headerfile", $2); };


handler_option : T_SHOW id_list
{
	   linked_list_t *i;

	   i = $2;
	   while (i != NULL) {
			 if ((!strncasecmp (i->string, "date",  4)) ||
				(!strncasecmp (i->string, "size",  4)) ||
				(!strncasecmp (i->string, "group", 5)) ||
				(!strncasecmp (i->string, "owner", 5)))
			 {
				    char *lower;

				    lower = new_string_to_lowercase (i->string);
				    free (i->string);
				    i->string = lower;

				    cherokee_handler_table_entry_set (current_handler_table_entry, i->string, i->string);				    
			 } else {
				    PRINT_ERROR ("Unknown parameter '%s' for \"Show\"", i->string);
			 }
				
			 i = i->next;
	   }

	   free_linked_list ($2, free);
};


host_name : T_ID
          | T_IP 
{
	   $$ = $1;
};

server : T_SERVER id_list '{' 
{
	   linked_list_t *i = $2;
	   CHEROKEE_NEW(vsrv, virtual_server);
	   current_virtual_server = vsrv;

	   /* Add the virtual server to the list
	    */
	   cherokee_list_add (SRV(server)->vservers, vsrv);

	   /* Add all the alias to the table
	    */
	   while (i != NULL) {
			 cherokee_table_add (SRV(server)->vservers_ref, i->string, vsrv);
			 i = i->next;
	   }
	   free_linked_list ($2, NULL);

} server_lines '}' {

	   current_virtual_server = NULL;
};


directory : T_DIRECTORY T_FULLDIR '{' 
{
	   int name_len;

	   name_len = strlen($2);

	   /* Fill the tmp struct
	    */
	   directory_content_tmp.vserver        = auto_virtual_server;
	   directory_content_tmp.entry          = handler_table_entry_new ();       /* new! */
	   directory_content_tmp.directory_name = make_finish_with_slash ($2, &name_len);
	   directory_content_tmp.handler_name   = NULL;
	   directory_content_tmp.document_root  = NULL;
} 
directory_options '}'
{
	   /* Basic checks
	    */
	   if (directory_content_tmp.handler_name == NULL) {
			 PRINT_ERROR ("Directory %s needs a handler; this directory entry is ignored.\n", directory_content_tmp.directory_name);
			 goto out;
	   }

	   /* Set the document_root in the entry
	    */
	   if (directory_content_tmp.document_root != NULL) {
			 cherokee_buffer_make_empty(directory_content_tmp.entry->document_root);
			 cherokee_buffer_add (directory_content_tmp.entry->document_root,
							  directory_content_tmp.document_root,
							  strlen (directory_content_tmp.document_root));
	   }

	   /* Load the module 
	    */
	   {
			 ret_t ret;
			 cherokee_module_info_t *info;

			 ret = cherokee_module_loader_load (SRV(server)->loader, directory_content_tmp.handler_name);
			 if (ret < ret_ok) {
				    PRINT_ERROR ("Error loading module '%s'\n", directory_content_tmp.handler_name);
				    return 1;
			 }

			 ret = cherokee_module_loader_get  (SRV(server)->loader, directory_content_tmp.handler_name, &info);
			 if (ret < ret_ok) {
				    PRINT_ERROR ("Error loading module '%s'\n", directory_content_tmp.handler_name);
				    return 1;
			 }

			 cherokee_handler_table_enty_get_info (directory_content_tmp.entry, info);
			 cherokee_handler_table_add (directory_content_tmp.vserver->plugins,
								    directory_content_tmp.handler_name, 
								    directory_content_tmp.entry);
	   }

	   /* Add to the virtual server "web_dir -> entry" table
	    */
 	   cherokee_virtual_server_set (directory_content_tmp.vserver, 
							  directory_content_tmp.directory_name, 
							  directory_content_tmp.handler_name,
							  directory_content_tmp.entry);

	   /* Clean
	    */
out:
	   if (directory_content_tmp.document_root != NULL) {
			 free (directory_content_tmp.document_root);
			 directory_content_tmp.document_root = NULL;
	   }
	   directory_content_tmp.vserver       = NULL;
	   directory_content_tmp.entry         = NULL;
	   directory_content_tmp.handler_name  = NULL;

	   current_handler_table_entry = NULL;
};


directory_option : handler
{	   
	   directory_content_tmp.handler_name = $1.name;
};


directory_option : T_DOCUMENT_ROOT T_FULLDIR
{
	   int len;
	   char *new_fdir;

	   len = strlen($2);
	   new_fdir = make_finish_with_slash ($2, &len);

	   directory_content_tmp.document_root = new_fdir;
};

directory_option : T_AUTH T_ID '{' auth_options '}'
{
};

auth_option : T_NAME T_QSTRING 
{
	   cherokee_buffer_t *realm;
	   realm = directory_content_tmp.entry->auth_realm;

	   cherokee_buffer_add (realm, $2, strlen($2));
	   free ($2);
};

auth_option : T_METHOD T_ID maybe_auth_option_params
{
	   ret_t ret;
	   cherokee_module_info_t *info;

	   ret = cherokee_module_loader_load (SRV(server)->loader, $2);
	   if (ret < ret_ok) {
			 PRINT_ERROR ("ERROR: Can't load validator module '%s'\n", $2);
			 return 1;
	   }

	   cherokee_module_loader_get  (SRV(server)->loader, $2, &info);

	   if (info->type != cherokee_validator) {
			 PRINT_ERROR ("ERROR: %s is not a validator module!!\n", $2);
	   }

	   directory_content_tmp.entry->validator_new_func = info->new_func;
};


maybe_auth_option_params : '{' auth_option_params '}'
                         |
                         ;

auth_option_params : 
                   | auth_option_params auth_option_param
                   ;

auth_option_param : T_PASSWDFILE T_FULLDIR 
{ cherokee_handler_table_entry_set (current_handler_table_entry, "file", $2); };



userdir : T_USERDIR T_ID '{' handler '}'
{
	   ret_t ret;
	   int   len;
	   cherokee_module_info_t         *info;
	   cherokee_handler_table_entry_t *dir;

	   /* Set the users public directory
	    */
	   len = strlen($2);
	   SRV(server)->userdir = make_finish_with_slash ($2, &len);

	   /* Build the handler
	    */
	   dir = handler_table_entry_new ();

	   ret = cherokee_module_loader_load (SRV(server)->loader, $4.name);
	   if (ret < ret_ok) {
			 PRINT_ERROR ("ERROR: Can not load handler module '%s'\n", $4.name);
			 return 1;
	   }

	   cherokee_module_loader_get  (SRV(server)->loader, $4.name, &info);

	   cherokee_handler_table_enty_get_info (dir, info);
	   SRV(server)->userdir_handler = dir;

	   free ($4.name);
};


directoryindex : T_DIRECTORYINDEX id_list
{
	   linked_list_t *i = $2;

	   while (i != NULL) {
			 cherokee_list_add_tail (&SRV(server)->index_list, i->string);
			 i = i->next;
	   }

	   free_linked_list ($2, NULL);
};

errordocument : T_ERROR_DOCUMENT T_NUMBER T_FULLDIR
{
	   cherokee_virtual_server_t *vsrv = auto_virtual_server;
	   
	   if (($2 < 400) || ($2 > 417))
	   {
			 PRINT_ERROR ("Error: Ignoring incorrect error code. Valid: 400..417. '%s'\n", $3);
	   }

	   vsrv->error_document400[$2-400].file = $3;
};

errordocument : T_ERROR_DOCUMENT T_NUMBER T_HTTP_URL
{
	   cherokee_virtual_server_t *vsrv = auto_virtual_server;
	   
	   if (($2 < 400) || ($2 > 417))
	   {
			 PRINT_ERROR ("Error: Ignoring incorrect error code. Valid: 400..417. '%s'\n", $3);
	   }

	   vsrv->error_document400[$2-400].url = $3;
};

%%

