/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "icons.h"
#include "list_ext.h"
#include "match.h"


ret_t 
cherokee_icons_new  (cherokee_icons_t **icons)
{
	ret_t ret;
	CHEROKEE_NEW_STRUCT(n, icons);

	ret = cherokee_table_new (&n->files);
	if (ret < ret_ok) return ret;

	ret = cherokee_table_new (&n->files_matching);
	if (ret < ret_ok) return ret;

	ret = cherokee_table_new (&n->suffixes);
	if (ret < ret_ok) return ret;

	n->default_icon   = NULL;
	n->directory_icon = NULL;
	n->parentdir_icon = NULL;
	
	*icons = n;
	return ret_ok;
}


ret_t
cherokee_icons_free (cherokee_icons_t *icons)
{
	/* TODO: Set a free_item function.
	 */
	cherokee_table_free2 (icons->files, free);
	cherokee_table_free2 (icons->suffixes, free);
	cherokee_table_free2 (icons->files_matching, free);

	if (icons->default_icon != NULL) {
		free (icons->default_icon);
		icons->default_icon = NULL;
	}

	if (icons->directory_icon != NULL) {
		free (icons->directory_icon);
		icons->directory_icon = NULL;
	}

	if (icons->parentdir_icon != NULL) {
		free (icons->parentdir_icon);
		icons->parentdir_icon = NULL;
	}

	free (icons);

	return ret_ok;
}


ret_t 
cherokee_icons_clean (cherokee_icons_t *icons)
{
	/* TODO: Set a free_item function.
	 */
	cherokee_table_clean2 (icons->files, free);
	cherokee_table_clean2 (icons->suffixes, free);
	cherokee_table_clean2 (icons->files_matching, free);

	if (icons->default_icon != NULL) {
		free (icons->default_icon);
		icons->default_icon = NULL;
	}

	if (icons->directory_icon != NULL) {
		free (icons->directory_icon);
		icons->directory_icon = NULL;
	}

	if (icons->parentdir_icon != NULL) {
		free (icons->parentdir_icon);
		icons->parentdir_icon = NULL;
	}

	return ret_ok;
}


ret_t 
cherokee_icons_read_config_file (cherokee_icons_t *icons, const char *filename)
{
	int   error;
	char *file;
	void *bufstate;

	extern FILE *yy_icons_in;
	extern int yy_icons_parse (void *);

	file = (filename) ? filename : CHEROKEE_CONFDIR"/icons.conf";

	yy_icons_in = fopen (file, "r");
	if (yy_icons_in == NULL) {
		PRINT_ERROR("Can't read the icons file: '%s'\n", file);
		return ret_error;
	}

	yy_icons_restart(yy_icons_in);

	bufstate = (void *) yy_icons__create_buffer (yy_icons_in, 65535);
	yy_icons__switch_to_buffer (bufstate);
	error = yy_icons_parse (icons);
	yy_icons__delete_buffer (bufstate);

	fclose (yy_icons_in);

	return (error)? ret_error : ret_ok;
}


ret_t 
cherokee_icons_read_config_string (cherokee_icons_t *icons, const char *string)
{
	int   error;
	void *bufstate;

	extern int  yy_icons__scan_string (void *);
	extern int  yy_icons_parse (void *);
	extern void yy_icons__switch_to_buffer (void *);
	extern void yy_icons__delete_buffer (void *);

	bufstate = (void *) yy_icons__scan_string (string);
	yy_icons__switch_to_buffer(bufstate);

	error = yy_icons_parse((void *)icons);

	yy_icons__delete_buffer (bufstate);

	return (error)? ret_error : ret_ok;
}


ret_t 
cherokee_icons_set_suffixes (cherokee_icons_t *icons, list_t *suf_list, char *icon)
{
	list_t *i;

	/* Add suffixes to the table
	 */
	list_for_each (i, suf_list) {
		cherokee_table_add (icons->suffixes, LIST_ITEM_INFO(i), strdup(icon));
	}
	
	return ret_ok;
}


ret_t 
cherokee_icons_set_files (cherokee_icons_t *icons, list_t *nam_list, char *icon)
{
	list_t *i;

	/* Add names to the table
	 */
	list_for_each (i, nam_list) {
		char *filename = LIST_ITEM_INFO(i);

		if ((index(filename, '*') != NULL) ||
		    (index(filename, '?') != NULL))
		{
			cherokee_table_add (icons->files_matching, filename, strdup(icon));
			continue;
		}

		cherokee_table_add (icons->files, filename, strdup(icon));
	}

	return ret_ok;
}


ret_t 
cherokee_icons_set_directory (cherokee_icons_t *icons, char *icon)
{
	if (icons->directory_icon != NULL) {
		free (icons->directory_icon);
		icons->directory_icon = NULL;
	}

	icons->directory_icon = icon;
}

ret_t 
cherokee_icons_set_parentdir (cherokee_icons_t *icons, char *icon)
{
	if (icons->parentdir_icon != NULL) {
		free (icons->parentdir_icon);
		icons->parentdir_icon = NULL;
	}

	icons->parentdir_icon = icon;
}

ret_t 
cherokee_icons_set_default (cherokee_icons_t *icons, char *icon)
{
	if (icons->default_icon != NULL) {
		free (icons->default_icon);
		icons->default_icon = NULL;
	}

	icons->default_icon = icon;
}


static int
match_file (const char *param_file, 
	    const char *pattern, 
	    const char *icon)
{
	return (! match (pattern, param_file));
}


ret_t 
cherokee_icons_get_icon (cherokee_icons_t *icons, const char *file, char **icon_ret)
{
	list_t *i;
	ret_t   ret;
	char   *suffix, *match_string;

	/* Look for the filename
	 */
	ret = cherokee_table_get (icons->files, file, icon_ret);
	if (ret == ret_ok) return ret_ok;

	/* Look for the suffix
	 */
	suffix = rindex (file, '.');
	if (suffix != NULL) {
		ret = cherokee_table_get (icons->suffixes, suffix+1, icon_ret);
		if (ret == ret_ok) return ret_ok;
	}
	
	/* Look for the wildcat matching
	 */
	ret = cherokee_table_while (icons->files_matching, match_file, file, &match_string, icon_ret);
	if (ret == ret_ok) return ret_ok;

	/* Default one
	 */
	if (icons->default_icon != NULL) {
		*icon_ret = icons->default_icon;
	}

	return ret_ok;
}

