/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"

#include <errno.h>

#include "virtual_server.h"


ret_t 
cherokee_virtual_server_new (cherokee_virtual_server_t **vserver)
{
	int i;

       	CHEROKEE_NEW_STRUCT (vsrv, virtual_server);

	vsrv->root   = NULL;

	vsrv->logger       = NULL;
	vsrv->logger_props = NULL;

	vsrv->bytes_sent   = 0;
	
#ifdef HAVE_TLS
	vsrv->server_cert = NULL;
	vsrv->server_key  = NULL;
	vsrv->ca_cert     = NULL;
#endif

	for (i=0; i<=http_type_400_max; i++) {
		vsrv->error_document400[i].file = NULL;
		vsrv->error_document400[i].url  = NULL;		
	}
	
	cherokee_handler_table_new (&vsrv->plugins);
	return_if_fail (vsrv->plugins!=NULL, ret_nomem);	   

	*vserver = vsrv;
	
	return ret_ok;
}


ret_t 
cherokee_virtual_server_free (cherokee_virtual_server_t *vserver)
{
	int i;

	if (vserver->root != NULL) {
		free (vserver->root);
		vserver->root = NULL;
	}

	if (vserver->logger != NULL) {
		cherokee_logger_free (vserver->logger);
		vserver->logger = NULL;
	}
	if (vserver->logger_props != NULL) {
		cherokee_table_free (vserver->logger_props);
		vserver->logger_props = NULL;
	}
	
	cherokee_handler_table_free (vserver->plugins);
	vserver->plugins = NULL;

	for (i=0; i<=http_type_400_max; i++) {
		if (vserver->error_document400[i].file) 
			free (vserver->error_document400[i].file);

		if (vserver->error_document400[i].url)
			free (vserver->error_document400[i].url);
	}

	free (vserver);
	
	return ret_ok;
}



ret_t 
cherokee_virtual_server_clean (cherokee_virtual_server_t *vserver)
{
	int i;

	if (vserver->root != NULL) {
		free (vserver->root);
		vserver->root = NULL;
	}
	
	if (vserver->logger != NULL) {
		cherokee_logger_free (vserver->logger);
		vserver->logger = NULL;
	}
	if (vserver->logger_props != NULL) {
		cherokee_table_free (vserver->logger_props);
		vserver->logger_props = NULL;
	}

	for (i=0; i<=http_type_400_max; i++) {
		if (vserver->error_document400[i].file) 
			free (vserver->error_document400[i].file);

		if (vserver->error_document400[i].url)
			free (vserver->error_document400[i].url);
	}

	return ret_ok;
}


ret_t 
cherokee_virtual_server_set (cherokee_virtual_server_t *vserver, char *web_dir,  char *handler_name, cherokee_handler_table_entry_t *plugin_entry)
{
	ret_t ret;

#if 0
	printf ("vserver::set webdir:'%s' -> %p\n", web_dir, plugin_entry);
#endif

	ret = cherokee_handler_table_add (vserver->plugins, web_dir, plugin_entry);
	if (ret != ret_ok) {
		switch (ret) {
		case ret_file_not_found:
			PRINT_ERROR ("Can't load handler '%s': File not found\n", handler_name);
			break;
		default:
			PRINT_ERROR ("Can't load handler '%s': Unknown error\n", handler_name);
		}
	}
	
	return ret;

}


ret_t 
cherokee_virtual_server_get_error_page (cherokee_virtual_server_t *vserver, cherokee_http_t error_code, char **filename_ref)
{
	if (! http_type_400(error_code)) {
		return ret_error;
	}

	if (vserver->error_document400[error_code-400].file != NULL) {
		*filename_ref = vserver->error_document400[error_code-400].file;
		return ret_ok;
	}
	
	return ret_not_found;
}


ret_t 
cherokee_virtual_server_get_error_url (cherokee_virtual_server_t *vserver, cherokee_http_t error_code, char **url_ref)
{
	if (! http_type_400(error_code)) {
		return ret_error;
	}

	if (vserver->error_document400[error_code-400].url != NULL) {
		*url_ref = vserver->error_document400[error_code-400].url;
		return ret_ok;
	}
	
	return ret_not_found;
}


#ifdef HAVE_TLS
static ret_t
generate_dh_params (gnutls_dh_params *dh_params)
{
	int rc;

	gnutls_dh_params_init (dh_params);
	gnutls_dh_params_generate2 (dh_params, 1024);

	return ret_ok;
}


/*
static ret_t
generate_rsa_params (gnutls_rsa_params* rsa_params)
{
	gnutls_rsa_params_init (rsa_params);
	gnutls_rsa_params_generate2 (rsa_params, 512);

	return ret_ok;
}
*/

ret_t 
cherokee_virtual_server_init_tls (cherokee_virtual_server_t *vsrv)
{
	int rc;

	if ((vsrv->ca_cert     == NULL) ||
	    (vsrv->server_cert == NULL) ||
	    (vsrv->server_key  == NULL)) 
	{
		return ret_error;
	}

        rc = gnutls_certificate_allocate_credentials (&vsrv->credentials);
	if (rc < 0) {
		PRINT_ERROR ("Couldn't allocate credentials.\n");
		return ret_error;
	}
	
	/* CA file
	 */
	rc = gnutls_certificate_set_x509_trust_file (vsrv->credentials,
						     vsrv->ca_cert,
						     GNUTLS_X509_FMT_PEM);
	if (rc < 0) {
		PRINT_ERROR ("Error reading X.509 CA Certificate: '%s'\n", vsrv->ca_cert);
		return ret_error;
	}

	/* Key file
	 */
	rc = gnutls_certificate_set_x509_key_file (vsrv->credentials,
						   vsrv->server_cert,
						   vsrv->server_key,
						   GNUTLS_X509_FMT_PEM);	
	if (rc < 0) {
		PRINT_ERROR ("Error reading X.509 key '%s' or certificate '%s' file\n", 
			     vsrv->server_key, vsrv->server_cert);
		return ret_error;
	}

	generate_dh_params  (&vsrv->dh_params);
//	generate_rsa_params (&vsrv->rsa_params);

	gnutls_certificate_set_dh_params (vsrv->credentials, vsrv->dh_params);
//	gnutls_certificate_set_rsa_params (vsrv->credentials, vsrv->rsa_params);

	return ret_ok;
}
#endif /* HAVE_TLS */
