/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "buffer.h"
#include "md5.h"

ret_t
cherokee_buffer_new  (cherokee_buffer_t **buf)
{
	CHEROKEE_NEW_STRUCT(n, buffer);

	n->buf  = NULL;
	n->size = 0;
	n->len  = 0;
	   
	*buf = n;
	   
	return ret_ok;
}


ret_t
cherokee_buffer_free (cherokee_buffer_t *buf)
{
	if (buf->buf) {
		free (buf->buf);
		buf->buf = NULL;
	}
	
	free (buf);
	
	return ret_ok;
}


ret_t
cherokee_buffer_clean (cherokee_buffer_t *buf)
{
	return cherokee_buffer_make_empty (buf);
}


ret_t
cherokee_buffer_add (cherokee_buffer_t *buf, char *txt, int size)
{	   
	int free = buf->size - buf->len;

	if (size <= 0)
		return ret_ok;

	/* Get memory
	 */
	if (free < (size+1)) {
		buf->buf = (char *) realloc(buf->buf, buf->size + size - free + 1);
		return_if_fail (buf->buf, ret_nomem);

		buf->size += size - free + 1;
	}
	   
	/* Copy	
	 */
	memcpy (buf->buf + buf->len, txt, size);

	buf->len += size;
	buf->buf[buf->len] = '\0';
	   
	return ret_ok;
}


ret_t 
cherokee_buffer_add_buffer (cherokee_buffer_t *buf, cherokee_buffer_t *buf2)
{
	return cherokee_buffer_add (buf, buf2->buf, buf2->len);
}


ret_t 
cherokee_buffer_add_va (cherokee_buffer_t  *buf, char *format, ...)
{
	int len;
	va_list ap;
	CHEROKEE_TEMP (tmp, 200);

	va_start (ap, format);
	len = vsnprintf (tmp, tmp_size, format, ap);
	va_end (ap);

	if (len >= 199) return ret_error;

	return cherokee_buffer_add (buf, tmp, len);
}


ret_t
cherokee_buffer_prepend (cherokee_buffer_t  *buf, char *txt, int size)
{
	int free = buf->size - buf->len;

	/* Get memory
	 */
	if (free <= size) {
		buf->buf = (char *) realloc(buf->buf, buf->size + size - free + 1);
		return_if_fail (buf->buf, ret_nomem);
		   
		buf->size += size - free + 1;
	}

	memmove (buf->buf+size, buf->buf, buf->len);

	memcpy (buf->buf, txt, size);
	buf->len += size;
	buf->buf[buf->len] = '\0';
	   
	return ret_ok;
}


int
cherokee_buffer_is_empty (cherokee_buffer_t *buf)
{
	return (buf->len == 0);
}


ret_t
cherokee_buffer_make_empty (cherokee_buffer_t *buf)
{
	if ((buf->buf != NULL) &&
	    (buf->len > 0))
	{
		buf->buf[0] = '\0';
	}

	buf->len = 0;	
	return ret_ok;
}


ret_t
cherokee_buffer_move_to_begin (cherokee_buffer_t *buf, int pos)
{
	return_if_fail (pos <= buf->size, ret_error);

	if (pos <= 0) return ret_ok;

	if (pos <= buf->len) {
		memcpy (buf->buf, buf->buf+pos, buf->size-pos);
		buf->len -= pos;
	}
	   
	return ret_ok;
}


ret_t
cherokee_buffer_ensure_size (cherokee_buffer_t *buf, int size)
{
	if (size > buf->size) {
		buf->buf = (char *) realloc(buf->buf, size);
		return_if_fail (buf->buf, ret_error);
		buf->size = size;
	}
	   
	return ret_ok;
}


/* This function is based on code from thttpd
 * Copyright: Jef Poskanzer <jef@acme.com>
 *
 * Copies and decodes a string.  It's ok for from and to to be the
 * same string.
 */

ret_t
cherokee_buffer_decode (cherokee_buffer_t *buffer)
{
	char *from;
	char *to;

	if (buffer->buf == NULL) return;

	from = to = buffer->buf;

	for (; *from != '\0'; ++to, ++from) {
		if (from[0] == '%' && isxdigit(from[1]) && isxdigit(from[2])) {
			if ((from[1] == '0') && (from[2] == '0')) {
				/* Replace null bytes (%00) with spaces,
				 * to prevent attacks 	
				 */
				*to = ' ';
			} else {
				*to = hexit(from[1]) * 16 + hexit(from[2]);
			}

			from += 2;
			buffer->len -= 2;
		} else {
			*to = *from;
		}
	}
	*to = '\0';

	return ret_ok;
}

ret_t 
cherokee_buffer_drop_endding (cherokee_buffer_t *buffer, int num_chars)
{
	buffer->buf[buffer->len - num_chars] = '\0';
	buffer->len -= num_chars;

	return ret_ok;
}


void  
cherokee_buffer_swap_chars (cherokee_buffer_t *buffer, char a, char b)
{
	int i = 0;

	if (buffer->buf == NULL) return;

	for (i=0; i<buffer->len; i++) {
		if (buffer->buf[i] == a) {
			buffer->buf[i] = b;
		}
	}
}


ret_t 
cherokee_buffer_remove_dups (cherokee_buffer_t *buffer, char c)
{
	char *a      = buffer->buf;
	int   offset = 0;
	
	if (buffer->len < 2) {
		return ret_ok;
	}

	do {
		if ((a[0] == c) && (a[offset+1] == c)) {
			offset++;
			continue;
		}
		
		*++a = a[offset];

	} while ((a && *a != '\0') && (a < buffer->buf + buffer->len) && (offset+1 < buffer->len));

	buffer->len -= offset;
	buffer->buf[buffer->len] = '\0';

	return ret_ok;
}


ret_t 
cherokee_buffer_remove_string (cherokee_buffer_t *buf, char *string, int string_len)
{
	char *tmp;
	int   offset;

	if (buf->len <= 0) {
		return ret_ok;
	}

	while ((tmp = strstr (buf->buf, string)) != NULL) {
		offset = tmp - buf->buf;
		memmove (tmp, tmp+string_len, buf->len - (offset+string_len) +1);
		buf->len -= string_len;
	}

	return ret_ok;
}


crc_t 
cherokee_buffer_crc32 (cherokee_buffer_t  *buf)
{
	return crc32_sz (buf->buf, buf->len);
}


ret_t 
cherokee_buffer_read_file (cherokee_buffer_t *buf, char *filename)
{
	int r, ret, f;
	int remain;
	struct stat info;

	/* Stat() the file
	 */
	r = stat (filename, &info);
	if (r != 0) return ret_error;

	/* Is a regular file?
	 */
	if (S_ISREG(info.st_mode) == 0) {
		return ret_error;
	}

	/* Maybe get memory
	 */
	remain = buf->size - buf->len;
	ret = cherokee_buffer_ensure_size (buf, info.st_size - remain + 1);
	if (ret != ret_ok) return ret;
	
	/* Open the file
	 */
	f = open (filename, O_RDONLY);
	if (f < 0) return ret_error;

	/* Read the content
	 */
	r = buf->len;
	buf->len = read (f, buf->buf+r, info.st_size);
	if (buf->len < 0) {
		buf->len = 0;
		return ret_error;
	}
	buf->len += r;
	
	/* Close it and exit
	 */
	close(f);

	return ret_ok;
}


ret_t
cherokee_buffer_print_debug (cherokee_buffer_t *buf, int len)
{
	int i, length;

	if ((len == -1) || (buf->len <= len)) {
		length = buf->len;
	} else {
		length = len;
	}


	for (i=0; i < length; i++) {
		if (i%16 == 0) {
			printf ("%08x ", i);
		}

		printf ("%02x", buf->buf[i] & 0xFF);

		if ((i+1)%2 == 0) {
			printf (" ");
		}

		if ((i+1)%16 == 0) {
			printf ("\n");
		}

		fflush(stdout);
	}
	printf (CRLF);

	return ret_ok;
}


ret_t 
cherokee_buffer_add_version (cherokee_buffer_t *buf, int port, cherokee_version_t ver)
{
	ret_t ret;
	static int  len = 0;
	static char port_str[6];
	static int  port_len = 0;

	if (len == 0) {
		len = strlen (VERSION);
		port_len = snprintf (port_str, 6, "%d", port);
	}

	switch (ver) {
	case ver_full_html:
		cherokee_buffer_ensure_size (buf, buf->len + 29 + len + 6 + port_len + 10);

		cherokee_buffer_add (buf, "<address>Cherokee web server ", 29);
		cherokee_buffer_add (buf, VERSION, len);
		cherokee_buffer_add (buf, " Port ", 6);
		cherokee_buffer_add (buf, port_str, port_len);
		cherokee_buffer_add (buf, "</address>", 10);
		break;

	case ver_port_html:
		cherokee_buffer_ensure_size (buf, buf->len + 34 + port_len + 10);

		cherokee_buffer_add (buf, "<address>Cherokee web server Port ", 34);
		cherokee_buffer_add (buf, port_str, port_len);
		cherokee_buffer_add (buf, "</address>", 10);
		break;

	default:	
		SHOULDNT_HAPPEN;
	}

	return ret;
}


/* b64_decode_table has been copy&pasted from the code of thttpd:  
 * Copyright <A9> 1995,1998,1999,2000,2001 by Jef Poskanzer <jef@acme.com>
 */

/* Base-64 decoding.  This represents binary data as printable ASCII
 * characters.  Three 8-bit binary bytes are turned into four 6-bit
 * values, like so:
 *
 *   [11111111]  [22222222]  [33333333]
 *   [111111] [112222] [222233] [333333]
 *
 * Then the 6-bit values are represented using the characters "A-Za-z0-9+/".
 */

static int 
b64_decode_table[256] = {
	-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 00-0F */
	-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 10-1F */
	-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,62,-1,-1,-1,63,  /* 20-2F */
	52,53,54,55,56,57,58,59,60,61,-1,-1,-1,-1,-1,-1,  /* 30-3F */
	-1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,  /* 40-4F */
	15,16,17,18,19,20,21,22,23,24,25,-1,-1,-1,-1,-1,  /* 50-5F */
	-1,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,  /* 60-6F */
	41,42,43,44,45,46,47,48,49,50,51,-1,-1,-1,-1,-1,  /* 70-7F */
	-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 80-8F */
	-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 90-9F */
	-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* A0-AF */
	-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* B0-BF */
	-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* C0-CF */
	-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* D0-DF */
	-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* E0-EF */
	-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1   /* F0-FF */
};


ret_t 
cherokee_buffer_decode_base64 (cherokee_buffer_t *buf)
{
	char space[128];
	int  space_idx = 0;
	int  i, phase  = 0;
	int  d, prev_d = 0;
	int  buf_pos   = 0;

	space_idx = 0;
	phase = 0;
	for (i=0; i < buf->len; i++) {
		d = b64_decode_table[(int) buf->buf[i]];
		if (d != -1) {
			switch (phase) {
			case 0:
				++phase;
				break;
			case 1:
				space[space_idx++] = (( prev_d << 2 ) | ( ( d & 0x30 ) >> 4 ));
				++phase;
				break;
			case 2:
				space[space_idx++] = (( ( prev_d & 0xf ) << 4 ) | ( ( d & 0x3c ) >> 2 ));
				++phase;
				break;
			case 3:
				space[space_idx++] = (( ( prev_d & 0x03 ) << 6 ) | d );
				phase = 0;
				break;
			}
			prev_d = d;
		} 

		if (space_idx == 127) {
			memcpy (buf->buf + buf_pos, space, 127);
			buf_pos += 127;
			space_idx = 0;
		}
        }

	space[space_idx]='\0';

	memcpy (buf->buf + buf_pos, space, space_idx+1);
	buf->len = buf_pos + space_idx;
	
	return ret_ok;
}



/* Documentation: 
 * RFC 1321, `The MD5 Message-Digest Algorithm'
 * http://www.alobbs.com/modules.php?op=modload&name=rfc&file=index&content_file=rfc1321.php
 */

ret_t 
cherokee_buffer_encode_md5 (cherokee_buffer_t *buf)
{
	int i;
	MD5_CTX context;
	unsigned char digest[16];

	MD5Init (&context);
	MD5Update (&context, buf->buf, buf->len);
	MD5Final (digest, &context);

	cherokee_buffer_ensure_size (buf, 34);
	
	for (i=0; i<16; i++) {
		sprintf (buf->buf+(i*2), "%02x", digest[i]);
	}
	buf->buf[32] = '\0';
	buf->len = 32;

	return ret_ok;
}


/* Documentation:
 * RFC 2617, HTTP Authentication: Basic and Digest Access Authentication
 * http://www.alobbs.com/modules.php?op=modload&name=rfc&file=index&content_file=rfc2617.php
 */

ret_t 
cherokee_buffer_encode_hex (cherokee_buffer_t *buf)
{
	unsigned int  i;
	unsigned char j;
	char *new_buf;

	new_buf = (char *) malloc((buf->len * 2)+1);
	if (new_buf == NULL) {
		return ret_error;
	}

	for (i = 0; i < buf->len; i++) {
		j = (buf->buf[i] >> 4) & 0xf;
		if (j <= 9)
			new_buf[i*2] = (j + '0');
		else
			new_buf[i*2] = (j + 'a' - 10);

		j = buf->buf[i] & 0xf;
		if (j <= 9)
			new_buf[i*2+1] = (j + '0');
		else
			new_buf[i*2+1] = (j + 'a' - 10);
	}

	new_buf[buf->len*2] = '\0';

	free (buf->buf);

	buf->len *= 2;
	buf->size = buf->len + 1;
	buf->buf  = new_buf;

	return ret_ok;
}
