/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"

#ifdef HAVE_CRYPT_H
# include <crypt.h>
#endif

#include "validator_htpasswd.h"

cherokee_module_info_t cherokee_htpasswd_info = {
	cherokee_validator,                /* type     */
	cherokee_validator_htpasswd_new    /* new func */
};

ret_t 
cherokee_validator_htpasswd_new (cherokee_validator_htpasswd_t **htpasswd, cherokee_table_t *properties)
{	  
	CHEROKEE_NEW_STRUCT(n,validator_htpasswd);

	/* Init 	
	 */
	MODULE(n)->func_init     = NULL;
	MODULE(n)->func_free     = (module_func_free_t)     cherokee_validator_htpasswd_free;
	VALIDATOR(n)->func_check = (validator_func_check_t) cherokee_validator_htpasswd_check;
	   
	n->htpasswd_file_ref = NULL;

	/* Get the properties
	 */
	if (properties) {
		ret_t ret;
			 
		ret = cherokee_table_get (properties, "file", &n->htpasswd_file_ref);
		if (ret < ret_ok) {
			PRINT_ERROR ("htpasswd validator needs a \"File\" property\n");
			return ret_error;
		}
	}

	*htpasswd = n;
	return ret_ok;
}


ret_t 
cherokee_validator_htpasswd_free (cherokee_validator_htpasswd_t *htpasswd)
{
	free (htpasswd);
	return ret_ok;
}


static ret_t
validate_crypt (cherokee_connection_t *conn, const char *crypted)
{
	ret_t ret;
	char *tmp;

#ifdef HAVE_PTHREAD 
	tmp = crypt_r (conn->passwd->buf, crypted);
#else
	tmp = crypt (conn->passwd->buf, crypted);
#endif

	ret = (strcmp(crypted, tmp)) ? ret_error : ret_ok;
	return ret;
}

static ret_t
validate_md5 (cherokee_connection_t *conn, const char *crypted)
{
	ret_t ret;
	cherokee_buffer_t *tmp;
	
	cherokee_buffer_new (&tmp);

	cherokee_buffer_add_buffer (tmp, conn->user);
	cherokee_buffer_add (tmp, ":", 1);
	cherokee_buffer_add_buffer (tmp, conn->passwd);
	printf ("1->'%s'\n", tmp->buf);
	cherokee_buffer_encode_md5 (tmp);
	printf ("2->'%s'\n", tmp->buf);
	cherokee_buffer_encode_hex (tmp);
	printf ("3->'%s'\n", tmp->buf);
	
	ret = (strcmp(crypted, tmp->buf)) ? ret_error : ret_ok;
	
	cherokee_buffer_free (tmp);

	return ret;
}


ret_t 
cherokee_validator_htpasswd_check (cherokee_validator_htpasswd_t *htpasswd, cherokee_connection_t *conn)
{
	FILE *f;
	int   len;
	char *cryp;
	ret_t ret;
	char *crypted;
	CHEROKEE_TEMP(line, 128);

	/* 1.- Check the login/passwd
	 */	  
	f = fopen (htpasswd->htpasswd_file_ref, "r");
	if (f == NULL) {
		return ret_error;
	}

	ret = ret_error;
	while (!feof(f)) {
		fgets (line, line_size, f);
			 
		len = strlen (line);
		if (line[len-1] == '\n') 
			line[len-1] = '\0';
			 
		/* Split into user and encrypted password. 
		 */
		cryp = strchr (line, ':');
		if (cryp == NULL) continue;
		*cryp++ = '\0';
			 
		/* Is this the right user? 
		 */
		if (strcmp (conn->user->buf, line) != 0) {
			continue;
		}
		
		/* Check the crypted password
		 */
		if (strncmp (cryp, "$apr1$", 6) == 0) {
			ret = validate_md5 (conn, cryp);
		} else {
			ret = validate_crypt (conn, cryp);
		}

		if (ret == ret_ok) {
			break;
		}
	}

	fclose(f);


	/* 2.- Security check:
	 * Is the client trying to download the passwd file?
	 */
	cherokee_buffer_add (conn->local_directory, conn->request->buf+1, conn->request->len-1);  /* 1: add    */

	ret = (strncmp (htpasswd->htpasswd_file_ref, 
			conn->local_directory->buf, 
			conn->local_directory->len) == 0) ? ret_error : ret_ok;

	cherokee_buffer_drop_endding (conn->local_directory, conn->request->len-1);              /* 1: remove */

	if (ret < ret_ok) {
		return ret_error;
	}

	return ret;
}

