/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * This piece of code by:
 *      Ricardo Cardenes Medina <ricardo@conysis.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/tcp.h>
#include <errno.h>
#include <sys/ioctl.h>

#include <unistd.h>
#include <fcntl.h>

#include "socket.h"

#ifdef HAVE_IPV6
typedef struct sockaddr_in6 real_socket_t;
#else
typedef struct sockaddr_in real_socket_t;
#endif

ret_t
cherokee_socket_new (cherokee_socket_t **socket)
{
	CHEROKEE_NEW_STRUCT (n, socket);

	/* Init 
	 */
	n->socket   = -1;
	n->addr_len = sizeof(n->addr_in);
	n->status   = socket_closed;

	/* Return it
	 */
	*socket = n;

	return ret_ok;
}


ret_t
cherokee_socket_free (cherokee_socket_t *socket)
{
	free (socket);
	return ret_ok;
}

ret_t       
cherokee_socket_close (cherokee_socket_t *socket)
{
	int ret;

	if (socket->socket <= 0) {
		return ret_error;
	}

	ret = close (socket->socket);
	socket->socket = -1;
	socket->status = socket_closed;

	return (ret == 0) ? ret_ok : ret_error;
}


const char *
cherokee_socket_ntop (cherokee_socket_t *socket, char *dst, size_t cnt)
{
	char *ret = NULL;
	errno = EAFNOSUPPORT;

	if (SOCKET_FD(socket) <= 0) {
		return NULL;
	}

#ifdef HAVE_INET_PTON
#if defined(HAVE_IPV6)
	if (SOCKET_AF(socket) == AF_INET6) {
		/* We assume that inet_ntop is there too */
		ret = inet_ntop(AF_INET6,
//				&((struct sockaddr_in6 *)CHEROKEE_SOCKET_ADDR(socket))->sin6_addr,
				&((struct sockaddr_in6 *)&SOCKET_ADDR(socket))->sin6_addr,
				dst, cnt);
	}
#endif /* HAVE_IPV6 */
	if (SOCKET_AF(socket) == AF_INET) {
		ret = inet_ntop(AF_INET,
//				&((struct sockaddr_in *)CHEROKEE_SOCKET_ADDR(socket))->sin_addr,
				&((struct sockaddr_in *)&SOCKET_ADDR(socket))->sin_addr,
				dst, cnt);
	}
#else /* !HAVE_INET_PTON */
	if (SOCKET_AF(socket) == AF_INET) {
		ret = inet_ntoa(((struct sockaddr_in *)SOCKET_ADDR(socket))->sin_addr);
		strncpy(dst, ret, cnt);
		ret = dst;
	}
#endif /* HAVE_INET_PTON */

	return ret;
}


ret_t
cherokee_socket_accept (cherokee_socket_t  *socket, int server_socket)
{
	int tmp = 1;
	int new_socket;

	/* Get the new connection
	 */
	new_socket = accept (server_socket, (struct sockaddr *)&socket->addr_in, &socket->addr_len);
	if (new_socket <= 0) {
		return ret_error;
	}

	/* Set some parameters to the socket
	 */

	/* <EXPERIMENTAL>
	 */
	fcntl (new_socket, F_SETFD, 1);  /* close-on-exec */
	
	/* Disable Nagle's algorithm for this connection.
	 * Written data to the network is not buffered pending
	 * acknowledgement of previously written data.
	 */
 	setsockopt (new_socket, IPPROTO_TCP, TCP_NODELAY, &tmp, sizeof(tmp));

	/*  </EXPERIMENTAL>
	 */
	
	/* Enables nonblocking I/O.
	 */
	ioctl (new_socket, FIONBIO, &tmp);

	SOCKET_FD(socket) = new_socket;

	return ret_ok;
}


ret_t       
cherokee_socket_set_status (cherokee_socket_t  *socket, cherokee_socket_status_t status)
{
	socket->status = status;

	if (status == socket_closed) {
		cherokee_socket_close (socket);
	}

	return ret_ok;
}
