/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>

#include "module_loader.h"
#include "loader.autoconf.h"

typedef void *func_new_t;

static ret_t
load_static_linked_modules (cherokee_module_loader_t *loader)
{
/* Yeah, yeah.. I know, it's horrible. :-(
 * If you know a better way to manage dinamic/static modules,
 * please, let me know.
 */
#include "loader.autoconf.inc"

	return ret_ok;
}


ret_t 
cherokee_module_loader_new  (cherokee_module_loader_t **loader)
{
	ret_t ret;
	
	CHEROKEE_NEW_STRUCT (n, module_loader);
	
	ret = cherokee_table_new(&n->table);
	if (ret < ret_ok) return ret;
	
	ret = load_static_linked_modules (n);
	if (ret < ret_ok) return ret;

	*loader = n;

	return ret_ok;
}


ret_t 
cherokee_module_loader_free (cherokee_module_loader_t *loader)
{
	cherokee_table_free (loader->table);

	free (loader);

	return ret_ok;
}


static void *
get_sym_from_enviroment (const char *sym)
{
	return dlsym (NULL, sym);
}


static void *
get_sym_from_dlopen_handler (void *dl_handle, const char *sym)
{
	void *ret;
	   
	/* Get the sym
	 */
	ret = (void *) dlsym(dl_handle, sym);
	if (ret == NULL) {
		PRINT_ERROR ("Can't get '%s' symbol from %p: %s\n", sym, dl_handle, dlerror());
		return NULL;
	}
	
	return ret;
}


static void
dylib_open (const char *libname, void **handler_out) 
{
	void *lib;
	int   flags;

#ifdef HAVE_RTLDNOW	
	flags = RTLD_NOW;
#else
	flags = RTLD_LAZY;
#endif

#ifdef HAVE_RTLDGLOBAL
	flags |= RTLD_GLOBAL;
#endif
	   
	/* Open the library	
	 */
	snprintf (gbl_buffer, gbl_buffer_size, 
		  CHEROKEE_PLUGINDIR "/libcherokee_%s." SO_SUFFIX, libname);
	
	lib = dlopen (gbl_buffer, flags);
	if (lib == NULL) {
		PRINT_ERROR ("dlopen (%s): %s %s\n", gbl_buffer, libname, dlerror());
		return;
	}

	*handler_out = lib;
}


static void
execute_init_func (const char *module, void *dl_handler)
{
	char * init_name;
	int    init_name_len;
	void (*init_func) (void);
	
	/* Build the init func
	 */
	init_name_len = strlen(module) + 7;
	init_name = (char *)malloc(init_name_len);
	snprintf (init_name, init_name_len, "%s_init", module);

	/* Get the function
	 */
	if (dl_handler == NULL) {
		init_func = get_sym_from_dlopen_handler (dl_handler, init_name);
	} else {
		init_func = get_sym_from_enviroment (init_name);
	}

	/* Execute the init func
	 */
	if (init_func != NULL) {
		init_func();
	}
	
	free (init_name);
}


static void
get_info (const char *module, cherokee_module_info_t **info, void **dl_handler)
{
	char * info_name;
	int    info_name_len;
	
	/* Build the init func
	 */
	info_name_len = strlen(module) + 16;
	info_name = (char *)malloc(info_name_len);
	snprintf (info_name, info_name_len, "cherokee_%s_info", module);
	
	/* Maybe it's statically linked
	 */
	*info       = get_sym_from_enviroment (info_name);
	*dl_handler = NULL;
	
	/* Or maybe we have to load a dinamic library
	 */	
	if (*info == NULL) {
		dylib_open (module, dl_handler);
		*info = get_sym_from_dlopen_handler (*dl_handler, info_name);
	}
	
	free (info_name);
}


ret_t 
cherokee_module_loader_load (cherokee_module_loader_t *loader, const char *modname)
{
	ret_t                   ret;
	void                   *dl_handle;
	cherokee_module_info_t *info;
	
	ret = cherokee_table_get (loader->table, modname, &dl_handle);
	if (ret == ret_ok) return ret_ok;
	
	/* Get the module info
	 */
	get_info (modname, &info, &dl_handle);
	if (info == NULL) {
		PRINT_ERROR ("Can't read \"info\" structure from %s\n", modname);
		return ret_error;
	}
	
	/* Execute init function
	 */
	execute_init_func (modname, dl_handle);
	
	ret = cherokee_table_add (loader->table, modname, info);
	if (ret == ret_ok) return ret_ok;
	
	return ret_ok;
}


ret_t 
cherokee_module_loader_get (cherokee_module_loader_t *loader, const char *modname, cherokee_module_info_t **info)
{
	return cherokee_table_get (loader->table, modname, info);
}
