/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

/*
** Copyright (c) 1999,2000 by Jef Poskanzer <jef@acme.com>
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
*/

/* Useful documentation:
 *
 * http://www.opengroup.org/onlinepubs/007904975/functions/poll.html
 */

#include "common.h"
#include "fdpoll.h"

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else 
#include <time.h>
#endif

#ifdef HAVE_SYS_RESOURCE_H
#include <sys/resource.h>
#else
#include <resource.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif


#ifdef HAVE_PTHREAD
# define LOCK_WRITER(fdp) pthread_rwlock_wrlock (&fdp->rwlock)
# define LOCK_READER(fdp) pthread_rwlock_rdlock (&fdp->rwlock)
# define UNLOCK(fdp)      pthread_rwlock_unlock (&fdp->rwlock)
#else
# define LOCK_WRITER(fdp)
# define LOCK_READER(fdp)
# define UNLOCK(fdp)
#endif

#define POLL_ERROR  (POLLHUP | POLLERR)
#define POLL_READ   (POLLIN | POLLPRI | POLL_ERROR)
#define POLL_WRITE  (POLLOUT | POLL_ERROR)


static inline int
get_fd_limit (void)
{
	   int nfiles;
	   struct rlimit rl;

	   if (getrlimit (RLIMIT_NOFILE, &rl) == 0)
	   {
			 nfiles = rl.rlim_cur;

			 if (rl.rlim_max == RLIM_INFINITY) {
				 /* arbitrary */
				 rl.rlim_cur = 8192;  
			 } else if (rl.rlim_max > rl.rlim_cur) {
				 rl.rlim_cur = rl.rlim_max;
			 }
			 
			 if (setrlimit( RLIMIT_NOFILE, &rl ) == 0) {
				    nfiles = rl.rlim_cur;
			 }

			 return nfiles;
	   }

	   return getdtablesize();
}



/***********************************************************************/
/* /dev/epoll                                                          */
/*                                                                     */
/* #include <sys/epoll.h>                                              */
/*                                                                     */
/* int epoll_ctl(int epfd, int op, int fd, struct epoll_event *event); */
/*                                                                     */
/***********************************************************************/
#ifdef USE_EPOLL

ret_t
fdpoll_epoll_new (cherokee_fdpoll_t **fdp)
{
	   CHEROKEE_NEW_STRUCT (n, fdpoll);

	   /* Look for max fd limit
	    */
	   n->nfiles       = get_fd_limit();
	   n->ep_readyfds  = 0;
	   n->ep_events    = (struct epoll_event *) malloc (sizeof(struct epoll_event) * n->nfiles);
	   n->epoll_rs2idx = (int *) malloc (sizeof(int) * n->nfiles);
	   n->epoll_idx2rs = (int *) malloc (sizeof(int) * n->nfiles);

	   n->ep_fd = epoll_create (n->nfiles + 1);
	   if (n->ep_fd < 0) {
		   PRINT_ERROR ("Fatal Error: Couldn't get /dev/epoll descriptor: epoll_create(%d)\n", n->nfiles + 1);
		   exit (EXIT_EPOLL_CREATE);
	   }

	   return_if_fail (n->ep_events,    ret_nomem);
	   return_if_fail (n->epoll_rs2idx, ret_nomem);
	   return_if_fail (n->epoll_idx2rs, ret_nomem);

#ifdef HAVE_PTHREAD
	   pthread_rwlock_init(&n->rwlock, NULL); 
#endif

	   /* Return the object
	    */
	   *fdp = n;

	   return ret_ok;
}

ret_t
fdpoll_epoll_free (cherokee_fdpoll_t *fdp)
{
	close (fdp->ep_fd);

	free (fdp->ep_events);
	free (fdp->epoll_rs2idx);
	free (fdp->epoll_idx2rs);
	
	free (fdp);
	
	return ret_ok;
}

ret_t
fdpoll_epoll_add (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	struct epoll_event ev;

	ev.data.fd = fd;
	switch (rw)
	{
	case 0: 
		ev.events = EPOLLIN; 
		break;
	case 1: 
		ev.events = EPOLLOUT; 
		break;
	default:
		ev.events = 0;
		break;
	}

	LOCK_WRITER(fdp);  // NEEDED?

	if (epoll_ctl (fdp->ep_fd, EPOLL_CTL_ADD, fd, &ev) < 0) 
	{
		PRINT_ERROR ("Error: epoll_ctl (EPOLL_CTL_ADD)\n");
		exit (EXIT_EPOLL_CTL_ADD);
	}

	UNLOCK(fdp);

	return ret_ok;
}

void inline
fdpoll_epoll_set_mode (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	struct epoll_event ev;

	ev.data.fd = fd;
	switch (rw)
	{
	case 0: 
		ev.events = EPOLLIN; 
		break;
	case 1: 
		ev.events = EPOLLOUT; 
		break;
	default:
		ev.events = 0;
		break;
	}

	LOCK_WRITER(fdp);  // NEEDED?

	if (epoll_ctl(fdp->ep_fd, EPOLL_CTL_MOD, fd, &ev) < 0) 
	{
		PRINT_ERROR ("Error: epoll_ctl (EPOLL_CTL_MOD)\n");
		exit (EXIT_EPOLL_CTL_MOD);
	}

	UNLOCK(fdp);

	return ret_ok;
}

ret_t
fdpoll_epoll_del (cherokee_fdpoll_t *fdp, int fd)
{
	struct epoll_event ev;

	if (epoll_ctl(fdp->ep_fd, EPOLL_CTL_DEL, fd, &ev) < 0)
	{
		PRINT_ERROR ("Error: epoll_ctl (EXIT_EPOLL_CTL_DEL)\n");
		exit (EXIT_EPOLL_CTL_DEL);
	}

	return ret_ok;
}

inline int
fdpoll_epoll_watch (cherokee_fdpoll_t *fdp, int timeout_msecs)
{
	int i, ridx;

	fdp->ep_readyfds = epoll_wait (fdp->ep_fd, fdp->ep_events, fdp->nfiles, timeout_msecs);

	for (i = 0, ridx = 0; i < fdp->ep_readyfds; ++i) {
		if (fdp->ep_events[i].events & (EPOLLIN | EPOLLOUT | EPOLLERR | EPOLLHUP))
		{
			fdp->epoll_idx2rs[fdp->ep_events[i].data.fd] = i;
			fdp->epoll_rs2idx[ridx] = i;
			ridx++;
		}
	}

	return ridx;
}

int
fdpoll_epoll_check (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	int fdidx = fdp->epoll_idx2rs[fd];

	if (fdidx < -1 || fdidx >= fdp->nfiles)
	{
		PRINT_ERROR ("Error: bad fdidx (%d) in epoll_check_fd!\n", fdidx);
		return 0;
	}

	
	if ((fdidx == -1) ||
	    (fdp->ep_events[fdidx].data.fd != fd) || 
	    (fdp->ep_events[fdidx].events & EPOLLERR))
	{
		return 0;
	}

//	switch (fdp->fd_rw[fd])
	switch (rw)
	{
	case 0: 
		return fdp->ep_events[fdidx].events & (EPOLLIN | EPOLLERR | EPOLLHUP);
	case 1: 
		return fdp->ep_events[fdidx].events & (EPOLLOUT | EPOLLERR | EPOLLHUP);
	}

	return 0;
}

ret_t
fdpoll_epoll_reset (cherokee_fdpoll_t *fdp, int fd)
{
	fdp->epoll_idx2rs[fd] = -1;

	return ret_ok;
}

#endif /* USE_EPOLL */



/***********************************************************************/
/* poll()                                                              */
/*                                                                     */
/* #include <sys/poll.h>                                               */
/*                                                                     */
/* int poll(struct pollfd *ufds, unsigned int nfds, int timeout);      */
/*                                                                     */
/***********************************************************************/
#ifdef USE_POLL

ret_t
fdpoll_poll_new (cherokee_fdpoll_t **fdp)
{
	   /* Get memory
	    */
	   CHEROKEE_NEW_STRUCT (n, fdpoll);

	   /* Look for max fd limit
	    */
	   n->nfiles = get_fd_limit();
	   n->npollfds = 0;
	 	   
	   /* Get memory
	    */
	   n->pollfds = (struct pollfd *) malloc (sizeof(struct pollfd) * n->nfiles);
	   return_if_fail (n->pollfds, ret_nomem);
	   
	   n->fdidx = (int*) malloc (sizeof(int) * n->nfiles);
	   return_if_fail (n->fdidx, ret_nomem);

#ifdef HAVE_PTHREAD
	   pthread_rwlock_init(&n->rwlock, NULL); 
#endif

	   /* Return it
	    */
	   *fdp = n;

	   return ret_ok;
}

ret_t
fdpoll_poll_free (cherokee_fdpoll_t *fdp)
{
	free (fdp->pollfds);
	free (fdp->fdidx);
	
	free (fdp);
	
	return ret_ok;
}

ret_t
fdpoll_poll_add (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	LOCK_WRITER(fdp);

	fdp->pollfds[fdp->npollfds].fd = fd;
	switch (rw)
        {
        case 0:  
		fdp->pollfds[fdp->npollfds].events = POLL_READ; 
		break;
        case 1: 
		fdp->pollfds[fdp->npollfds].events = POLL_WRITE;
		break;
        }
	fdp->fdidx[fd] = fdp->npollfds;
	fdp->npollfds++;

	UNLOCK(fdp);
	
	return ret_ok;
}

void inline
fdpoll_poll_set_mode (cherokee_fdpoll_t *fdp, int fd, int rw)
{
//	LOCK_WRITER(fdp);

	fdp->pollfds[fdp->fdidx[fd]].events = rw ? POLL_WRITE : POLL_READ;

//	UNLOCK(fdp);
}

ret_t
fdpoll_poll_del (cherokee_fdpoll_t *fdp, int fd)
{
	int idx;
	ret_t ret = ret_ok;

	LOCK_WRITER(fdp);
	   
	idx = fdp->fdidx[fd];
	if (idx < 0 || idx >= fdp->nfiles) {
		ret = ret_error;
		goto out;
	}

	fdp->npollfds--;
	fdp->pollfds[idx] = fdp->pollfds[fdp->npollfds];
	fdp->fdidx[fdp->pollfds[idx].fd] = idx;
	
out:
	UNLOCK(fdp);
	return ret;
}

inline int
fdpoll_poll_watch (cherokee_fdpoll_t *fdp, int timeout_msecs)
{
	int ret;

//	LOCK_WRITER(fdp);

	ret = poll (fdp->pollfds, fdp->npollfds, timeout_msecs);

//	UNLOCK(fdp);
	return ret;
}

int
fdpoll_poll_check (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	int ret;

//	LOCK_READER(fdp);

	switch (rw) {
        case 0: 
		ret = fdp->pollfds[fdp->fdidx[fd]].revents & (POLLIN | POLLERR | POLLHUP | POLLNVAL);
		break;
        case 1: 
		ret = fdp->pollfds[fdp->fdidx[fd]].revents & (POLLOUT | POLLERR | POLLHUP | POLLNVAL);
		break;
        default: 
		SHOULDNT_HAPPEN;
	}


//	UNLOCK(fdp);
	return ret;
}

ret_t
fdpoll_poll_reset (cherokee_fdpoll_t *fdp, int fd)
{
}

#endif /* USE_EPOLL */





