/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 * 
 * This piece of code by:
 *      Pablo Neira Ayuso <pneira@optimat.com>
 *      Miguel Angel Ajo Pelayo <ajo@godsmaze.org> 
 * 
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <unistd.h>
#include <syslog.h>

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else 
#include <time.h>
#endif

#include "logger_ncsa.h"
#include "connection.h"
#include "server.h"


/* Some constants
 */
static char *method[]  = {"GET", "POST", "HEAD", "UNKNOWN", NULL};
static char *version[] = {"HTTP/0.9", "HTTP/1.0", "HTTP/1.1", "UNKNOWN", NULL};
static char *month[]   = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
			 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec", NULL};

#define IN_ADDR(c) ((struct in_addr)(c).sin_addr)


ret_t
cherokee_logger_ncsa_new  (cherokee_logger_t **logger, cherokee_table_t *properties)
{
	CHEROKEE_NEW_STRUCT (n, logger_ncsa);
	
	/* Init the base class object
	 */
	cherokee_logger_init_base(LOGGER(n));

	LOGGER(n)->init         = (logger_func_init_t) cherokee_logger_ncsa_init;
	LOGGER(n)->free         = (logger_func_free_t) cherokee_logger_ncsa_free;
	LOGGER(n)->write_error  = (logger_func_write_error_t) cherokee_logger_ncsa_write_error;
	LOGGER(n)->write_access = (logger_func_write_access_t) cherokee_logger_ncsa_write_access;
	LOGGER(n)->write_string = (logger_func_write_string_t) cherokee_logger_ncsa_write_string;

	*logger = LOGGER(n);
	
	/* Init
	 */
	n->errorlog_fd        = NULL;
	n->accesslog_fd       = NULL;
	n->accesslog_filename = NULL;
	n->errorlog_filename  = NULL;

	if (properties != NULL) {
		n->accesslog_filename = cherokee_table_get (properties, "AccessLog");
		n->errorlog_filename  = cherokee_table_get (properties, "ErrorLog");
	}
	
	return ret_ok;
}


ret_t 
cherokee_logger_ncsa_init (cherokee_logger_ncsa_t *logger)
{
	if ((logger->accesslog_filename == NULL) ||
	    (logger->errorlog_filename == NULL))
	{
		openlog ("Cherokee", LOG_CONS | LOG_PID | LOG_NDELAY, LOG_LOCAL1);
		return ret_ok;
	}

	logger->accesslog_fd = fopen (logger->accesslog_filename, "a+");
	if (logger->accesslog_fd == NULL) {
		PRINT_ERROR("cherokee_logger_ncsa: error opening %s for append\n", logger->accesslog_filename); 
		return ret_error;
	}

        logger->errorlog_fd  = fopen (logger->errorlog_filename, "a+");
	if (logger->errorlog_fd == NULL) {
		PRINT_ERROR("cherokee_logger_ncsa: error opening %s for append\n", logger->errorlog_filename); 
		return ret_error;
	}

	return ret_ok;
}


ret_t
cherokee_logger_ncsa_free (cherokee_logger_ncsa_t *logger)
{
	int ret, n = 2;

	if (logger->errorlog_fd != NULL) {
		ret = fclose (logger->errorlog_fd);
		n--;
	}

	if (logger->accesslog_fd != NULL) {
		ret |= fclose (logger->errorlog_fd);
		n--;
	}

	if (n != 0) {
		closelog();
	}

	free (logger);
	
	return (ret == 0) ? ret_ok : ret_error;
}


ret_t 
cherokee_logger_ncsa_write_error  (cherokee_logger_ncsa_t *logger, const char *format, ...)
{
	va_list ap;

	if (logger->errorlog_fd != NULL) {
		int ret = 0;

		va_start (ap, format);
		ret += vfprintf (logger->errorlog_fd, format, ap);
		va_end (ap);

		return (ret > 0) ? ret_ok : ret_error;
	}

	va_start (ap, format);
	vsyslog (LOG_ERR, format, ap);
	va_end (ap);

	return ret_ok;
}


ret_t 
cherokee_logger_ncsa_write_string (cherokee_logger_ncsa_t *logger, const char *format, ...)
{
	va_list ap;

	if (logger->accesslog_fd != NULL) {
		int ret = 0;

		va_start (ap, format);
		ret += vfprintf (logger->accesslog_fd, format, ap);
		va_end (ap);

		return (ret > 0) ? ret_ok : ret_error;
	} 

	va_start (ap, format);
	vsyslog (LOG_INFO, format, ap);
	va_end (ap);

	return ret_ok;
}


ret_t
cherokee_logger_ncsa_write_access (cherokee_logger_ncsa_t *logger, cherokee_connection_t *cnt)
{
  	long int z;
	int  len;
	struct tm conn_time;
	CHEROKEE_TEMP (tmp, 200);
    
	/* thread-safe version of localtime 
	 */
	localtime_r(&CONN_SRV(cnt)->bogo_now, &conn_time);
	
#ifdef HAVE_INT_TIMEZONE
	z = - (timezone / 60);
#else
#warning TODO
	z = 0;
#endif

	len = snprintf (tmp, tmp_size,
			"%s - - [%02d/%s/%d:%02d:%02d:%02d %c%02d%02d] \"%s %s %s\" %d %ld\n",
			inet_ntoa(IN_ADDR(cnt->addr_in)),
			conn_time.tm_mday, 
			month[conn_time.tm_mon], 
			1900 + conn_time.tm_year,
			conn_time.tm_hour, 
			conn_time.tm_min, 
			conn_time.tm_sec,
			(z < 0) ? '-' : '+', 
			(int) z/60, 
			(int) z%60, 
			method[cnt->method],
			cnt->request->buf, 
			version[cnt->version], 
			cnt->error_code,
			cnt->range_end - cnt->range_start);
	
	if (logger->accesslog_fd != NULL) {
		size_t wrote;

		wrote = fwrite (tmp, 1, len, logger->accesslog_fd); 
                fflush(logger->accesslog_fd);

		return (wrote > 0) ? ret_ok : ret_error;
	}

	syslog (LOG_INFO, tmp);
	return ret_ok;
}




/*   Library init function
 */

static int _ncsa_is_init = 0;

void
ncsa_init ()
{
	/* Init flag
	 */
	if (_ncsa_is_init) {
		return;
	}
	_ncsa_is_init = 1;
}


