/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "logger_table.h"
#include "static_loggers.h"
#include "library.h"


ret_t 
cherokee_logger_table_new  (cherokee_logger_table_t **lt)
{
	   int i;
	   CHEROKEE_NEW_STRUCT(n, logger_table);

	   /* Init the table
	    */
	   cherokee_table_init (&n->table);

	   /* Add the loggers compiled statically
	    */
	   for (i=0; static_loggers[i].name; i++) {
			 ret_t ret;
			 ret = cherokee_table_add (&n->table, static_loggers[i].name, static_loggers[i].func); 
			 if (ret != ret_ok) 
				    return ret;
	   }

	   /* Return the object
	    */
	   *lt = n;

	   return ret_ok;
}


static inline logger_func_new_t
load_logger_library (char *plugin_name)
{
	cherokee_library_t *clib;
	func_new_t ret;
	char sym[200];
	
	cherokee_library_new (&clib, plugin_name);

 	snprintf (sym, 200, "cherokee_logger_%s_new", plugin_name);
	cherokee_library_get (clib, sym, &ret);

	cherokee_library_free (clib);
	
	return ret;
}


ret_t 
cherokee_logger_table_clean (cherokee_logger_table_t *et)
{
	   return cherokee_table_clean (&et->table);
}


ret_t 
cherokee_logger_table_free (cherokee_logger_table_t *et)
{
	   return cherokee_table_free (&et->table);
}

ret_t
cherokee_logger_table_get (cherokee_logger_table_t *et, char *logger, cherokee_logger_table_entry_t **entry) 
{
	cherokee_logger_table_entry_t *n;

	/* Get the constructor from the table
	 */
	n = cherokee_table_get (&et->table, logger);
	if (n == NULL) return ret_error;
	
	/* Return the object
	 */
	*entry = n;

	return ret_ok;
}


ret_t
cherokee_logger_table_load (cherokee_logger_table_t *lt, char *logger)
{
	ret_t ret;
	logger_func_new_t func;

	func = load_logger_library (logger);
	if (func == NULL) {
		PRINT_ERROR ("Can't load %s logger\n", logger);
		return ret_error;
	}

	{
		   CHEROKEE_NEW (n, logger_table_entry);
		   n->func_new = func;

		   ret = cherokee_table_add (&lt->table, logger, n);
		   if (ret != ret_ok) return ret;
	}

	return ret_ok;

}


ret_t 
cherokee_logger_table_new_logger (cherokee_logger_table_t *et, char *logger, cherokee_table_t *props, cherokee_logger_t **new_logger)
{
	ret_t ret;
	cherokee_logger_table_entry_t *entry;

	ret = cherokee_logger_table_get (et, logger, &entry);
	if (ret != ret_ok) return ret;

	ret = entry->func_new ((void **)new_logger, props);
	if (ret != ret_ok) return ret;

	ret = cherokee_logger_init (*new_logger);
	return ret;
}


/* 
 * Logger entry methods 
 */

ret_t 
cherokee_logger_table_entry_new (cherokee_logger_table_entry_t **lentry)
{
	CHEROKEE_NEW_STRUCT (n, logger_table_entry);

	n->func_new   = NULL;

	*lentry = n;

	return ret_ok;
}

ret_t 
cherokee_logger_table_entry_free (cherokee_logger_table_entry_t *lentry)
{
	   free (lentry);
	   return ret_ok;
}
