/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "library.h"



static void *
get_sym_from_dlopen_handler (void *dl_handle, const char *sym)
{
	   void *ret;

	   /* Get the sym
	    */
	   ret = (void *) dlsym(dl_handle, sym);
	   if (ret == NULL) {
			 fprintf (stderr, "dlsym (%s): %s\n", sym, dlerror());
			 return NULL;
	   }
	   
	   return ret;
}


static void *
get_sym_from_dylib (const char *dylib, char *sym, func_new_t *out_lib)
{
	void  *lib;
	int    dl_flags;
	void  *ret;
	
	/* Load the library
	 */	

#ifdef HAVE_RTLDNOW
	dl_flags = RTLD_NOW;
#else
	dl_flags = RTLD_LAZY;
#endif

#ifdef HAVE_RTLDGLOBAL
	dl_flags |= RTLD_GLOBAL;
#endif

	/* Maybe the sym is accessible
	 */
	if ((ret = dlsym(NULL, sym)) != NULL) return ret;

	/* Open the library
	 */
	snprintf (gbl_buffer, gbl_buffer_size, CHEROKEE_PLUGINDIR "/libcherokee_%s." SO_SUFFIX, dylib);

	lib = dlopen (gbl_buffer, dl_flags);
	if (lib == NULL) {
		fprintf (stderr, "dlopen (%s): %s %s\n", gbl_buffer, sym, dlerror());
		return NULL;
	}
	*out_lib = lib;

	return get_sym_from_dlopen_handler(lib, sym);
}


ret_t 
cherokee_library_new  (cherokee_library_t **lib, const char *module)
{
	   char * init_name;
	   int    init_name_len;
	   void (*init_func) (void);

	   CHEROKEE_NEW_STRUCT (n, library);

	   /* Build the init func
	    */
	   init_name_len = strlen(module) + 7;
	   init_name = (char *)malloc(init_name_len);
	   snprintf (init_name, init_name_len, "%s_init", module);

	   /* Get the func
	    */
	   init_func = get_sym_from_dylib (module, init_name, &n->lib);

	   /* Execute the init func
	    */
	   if (init_func != NULL) {
			 init_func();
	   }

	   *lib=n;

	   return ret_ok;
}


ret_t 
cherokee_library_free (cherokee_library_t *lib)
{
	   free (lib);

	   return ret_ok;
}


ret_t
cherokee_library_get (cherokee_library_t *lib, const char *symbol_name, func_new_t *symbol)
{
	   void *sym;

	   sym = get_sym_from_dlopen_handler (lib->lib, symbol_name);
	   *symbol = sym;

	   return ret_ok;
}

