/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "encoder_table.h"
#include "static_encoders.h"


ret_t 
cherokee_encoder_table_new  (cherokee_encoder_table_t **et)
{
	   int i;
	   cherokee_encoder_table_t *n = (cherokee_encoder_table_t *) malloc (sizeof(cherokee_encoder_table_t));

	   /* Init the table
	    */
	   cherokee_table_init (&n->table);

	   /* Add the encoders compiled statically
	    */
	   for (i=0; static_encoders[i].name; i++) {
			 ret_t ret;
			 ret = cherokee_table_add (&n->table, static_encoders[i].name, static_encoders[i].func); 
			 if (ret != ret_ok) 
				    return ret;
	   }

	   /* Return the object
	    */
	   *et = n;

	   return ret_ok;
}


static inline encoder_func_new_t
load_encoder_library (char *plugin_name)
{
	char lib[200];
	char sym[200];

	void (*init) (void);
	encoder_func_new_t ret;

 	snprintf (lib, 200, "libcherokee_%s." SO_SUFFIX, plugin_name);
	snprintf (sym, 200, "%s_init", plugin_name);

	init = get_sym_from_dylib (lib, sym);
	if (init != NULL) {
		init();
	}

 	snprintf (sym, 200, "cherokee_encoder_%s_new", plugin_name);
	ret = get_sym_from_dylib (lib, sym);
	
	return ret;
}


ret_t 
cherokee_encoder_table_clean (cherokee_encoder_table_t *et)
{
	   return cherokee_table_clean (&et->table);
}


ret_t 
cherokee_encoder_table_free (cherokee_encoder_table_t *et)
{
	   return cherokee_table_free (&et->table);
}

ret_t
cherokee_encoder_table_get (cherokee_encoder_table_t *et, char *encoder, cherokee_encoder_table_entry_t **entry) 
{
	cherokee_encoder_table_entry_t *n;

	/* Get the constructor from the table
	 */
	n = cherokee_table_get (&et->table, encoder);
	if (n == NULL) return ret_error;
	
	/* Return the object
	 */
	*entry = n;

	return ret_ok;
}


ret_t
cherokee_encoder_table_load (cherokee_encoder_table_t *et, char *encoder)
{
	ret_t ret;
	encoder_func_new_t func;

	func = load_encoder_library (encoder);
	if (func == NULL) {
		PRINT_ERROR ("Can't load %s encoder\n", encoder);
		return ret_error;
	}

	{
		CHEROKEE_NEW (n, encoder_table_entry);

		ret = cherokee_encoder_entry_set_new_func (n, func);
		if (ret != ret_ok) return ret;

		ret = cherokee_table_add (&et->table, encoder, n);
		if (ret != ret_ok) return ret;
	}

	return ret_ok;
}


ret_t 
cherokee_encoder_table_new_encoder (cherokee_encoder_table_t *et, char *encoder, char *ext, cherokee_encoder_t **new_encoder)
{
	ret_t ret;
	int   make_object = 1;
	cherokee_encoder_table_entry_t *entry;
	cherokee_matching_list_t       *matching;

	ret = cherokee_encoder_table_get (et, encoder, &entry);
	if (ret != ret_ok) return ret;

	if (cherokee_encoder_entry_has_matching_list (entry)) {
		ret = cherokee_encoder_entry_get_matching_list (entry, &matching);
		if (ret != ret_ok) return ret;

		make_object = cherokee_matching_list_match (matching, ext);
	}

	if (make_object) {
		encoder_func_new_t new_func;

		ret = cherokee_encoder_entry_get_new_func (entry, &new_func);
		if (ret != ret_ok) return ret;
		
		ret = new_func (new_encoder);
		if (ret != ret_ok) return ret;
	}

	return ret_ok;
}


/* 
 * Encoder entry methods 
 */

ret_t 
cherokee_encoder_table_entry_new (cherokee_encoder_table_entry_t **eentry)
{
	CHEROKEE_NEW_STRUCT (n, encoder_table_entry);

	n->matching = NULL;
	n->func_new = NULL;

	*eentry = n;

	return ret_ok;
}


ret_t 
cherokee_encoder_entry_set_new_func (cherokee_encoder_table_entry_t *eentry, encoder_func_new_t new_func)
{
	eentry->func_new = new_func;
	return ret_ok;
}

ret_t 
cherokee_encoder_entry_get_new_func (cherokee_encoder_table_entry_t *eentry, encoder_func_new_t *new_func)
{
	*new_func = eentry->func_new;
	return ret_ok;
}


ret_t 
cherokee_encoder_entry_set_matching_list (cherokee_encoder_table_entry_t *eentry, cherokee_matching_list_t *matching)
{
	eentry->matching = matching;
	return ret_ok;
}

int
cherokee_encoder_entry_has_matching_list (cherokee_encoder_table_entry_t *eentry)
{
	return (eentry->matching != NULL);
}


ret_t 
cherokee_encoder_entry_get_matching_list (cherokee_encoder_table_entry_t *eentry, cherokee_matching_list_t **matching)
{
	*matching = eentry->matching;
	return ret_ok;
}
