/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "encoder_table.h"
#include "static_encoders.h"


ret_t 
cherokee_encoder_table_new  (cherokee_encoder_table_t **et)
{
	   int i;
	   cherokee_encoder_table_t *n = (cherokee_encoder_table_t *) malloc (sizeof(cherokee_encoder_table_t));

	   /* Init the table
	    */
	   cherokee_table_init (&n->table);

	   /* Add the encoders compiled statically
	    */
	   for (i=0; static_encoders[i].name; i++) {
			 ret_t ret;
			 ret = cherokee_table_add (&n->table, static_encoders[i].name, static_encoders[i].func); 
			 if (ret != ret_ok) 
				    return ret;
	   }

	   /* Return the object
	    */
	   *et = n;

	   return ret_ok;
}


ret_t 
cherokee_encoder_table_clean (cherokee_encoder_table_t *et)
{
	   return cherokee_table_clean (&et->table);
}


ret_t 
cherokee_encoder_table_free (cherokee_encoder_table_t *et)
{
	   return cherokee_table_free (&et->table);
}


static inline encoder_func_new_t
load_encoder_library (char *plugin_name)
{
	char lib[200];
	char sym[200];

	void (*init) (void);
	encoder_func_new_t ret;

 	snprintf (lib, 200, "libcherokee_%s." SO_SUFFIX, plugin_name);
	snprintf (sym, 200, "%s_init", plugin_name);

	init = get_sym_from_dylib (lib, sym);
	if (init != NULL) {
		init();
	}

 	snprintf (sym, 200, "cherokee_encoder_%s_new", plugin_name);
	ret = get_sym_from_dylib (lib, sym);
	
	return ret;
}


ret_t
cherokee_encoder_table_get (cherokee_encoder_table_t *et, char *encoder, cherokee_encoder_t **new) 
{
	ret_t ret;
	encoder_func_new_t func;

	/* Get the constructor from the table
	 */
	func = cherokee_table_get (&et->table, encoder);
	if (func == NULL) return ret_error;
	
	/* If it exists return a new object
	 */
	ret = func(new);
	return ret;
}


ret_t
cherokee_encoder_table_add (cherokee_encoder_table_t *et, char *encoder, void *new_func)
{
	   return cherokee_table_add (&et->table, encoder, new_func);
}


ret_t
cherokee_encoder_table_load (cherokee_encoder_table_t *et, char *encoder)
{
	ret_t ret;
	encoder_func_new_t  func;
	
	func = load_encoder_library (encoder);

	ret = cherokee_table_add (&et->table, encoder, func);
	if (ret != ret_ok) return ret;
	
	return ret_ok;
}
