/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 * 
 * This piece of code by:
 *      Pablo Neira Ayuso <pneira@optimat.com>
 *      Miguel Angel Ajo Pelayo <ajo@godsmaze.org> 
 * 
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <sys/types.h>
#include <unistd.h>

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else 
#include <time.h>
#endif

#include "logger_ncsa.h"
#include "connection.h"
#include "server.h"


#define IN_ADDR(c)	((struct in_addr) (c).sin_addr)

static char *month[]   = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
			  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec", NULL};

static char *method[]  = {"GET", "POST", "HEAD", "UNKNOWN", NULL};

static char *version[] = {"HTTP/0.9", "HTTP/1.0", "HTTP/1.1", "UNKNOWN", NULL};



ret_t
cherokee_logger_ncsa_new  (cherokee_logger_t **logger, cherokee_table_t *properties)
{
	CHEROKEE_NEW_STRUCT (n, logger_ncsa);
	
	/* Init the base class object
	 */
	cherokee_logger_init_base(LOGGER(n));

	LOGGER(n)->init        = (logger_func_init_t)  cherokee_logger_ncsa_init;
	LOGGER(n)->free        = (logger_func_free_t)  cherokee_logger_ncsa_free;
	LOGGER(n)->write       = (logger_func_write_t) cherokee_logger_ncsa_write;

	*logger = LOGGER(n);
	
	/* Init
	 */
	
	n->accesslog_filename	= cherokee_table_get (properties, "AccessLog");
	n->errorlog_filename	= cherokee_table_get (properties, "ErrorLog");	
	
	if (n->accesslog_filename == NULL)
		PRINT_ERROR("logger_NCSA_common: AccessLog property not defined in config\n"); 		

        if (n->errorlog_filename == NULL)
		PRINT_ERROR("logger_NCSA_common: ErrorLog property not defined in config\n");

	if (!n->accesslog_filename || !n->errorlog_filename) 
		return ret_error;		

	return ret_ok;
}


ret_t 
cherokee_logger_ncsa_init (cherokee_logger_ncsa_t *logger)
{
	
	/* Try openning the access log file for append 
	 */
	logger->accesslog_fd = fopen(logger->accesslog_filename, "a+");
        if (logger->accesslog_fd == NULL) 
	{
		PRINT_ERROR("logger_NCSA_common: error opening %s for append\n",logger->accesslog_filename); 
        	return(ret_error);
	}


	/* Try openning the error log file for append 
	 */
        logger->errorlog_fd = fopen(logger->errorlog_filename, "a+");
	if (logger->errorlog_fd == NULL) 
	{
	        PRINT_ERROR("logger_NCSA_common: error opening %s for append\n",logger->errorlog_filename);
		return(ret_error);
	}


	return ret_ok;
}


ret_t
cherokee_logger_ncsa_free (cherokee_logger_ncsa_t *logger)
{
	
	if (fclose(logger->errorlog_fd)  != 0 ||
	    fclose(logger->accesslog_fd) != 0) {
		return ret_error;
	}

	free (logger);
	
	return ret_ok;
}


ret_t
cherokee_logger_ncsa_write (cherokee_logger_ncsa_t *logger, cherokee_connection_t *cnt)
{
	
  	long int z;
	int  len_wrote;
	struct tm conn_time;
   
        if (logger->accesslog_fd == NULL)
		SHOULDNT_HAPPEN;

	/* thread-safe version of localtime */
	localtime_r(&CONN_SRV(cnt)->bogo_now, &conn_time);
	
#ifdef HAVE_INT_TIMEZONE
	z = - (timezone / 60);
#else
#warning TODO
	z = 0;
#endif

	len_wrote = fprintf (logger->accesslog_fd, 
		  "%s - - [%02d/%s/%d:%02d:%02d:%02d %c%02d%02d] \"%s %s %s\" %d %ld\n",
		  inet_ntoa(IN_ADDR(cnt->addr_in)),
		  conn_time.tm_mday, 
		  month[conn_time.tm_mon], 
		  1900 + conn_time.tm_year,
		  conn_time.tm_hour, 
		  conn_time.tm_min, 
		  conn_time.tm_sec,
		  (z < 0) ? '-' : '+', 
		  (int) z/60, 
		  (int) z%60, 
		  method[cnt->method],
		  cnt->request->buf, 
		  version[cnt->version], 
		  cnt->error_code,
		  cnt->range_end - cnt->range_start);

	if (len_wrote > 0) {
                fflush(logger->accesslog_fd);
                return(ret_ok);
        }

        return(ret_error);

}






