%{
/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <pwd.h>
#include <grp.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>

#include "mime.h"
#include "common.h"
#include "server.h"
#include "virtual_server.h"
#include "plugin_table_entry.h"
#include "encoder.h"

/* Define the parameter name of the yyparse() argument
 */
#define YYPARSE_PARAM server

%}


%union {
	   int   number;
	   char *string;
	   void *ptr;

	   struct {
			 char *name;
			 void *ptr;
	   } name_ptr;
}


%{
/* What is the right way to import this prototipe?
 */
extern int yylex(void);


extern char *yytext;
extern int   yylineno;

cherokee_virtual_server_t     *current_virtual_server     = NULL;

char                          *current_directory_name     = NULL;
cherokee_plugin_table_entry_t *current_plugin_table_entry = NULL;


#define auto_virtual_server ((current_virtual_server) ? current_virtual_server : SRV(server)->vserver_default)


char *
make_finish_with_slash (char *string, int *len)
{
	   char *new;

	   if (string[(*len)-1] == '/') {
			 return string;
	   }

	   new = (char*) malloc ((*len)+2);
	   memcpy (new, string, *len);
	   new[(*len)] = '/';
	   new[(*len)+1] = '\0';
	   free (string);

	   (*len)++;

	   return new;
}


cherokee_plugin_table_entry_t *
plugin_table_entry_new (void)
{
	   cherokee_plugin_table_entry_t *entry;

	   cherokee_plugin_table_entry_new (&entry);
	   current_plugin_table_entry = entry;

	   return entry;
}


void
yyerror(char* msg)
{
        PRINT_ERROR ("Error parsing configuration: '%s', line %d, symbol '%s'\n", 
				 msg, yylineno, yytext);
}

%}



%token T_QUOTE
%token T_TIMEOUT T_KEEPALIVE T_DOCUMENT_ROOT T_LOG T_MIME_FILE T_DIRECTORY T_HANDLER T_USER T_GROUP 
%token T_SERVER T_USERDIR T_URL T_PIDFILE T_LISTEN T_FILEDIR T_HIDE_VERSION T_ENCODER

%token <number> T_NUMBER T_PORT 
%token <string> T_QSTRING T_FULLDIR T_ID T_HTTP_URL T_HOSTNAME T_IP 

%type <name_ptr> directory_option handler
%type <string> host_name
%type <ptr> encoder

%%

conffile : /* Empty */
         | lines
         ; 

lines : line
	 | lines line
	 ;

server_lines : server_line
             | server_lines server_line
             ;

line : server_line
     | common_line
     ;

common_line : server
            | port
            | listen
            | log
            | hideversion
            | mime
            | timeout
            | keepalive
            | pidfile
            | user1 
            | user2
            | group1 
            | group2
            | userdir
            ;

server_line : directory
            | document_root
            | encoder
            ;

handler_options : 
                | handler_options handler_option
                ;

directory_options : 
                | directory_options directory_option
                ;




port : T_PORT T_NUMBER 
{
	   SRV(server)->port = $2;
};

listen : T_LISTEN host_name
{
	   SRV(server)->listen_to = $2;
};

document_root : T_DOCUMENT_ROOT T_FULLDIR
{
	   cherokee_virtual_server_t *vserver;

	   vserver = auto_virtual_server;
	   vserver->root = $2;
	   vserver->root_len = strlen ($2);

	   /* Check for the endding slash
	    */	   
	   vserver->root = make_finish_with_slash (vserver->root, &(vserver->root_len));
};

log : T_LOG T_NUMBER
{
	   cherokee_server_log (SRV(server), $2);
};

hideversion : T_HIDE_VERSION T_NUMBER
{
	   SRV(server)->hideversion = $2;
};

mime : T_MIME_FILE T_FULLDIR
{
	   cherokee_mime_t *mime;

	   cherokee_mime_init_with_file (&mime, (char *)$2);
	   free ($2);
};

timeout : T_TIMEOUT T_NUMBER
{
	   SRV(server)->timeout = $2;
};

keepalive : T_KEEPALIVE T_NUMBER
{
	   SRV(server)->keepalive = $2;
};

encoder : T_ENCODER T_ID
{
	   encoder_func_new_t ret;
	   ret = cherokee_encoder_table_get (SRV(server)->encoders, $2);
	   free ($2);
	   $$ = ret;
};

encoder : T_ENCODER T_ID '{' encoder_options '}'
{
	   encoder_func_new_t ret;

	   ret = cherokee_encoder_table_get (SRV(server)->encoders, $2);
	   free ($2);

	   $$ = ret;
};

encoder_options :
{

};

pidfile : T_PIDFILE T_FULLDIR
{
	   FILE *file;

	   file = fopen ($2, "w");
	   if (file == NULL) {
			 PRINT_ERROR ("Can't write PID file '%s'\n", $2);
			 return 1;
	   }

	   snprintf (gbl_buffer, gbl_buffer_size, "%d\n", getpid());
	   fwrite (gbl_buffer, 1, strlen(gbl_buffer), file);
	   fclose (file);

	   free ($2);
};

user1 : T_USER T_ID
{
	   struct passwd *pwd;
	   
	   pwd = (struct passwd *) getpwnam ($2);
	   if (pwd == NULL) {
			 PRINT_ERROR ("User '%s' not found in the system", $2);
			 return 1;
	   }

	   SRV(server)->user = pwd->pw_uid;

	   free ($2);
};

user2 : T_USER T_NUMBER
{
	   SRV(server)->user = $2;
};

group1 : T_GROUP T_ID
{
	   struct group *grp;

	   grp = (struct group *) getgrnam ($2);
	   if (grp == NULL) {
			 PRINT_ERROR ("Group '%s' not found in the system", $2);
			 return 1;
	   }

	   SRV(server)->group = grp->gr_gid;

	   free ($2);
};

group2 : T_GROUP T_NUMBER
{
	   SRV(server)->group = $2;
};

handler : T_HANDLER T_ID '{' handler_options '}' {
	   $$.name = $2;
	   $$.ptr = current_plugin_table_entry;
};

handler : T_HANDLER T_ID 
{
	   $$.name = $2;
	   $$.ptr = current_plugin_table_entry;
};


handler_option : T_URL T_HTTP_URL
{
	   cherokee_plugin_table_entry_set (current_plugin_table_entry, "url", $2);
};

handler_option : T_FILEDIR T_FULLDIR
{
	   cherokee_plugin_table_entry_set (current_plugin_table_entry, "filedir", $2);
};


host_name : T_ID
          | T_IP 
{
	   $$ = $1;
};

server : T_SERVER host_name  '{' 
{
	   cherokee_virtual_server_t *vserver;

	   cherokee_virtual_server_new (&vserver);
	   vserver->domain = $2;

	   cherokee_table_add (SRV(server)->vservers, $2, vserver);

	   current_virtual_server = vserver;
} server_lines '}' {
	   current_virtual_server = NULL;
};

directory : T_DIRECTORY T_FULLDIR '{' 
{
	   char *name;
	   int   name_len;

	   /* Directory must finish with a slash
	    */
	   name_len = strlen($2);
	   name = make_finish_with_slash ($2, &name_len);
	   current_directory_name = name;

	   /* Create new table entry
	    * It sets the current_table_entry
	    */
	   plugin_table_entry_new ();
} 
directory_options '}';


directory_option : handler
{	   
	   cherokee_virtual_server_t *vserver = auto_virtual_server;

	   cherokee_plugin_table_fill_entry (vserver->plugins, $1.name, current_plugin_table_entry);

 	   cherokee_virtual_server_set (vserver, 
							  current_directory_name, 
							  $1.name, $1.ptr);
};


userdir : T_USERDIR T_ID '{' handler '}'
{
	   int len;
	   cherokee_plugin_table_entry_t *dir;

	   /* Have to finish with a slash
	    */
	   len = strlen($2);
	   SRV(server)->userdir = make_finish_with_slash ($2, &len);

	   dir = plugin_table_entry_new ();
	   cherokee_plugin_table_fill_entry (SRV(server)->vserver_default->plugins, $4.name, dir);
	   SRV(server)->userdir_handler = dir;

	   free ($4.name);
};


%%

