/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee Benchmarker
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"

#include <errno.h>
#include <sys/ioctl.h>

#ifdef HAVE_SYS_FILIO_H
# include <sys/filio.h>     /* defines FIONBIO and FIONREAD */
#endif
#ifdef HAVE_SYS_SOCKIO_H
# include <sys/sockio.h>    /* defines SIOCATMARK */
#endif

#include "connection.h"
#include "thread.h"
#include "bench.h"



ret_t 
cherokee_bench_connection_new (cherokee_bench_connection_t **conn)
{
	CHEROKEE_NEW_STRUCT(n,bench_connection);
	
	INIT_LIST_HEAD((list_t *)&n->list_entry);

	n->thread         = NULL;
	n->req            = NULL;
	n->content_length = -1;
	n->readed         = 0;
	n->state          = init;
	n->keepalive      = keepalive_enable;
	n->error_code     = http_unset;

	cherokee_bench_info_init (&n->info);
	memset (&n->address, 0, sizeof(n->address));

	cherokee_socket_new (&n->socket);

	cherokee_buffer_new (&n->req_string);
	cherokee_buffer_new (&n->buffer);
	cherokee_header_new (&n->header);

	*conn = n;
	return ret_ok;
}


ret_t 
cherokee_bench_connection_free  (cherokee_bench_connection_t *conn)
{
	cherokee_socket_free (conn->socket);

	cherokee_header_free (conn->header);
	cherokee_buffer_free (conn->buffer);
	cherokee_buffer_free (conn->req_string);
	
	free (conn);
	return ret_ok;
}


static ret_t 
connection_reset_closed (cherokee_bench_connection_t *conn)
{
	cherokee_bench_thread_t *thread = CONN_THREAD(conn);
	cherokee_bench_t        *bench  = THREAD_BENCH(thread);
	
	/* Close the socket
	 */
	cherokee_fdpoll_del (thread->fdpoll, SOCKET_FD(conn->socket));
	cherokee_socket_close (conn->socket);
	
	/* Store the error..
	 */
	thread->errors.aborted++;

	return ret_ok;
}


static ret_t 
connection_reset_keepalive (cherokee_bench_connection_t *conn)
{
	conn->info.keepalive++;

	/* Set socket to write in order to send headers
	 */
	cherokee_bench_connection_set_mode (conn, socket_writing);

	return ret_ok;
}


ret_t 
cherokee_bench_connection_reset (cherokee_bench_connection_t *conn)
{
	/* IMPORTANT:
	 * Don't touch conn->buffer, it could contain more response headers.
	 */

	/* Clean the header object and the info
	 */
	cherokee_header_clean (conn->header);
	cherokee_bench_info_clean (&conn->info);

	/* Clean properties
	 */
	conn->content_length = -1;
	conn->state          = init;
	conn->error_code     = http_unset;

	if (conn->keepalive == keepalive_enable) {
		return connection_reset_keepalive (conn);
	}
	
	return connection_reset_closed (conn);
}


static ret_t
build_header (cherokee_bench_connection_t *conn)
{
	cherokee_bench_request_t *req = CONN_REQ(conn);
	cherokee_buffer_t        *buf = conn->req_string;

	cherokee_buffer_make_empty (buf);
	cherokee_bench_request_build_string (req, buf);
	
	if (REQUEST_PIPELINE(req) > 1) {
		cherokee_buffer_multiply (buf, REQUEST_PIPELINE(req));
	}

	return ret_ok;
}


static ret_t
process_header (cherokee_bench_connection_t *conn)
{
	ret_t                    ret;
	char                    *end;
	cherokee_buffer_t       *tmp;
	cherokee_bench_thread_t *thread = CONN_THREAD(conn);
	cherokee_bench_t        *bench  = THREAD_BENCH(thread);
	

	/* Has the buffer a complete header?
	 */
	end = strstr (conn->buffer->buf, CRLF CRLF);
	if (end == NULL) {
		return ret_eagain;
	}
	
	/* Parse the buffer
	 */
	ret = cherokee_header_parse (conn->header, conn->buffer, header_type_response);
	if (ret < ret_ok) {
		PRINT_ERROR ("Error parsing headers: '%s'\n", conn->buffer->buf);
		return ret;
	}

	/* Get the response code
	 */
	conn->error_code = HDR_RESPONSE(conn->header);

	if (http_type_200 (conn->error_code)) {
		conn->info.error[1].code[200-conn->error_code]++;

	} else if (http_type_300 (conn->error_code)) {
		conn->info.error[2].code[300-conn->error_code]++;

	} else if (http_type_400 (conn->error_code)) {
		conn->info.error[3].code[400-conn->error_code]++;

	} else {
		SHOULDNT_HAPPEN;
	}

	
	/* If it is not a "ok" responde.. disable the keepalive
	 */
	if (http_type_400(conn->error_code)) {
		conn->keepalive = keepalive_disable;
		return ret_ok;
	}


	/* We have to get the useful header parameters here.
	 * After this function, the header will be destroyed.
	 */

	cherokee_buffer_new (&tmp);  /* 1 */

	/* Read the "Connection:" header
	 */
	ret = cherokee_header_copy_known (conn->header, header_connection, tmp);
	if (ret < ret_ok) {
		conn->keepalive = keepalive_disable;
	}

	/* Set Keep-alive
	 */
	if (strncasecmp (tmp->buf, "keep-alive", 10) == 0) {
		conn->keepalive = keepalive_enable;
	} else {
		conn->keepalive = keepalive_disable;
	}

	if (BENCH_KEEPALIVE(bench) == keepalive_disable) {
		conn->keepalive = keepalive_disable;
	}

	/* Read the Content-Length header
	 */
	cherokee_buffer_make_empty (tmp);
	ret = cherokee_header_copy_known (conn->header, header_content_length, tmp);
	if (ret != ret_ok) {
		PRINT_ERROR ("Can not read the content-length header\n");
		conn->keepalive = keepalive_disable;
		return ret;
	}
	conn->content_length = atoi (tmp->buf);

	cherokee_buffer_free (tmp);  /* 2 */


	/* Drop out the header
	 */
	cherokee_buffer_move_to_begin (conn->buffer, (end - conn->buffer->buf)+4);
	
	return ret_ok;
}


ret_t 
cherokee_bench_connection_set_mode (cherokee_bench_connection_t *conn, cherokee_socket_status_t status)
{
	cherokee_bench_thread_t *thread = CONN_THREAD(conn);

	cherokee_socket_set_status (conn->socket, status);	
	cherokee_fdpoll_set_mode (THREAD_FDPOLL(thread), SOCKET_FD(conn->socket), status);

	return ret_ok;
}


/* ret_ok    - ok, more work to do
 * ret_error - fatal error
 */

ret_t 
cherokee_bench_connection_step (cherokee_bench_connection_t *conn)
{
	cherokee_bench_thread_t *thread = CONN_THREAD(conn);
	cherokee_bench_t        *bench  = THREAD_BENCH(thread);
	ret_t    ret;
	ssize_t  len;
	char    *tmp;

	switch (conn->state) {
	case init:
		ret = build_header (conn);
		if (ret < ret_ok) {
			PRINT_ERROR ("Can not build the header\n");
			return ret;
		}

		cherokee_bench_connection_set_mode (conn, 1);
		conn->state = writing;

	case writing:
		ret = cherokee_socket_write (conn->socket, conn->req_string, &len);
		switch (ret) {
		case ret_ok:
			conn->info.tx += len;
			break;
		case ret_eof:
			thread->errors.aborted++;
			conn->keepalive = keepalive_disable;
			cherokee_bench_connection_reset (conn);
			return ret_ok;
		case ret_error:
			return ret_error;
		case ret_eagain:
			return ret_eagain;
		default:
			SHOULDNT_HAPPEN;
		}
		
		/* 1.- Partial write
		 */
		if (len < conn->req_string->len) {
			cherokee_buffer_move_to_begin (conn->req_string, len);
			return ret_ok;
		}

		/* 2.- Completed write
		 */
		cherokee_buffer_make_empty (conn->req_string);
		cherokee_bench_connection_set_mode (conn, 0);
		conn->state = reading_header;
		return ret_ok;

	case reading_header:
		ret = cherokee_socket_read (conn->socket, conn->buffer, CONN_READ_SIZE, &len);
		switch (ret) {
		case ret_eagain:
			return ret_eagain;

		case ret_ok:
			/* Count the header bytes
			 */
			conn->info.rx += len;
 
			/* Maybe parse the header
			 */
			ret = process_header (conn);
			switch (ret) {
			case ret_eagain:  
				return ret_eagain;
			case ret_error:   
				return ret_error;
			case ret_ok:
				conn->state = reading_body;
				break;
			default:
				SHOULDNT_HAPPEN;
			}
			break;
		case ret_eof:
			thread->errors.aborted++;
			conn->keepalive = keepalive_disable;
			cherokee_bench_connection_reset (conn);
			return ret_ok;
		default:
			return ret;
		}

	case reading_body:
		/* It has read the full content, and some from the next header
		 */
		if (conn->buffer->len >= conn->content_length) {
			cherokee_buffer_move_to_begin (conn->buffer, conn->content_length);			
			
			/* Collect the information about this connection
			 */
			conn->info.transfers++;
			cherokee_bench_info_add (&thread->info, &conn->info);

			/* Reset the connection
			 */
			cherokee_bench_connection_reset (conn);
			return ret_ok;
		}
		
		ret = cherokee_socket_read (conn->socket, conn->buffer, CONN_READ_SIZE, &len);
		if (ret < ret_ok) return ret;

		conn->info.rx += len;
		break;

	default:
		SHOULDNT_HAPPEN;
	}
	return ret_ok;
}


ret_t 
cherokee_bench_connection_connect (cherokee_bench_connection_t *conn)
{
	int   r;
	ret_t ret;
	cherokee_bench_thread_t  *thread = CONN_THREAD(conn);
	cherokee_bench_t         *bench  = THREAD_BENCH(thread);
	cherokee_bench_request_t *req    = CONN_REQ(conn);
	cherokee_url_t           *url    = REQUEST_URL(req);

	/* Fill the address
	 */
	SOCKET_FD(conn->socket) = socket (AF_INET, SOCK_STREAM, 0);
	if (SOCKET_FD(conn->socket) < 0) {
		return ret_error;
	}

	conn->address.sin_family = AF_INET;
	conn->address.sin_port   = htons(URL_PORT(url));

	inet_pton (AF_INET, URL_HOST(url)->buf, &conn->address.sin_addr);

#if 0
	printf ("conecting to '%s' port %d\n", URL_HOST(url)->buf, URL_PORT(url));
#endif

	/* Connect to server
	 */
	r = connect (SOCKET_FD(conn->socket), (struct sockaddr *)&conn->address, sizeof(conn->address));
	if (r < 0) {
		return ret_error;
	}

	/* Enables nonblocking I/O.
	 */
	r = 1;
	ioctl (SOCKET_FD(conn->socket), FIONBIO, &r);

	/* Add the socket to the file descriptors poll
	 */
	ret = cherokee_fdpoll_add (thread->fdpoll, SOCKET_FD(conn->socket), 1);
	if (ret > ret_ok) {
		PRINT_ERROR ("Can not add file descriptor (%d) to fdpoll\n", r);
		return ret;
	}

	/* Set in writing mode
	 */
	cherokee_bench_connection_set_mode (conn, socket_writing);	

	return ret_ok;
}


