/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * This piece of code by:
 *      Miguel Angel Ajo Pelayo <ajo@godsmaze.org>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "logger.h"


ret_t
cherokee_logger_init_base (cherokee_logger_t *logger)
{
	/* Init the base class
	 */
	cherokee_module_init_base (MODULE(logger));

	/* Pure virtual methods
	 */
	logger->write_access = NULL;
	logger->write_error  = NULL;

	cherokee_buffer_new (&logger->buffer);

	CHEROKEE_MUTEX_INIT (&logger->mutex, NULL);

	return ret_ok;
}




/* Virtual method hidding layer
 */
ret_t
cherokee_logger_free (cherokee_logger_t *logger)
{
	ret_t ret;
	
	if (logger->buffer) {
		cherokee_buffer_free (logger->buffer);
		logger->buffer = NULL;
	}

	CHEROKEE_MUTEX_DESTROY (&logger->mutex);

	if (MODULE(logger)->free == NULL) {
		return ret_error;
	}
	return MODULE(logger)->free (logger);
}


ret_t
cherokee_logger_flush (cherokee_logger_t *logger)
{
	ret_t ret = ret_error;

	if (logger->flush) {
		CHEROKEE_MUTEX_LOCK (&logger->mutex);
		ret = logger->flush(logger);
		CHEROKEE_MUTEX_UNLOCK (&logger->mutex);
	}

	return ret;
}


ret_t
cherokee_logger_init (cherokee_logger_t *logger)
{
	logger_func_init_t init_func;

	init_func = (logger_func_init_t) MODULE(logger)->init;

	if (init_func) {
		return init_func (logger);
	}
	
	return ret_error;
}


inline ret_t
cherokee_logger_write_access (cherokee_logger_t *logger, void *conn)
{
	ret_t ret = ret_error;

	if (logger->write_access) {
		CHEROKEE_MUTEX_LOCK (&logger->mutex);
		ret = logger->write_access (logger, conn);
		CHEROKEE_MUTEX_UNLOCK (&logger->mutex);
	}
	
	return ret;
}


ret_t 
cherokee_logger_write_error (cherokee_logger_t *logger, void *conn)
{
	ret_t ret = ret_error;

	if (logger->write_error) {
		CHEROKEE_MUTEX_LOCK (&logger->mutex);
		ret = logger->write_error (logger, conn);
		CHEROKEE_MUTEX_UNLOCK (&logger->mutex);
	}

	return ret;
}


ret_t 
cherokee_logger_write_string (cherokee_logger_t *logger, const char *format, ...)
{
	va_list ap;

	if (logger == NULL) 
		return ret_ok;

	if (logger->write_string) {
		CHEROKEE_TEMP (tmp, 200);

		CHEROKEE_MUTEX_LOCK(&logger->mutex);
		va_start (ap, format);
		vsnprintf (tmp, tmp_size, format, ap);
		va_end (ap);
		CHEROKEE_MUTEX_UNLOCK(&logger->mutex);

		return logger->write_string (logger, tmp);
	}

	return ret_error;
}
