/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Ayose Cazorla Len <setepo@gulic.org>
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <sys/poll.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <stdarg.h>

#include "module.h"
#include "handler_cgi.h"
#include "connection.h"
#include "server.h"

/* hmm.. it will be wonderful if cherokee had a
 * cherokee_connection_get_mehod_name (or similar), and use it 
 * instead of this vector
static char *method[]  = {"GET", "POST", "HEAD", NULL};
 */


cherokee_module_info_t cherokee_cgi_info = {
	cherokee_handler,           /* type     */
	cherokee_handler_cgi_new    /* new func */
};


#define SILENT

void _LOG(const char* msg, ...)
{
#ifndef SILENT
	va_list l; 

	va_start(l, msg);
	vfprintf(stderr, msg, l);
	va_end(l);
#endif
}


ret_t
cherokee_handler_cgi_new  (cherokee_handler_t **hdl, void *cnt, cherokee_table_t *props)
{
	CHEROKEE_NEW_STRUCT (n, handler_cgi);
	
	/* Init the base class object
	 */
	cherokee_handler_init_base(HANDLER(n), cnt);

	MODULE(n)->init         = (handler_func_init_t) cherokee_handler_cgi_init;
	MODULE(n)->free         = (handler_func_free_t) cherokee_handler_cgi_free;
	HANDLER(n)->step        = (handler_func_step_t) cherokee_handler_cgi_step;
	HANDLER(n)->add_headers = (handler_func_add_headers_t) cherokee_handler_cgi_add_headers;

	HANDLER(n)->connection  = cnt;
	HANDLER(n)->support     = hsupport_complex_headers;

	/* Process the request_string, and build the arguments table..
	 * We'll need it later
	 */
	cherokee_connection_parse_args (cnt);
	
	/* Init
	 */
	n->pipeInput      = 0;
	n->pipeOutput     = 0;
	n->post_data_sent = 0;
	n->pid            = -1;
	n->pathInfo       = NULL;
	n->fileName       = NULL;
	n->data           = NULL;
	n->parameter      = NULL;

	/* Return the object
	 */
	*hdl = HANDLER(n);

	return ret_ok;
}

static ret_t
_set_envs_cgi (cherokee_connection_t *conn, cherokee_handler_cgi_t* n)
{
	int   r;
	char *p = NULL;
	int   p_len;
	cherokee_buffer_t *tmp;
	char remote_ip[INET6_ADDRSTRLEN+1];
	ret_t ret;
	CHEROKEE_TEMP(temp, 32);

	/* Set the enviroment variables
	 */
	r  = setenv("SERVER_SIGNATURE",  "<address>Cherokee web server</address>", 1);
	r += setenv("SERVER_SOFTWARE",   "Cherokee " VERSION, 1);
	r += setenv("GATEWAY_INTERFACE", "CGI/1.1", 1);
	r += setenv("DOCUMENT_ROOT",     conn->local_directory->buf, 1);

	memset (remote_ip, 0, sizeof(remote_ip));
	cherokee_socket_ntop (conn->socket, remote_ip, sizeof(remote_ip)-1);
	r += setenv ("REMOTE_ADDR", remote_ip, 1);

	/* HTTP_HOST and SERVER_NAME. The difference between them is that
	 * HTTP_HOST can include the :PORT text, and SERVER_NAME only
	 * the name 
	 */
	cherokee_buffer_new (&tmp);
	cherokee_header_copy_known (conn->header, header_host, tmp);
	r += setenv ("HTTP_HOST", tmp->buf, 1);

	if ((p = index(tmp->buf, ':')) != NULL) *p = '\0';
	r += setenv ("SERVER_NAME", tmp->buf, 1);


	/* Cookies :-)
	 */
	cherokee_buffer_clean(tmp);
	ret = cherokee_header_copy_unknown (conn->header, "Cookie", 6, tmp);
	if (ret == ret_ok)
		r += setenv ("HTTP_COOKIE", tmp->buf, 1);

	/* User Agent
	 */
	cherokee_buffer_clean (tmp);
	ret = cherokee_header_copy_known (conn->header, header_user_agent, tmp);
	if (ret == ret_ok) {
		r += setenv ("HTTP_USER_AGENT", tmp->buf, 1);
	}

	/* Content-type and Content-lenght (if available) 
	 */
	cherokee_buffer_clean(tmp);
	ret = cherokee_header_copy_unknown (conn->header, "Content-Type", 12, tmp);
	if (ret == ret_ok)
		r += setenv ("CONTENT_TYPE", tmp->buf, 1);

	cherokee_buffer_clean(tmp); 
	ret = cherokee_header_copy_known (conn->header, header_content_length, tmp);
	if (ret == ret_ok)
		r += setenv ("CONTENT_LENGTH", tmp->buf, 1);

	cherokee_buffer_free(tmp);

	/* is there any query_string. It seems that, if it is empty,
	 * doens't have the \0 in the correct place (at start)
	 */
	if(conn->query_string->len > 0)
		r += setenv ("QUERY_STRING", conn->query_string->buf, 1);
	else
		r += setenv ("QUERY_STRING", "", 1);

	snprintf (temp, temp_size, "%d", CONN_SRV(conn)->port);
	r += setenv ("SERVER_PORT", temp, 1);

	/* Set the HTTP version
	 */
	ret = cherokee_http_version_to_string (conn->header->version, &p, &p_len);
	if (ret >= ret_ok) {
		r += setenv ("SERVER_PROTOCOL", p, 1);
	}

	/* Set the method
	 */
	ret = cherokee_http_method_to_string (conn->header->method, &p, &p_len);
	if (ret >= ret_ok) {
		r += setenv ("REQUEST_METHOD", p, 1);
	}
	
	/* Set the host name
	 */
	if (!cherokee_buffer_is_empty (conn->host)) {
		p = index (conn->host->buf, ':');
		if (p != NULL) *p = '\0';

		r = setenv ("SERVER_NAME", conn->host->buf, 1);

		if (p != NULL) *p = ':';
		if (r != 0) return ret_error;
	}

	/* Sets REQUEST_URI 
	 */
	cherokee_buffer_new(&tmp);
	cherokee_header_copy_request (conn->header, tmp);
	setenv("REQUEST_URI", tmp->buf, 1);

	/* SCRIPT_NAME is the same that REQUEST_URI, but without the PATH_INFO 
	 */
	cherokee_buffer_drop_endding(tmp, n->pathInfo ? n->pathInfo->len : 0);
	setenv("SCRIPT_NAME", tmp->buf, 1);
	cherokee_buffer_free (tmp);

	/* Fake path */
	setenv ("PATH", "/bin:/usr/bin:/sbin:/usr/sbin", 1);

	if (n->pathInfo) {
		setenv("PATH_INFO", n->pathInfo->buf, 1);
	}

	if (n->fileName) {
		setenv("SCRIPT_FILE_NAME", n->fileName->buf, 1);
	}

	/*
	 * TODO Fill the others CGI environment variables
	 *
	 * http://hoohoo.ncsa.uiuc.edu/cgi/env.html
	 * http://cgi-spec.golux.com/cgi-120-00a.html
	 */
	return ret_ok;
}

ret_t
_extract_path (cherokee_handler_cgi_t *cgi)
{
	int dr_len;
	char *cur;
	cherokee_connection_t *conn;

	conn = CONN(HANDLER(cgi)->connection);

	/* Append URI to DocumentRoot 
	 */
	dr_len = conn->local_directory->len;
	cherokee_buffer_add_buffer (conn->local_directory, conn->request); 

	/* Search the executable file 
	 */
	for (cur = conn->local_directory->buf + dr_len + 1; *cur; ++cur)
	{
		if (*cur == '/') {
			struct stat st;

			*cur = 0;
			if (stat (conn->local_directory->buf, &st) == -1)
			{
				*cur = '/';
				_LOG("CGI: Not found %s\n", conn->local_directory->buf);
				conn->error_code = http_not_found;
				return ret_error;
			}

			if (S_ISDIR(st.st_mode))
				*cur = '/';
			else
			{
				/* This can be the CGI to run 
				 */
				cherokee_buffer_new (&cgi->pathInfo);
				cherokee_buffer_new (&cgi->fileName);

				cherokee_buffer_add_buffer (cgi->fileName, 
							    conn->local_directory);

				/* Path info 
				 */
				*cur = '/';
				cherokee_buffer_add (cgi->pathInfo, 
						     cur, strlen(cur));

				break;
			}
		}
	}

	/* Is the filename set? 
	 */
	if (cgi->fileName == NULL) {
		cherokee_buffer_new (&cgi->fileName);
		cherokee_buffer_add_buffer (cgi->fileName, conn->local_directory);
	}

	/* Restore local_directory 
	 */
	conn->local_directory->len = dr_len;
	conn->local_directory->buf[dr_len] = '\0';

	return ret_ok;
}


ret_t
_send_post_data(cherokee_handler_cgi_t *cgi)
{
	cherokee_connection_t *conn;
	conn = HANDLER_CONN(cgi);

	if ((cgi->post_data_sent < conn->post_len) && (conn->post != NULL))
	{
		int r;
		r = write(cgi->pipeOutput,
			conn->post->buf + cgi->post_data_sent,
			conn->post_len - cgi->post_data_sent);

		if (r == -1) {
			if(errno != EAGAIN) {
				_LOG("Can't write to the client\n");
				return ret_error;
			}
		} else {
			_LOG("Write %d bytes of POST\n", r);
			cgi->post_data_sent += r;
		}

		/* Are all the data sent? */
		if (cgi->post_data_sent >= conn->post_len) {
			close(cgi->pipeOutput);
			cgi->pipeOutput = -1;
		}

	} else {

		/* Close the pipe, since we don't need it anymore */
		close(cgi->pipeOutput);
		cgi->pipeOutput = -1;
	}

	return ret_ok;
}


ret_t 
cherokee_handler_cgi_init (cherokee_handler_cgi_t *cgi)
{
	ret_t ret;
	int   re;
	int   pid;
	int   pipesServer[2], pipesCGI[2];
	cherokee_connection_t *conn = HANDLER_CONN(cgi);


	/* Extracts PATH_INFO and filename from request uri 
	 */
	ret = _extract_path(cgi);
	if (ret < ret_ok) { 
		return ret;
	}

	/* Creates the pipes ...
	 */
	re   = pipe (pipesCGI);
	ret |= pipe (pipesServer);

	if (re != 0) {
		conn->error_code = http_internal_error;
		return ret_ok;
	}

	/* .. and fork the process 
	 */
	pid = fork();
	if (pid == 0)
	{
		int re;
		char *absolute_path, *file;

		/* Sets the new environ. 
		 */			
		_set_envs_cgi (conn, cgi);

		/* Change stdin and out 
		 */
		close (pipesCGI[0]);
		close (pipesServer[1]);
		
		dup2 (pipesServer[0], 0);  /* stdin  */
		dup2 (pipesCGI[1], 1);     /* stdout */

		_LOG("CGI: Executing %s %s\n", cgi->fileName->buf,
		     (cgi->parameter)? cgi->parameter->buf : "");

		/* Change the current directory to to the program's own directory.
		 */
		absolute_path = cgi->fileName->buf;
		file = strrchr (absolute_path, '/');
		*file = '\0';
		chdir (absolute_path);
		*file = '/';
		file++;

		/* Execute the new CGI 
		 */
		if (cgi->parameter == NULL) {
			re = execl (absolute_path, file, NULL);
		} else {
			re = execl (absolute_path, file, cgi->parameter->buf, NULL);	
		}

		if (re < 0) {
			PRINT_ERROR ("CGI error: execl(%s, %s): %s\n", absolute_path, file, strerror(errno));
			exit(1);
		}

		/* OH MY GOD!!! an error is here 
		 */
		SHOULDNT_HAPPEN;
		exit(1);
	} 
	else if (pid < 0) {
		conn->error_code = http_internal_error;
		return ret_ok;
	}

	_LOG("CGI: pid %d\n", pid);

	close (pipesServer[0]);
	close (pipesCGI[1]);

	cgi->pid        = pid;
	cgi->pipeInput  = pipesCGI[0];
	cgi->pipeOutput = pipesServer[1];

	if (conn->post != NULL) {
		/* If we have a POST data to send, sets out output non-blocking 
		 */
		fcntl (cgi->pipeOutput, F_SETFL, O_NONBLOCK);

		/* Start to send data 
		 */
		_send_post_data(cgi);
	} else {
		/* We don't need to send anything to the client,
		 * so we close the pipe
		 */
		close(cgi->pipeOutput);
		cgi->pipeOutput = -1;
	}

	/* TODO Yeah, I know, this is not a perfect solution, but it is very
	 * dificult do this in cherokee
	 */
	cherokee_buffer_new (&cgi->data);
	cherokee_buffer_ensure_size (cgi->data, 2 * 1024); /* 2 kb for headers */

	return ret_ok;
}


static int
do_reap (void)
{
	pid_t pid;
	int   child_count;
	int   status;

	/* Reap defunct children until there aren't any more. 
	 */
	for (child_count = 0; ; ++child_count)
        {
		pid = waitpid (-1, &status, WNOHANG);

                /* none left */
		if (pid == 0) break;

		else if (pid < 0) {
			/* because of ptrace */
			if (errno == EINTR) continue;
			break;
		}
        }

	return child_count;
}


ret_t
cherokee_handler_cgi_free (cherokee_handler_cgi_t *cgi)
{
	pid_t pid_status;
	int   status = 0;

	_LOG("CGI: closing\n\n");

	/* Close the connection with the CGI
	 */
	close(cgi->pipeInput);
	close(cgi->pipeOutput);

        /* Maybe kill the CGI
	 */
#if 0
	if (cgi->pid > 0) {
		pid_status = waitpid (cgi->pid, &status, WNOHANG);
		if (pid_status <= 0) {
			kill (cgi->pid, SIGTERM);
		}
	}
#endif

	/* Free the rest of the handler CGI memory
	 */
	if(cgi->data) {
		cherokee_buffer_free (cgi->data);
	}

	if(cgi->pathInfo) {
		cherokee_buffer_free (cgi->pathInfo);
	}

	if(cgi->fileName) {
		cherokee_buffer_free (cgi->fileName);
	}

	if (cgi->parameter) {
		cherokee_buffer_free (cgi->parameter);
	}

	free (cgi);

	/* for some reason, we have seen that the SIGCHLD signal does not call to
	 * our handler in a server with a lot of requests, so the wait() call,
	 * necessary to free the resources used by the CGI, is not called. So I
	 * think that a possible solution couble be to put the waitpid call in the
	 * _free method of this handler, so when the handler ends, this will free
	 * the resources used by our cool CGI.
	 */
	do_reap();

	return ret_ok;
}


ret_t
_read_from_cgi (cherokee_handler_cgi_t *cgi, cherokee_buffer_t *buffer)
{
	int len, ret_poll;
	struct pollfd polling = { cgi->pipeInput, POLLIN|POLLPRI, 0 };
	CHEROKEE_TEMP(tmp,1024);

	_LOG("CGI: Read from CGI... ");

	ret_poll = poll (&polling, 1, 10);

	if (ret_poll == 0)
	{
		/* Timeout. There is no data to read:
		 */
		_LOG("nothing to read. Timeout\n");
		return ret_eagain;
	}

	if ((polling.revents & (POLLIN|POLLPRI)) == 0)	{
		/* An error. Perhaps, the CGI close its pipe 
		 */
		_LOG("Error in poll. revents = %i\n", polling.revents);
		return ret_error;
	}

	if (ret_poll == -1) {
		/* oops
		 */
		_LOG("returned error from poll: %s\n", strerror(errno));
		return ret_error;
	}


	/* Read the data from the pipe:
	 */
	len = read (cgi->pipeInput, tmp, tmp_size);
	if (len > 0) {
		cherokee_buffer_add (buffer, tmp, len);

	} else if (len == 0) {
		_LOG("no bytes read\n");
		return ret_eof;

	} else {
		_LOG("returned error form read: %s\n", strerror(errno));
		return ret_error;
	}

	_LOG(" %d bytes read.\n", len);

	return ret_ok;
}


ret_t
cherokee_handler_cgi_step (cherokee_handler_cgi_t *cgi, cherokee_buffer_t *buffer)
{
	cherokee_connection_t *conn;
	conn = HANDLER_CONN(cgi);

	/* Maybe it has some stored data to be send
	 */
	if (cgi->data != NULL) {
		ret_t ret;
		
		_LOG("CGI: sending stored data: %d bytes\n", cgi->data->len);
		
		/* Flush this buffer 
		 */
		if (!cherokee_buffer_is_empty (cgi->data)) {
			cherokee_buffer_add_buffer (buffer, cgi->data);
			ret = ret_ok;
		} else
			ret = ret_eagain;
		
		cherokee_buffer_free (cgi->data);
		cgi->data = NULL;

		return ret;
	}

	/* Sent POST data 
	 */
	if (cgi->pipeOutput != -1) {
		_send_post_data (cgi);
	}
	
	return _read_from_cgi (cgi, buffer);
}


ret_t
cherokee_handler_cgi_add_headers (cherokee_handler_cgi_t *cgi, cherokee_buffer_t *buffer)
{
	ret_t  ret;
	char  *content;
	int    end_len;

	ret = _read_from_cgi (cgi, cgi->data);
	if (ret != ret_ok) return ret;

	content = strstr(cgi->data->buf, "\n\n");
	if (content != NULL) {
		end_len = 2;
	} else {
		content = strstr(cgi->data->buf, CRLF CRLF);
		end_len = 4;
	}

	if (content != NULL) {
		int len = content - cgi->data->buf;
		
		cherokee_buffer_add (buffer, cgi->data->buf, len);
		cherokee_buffer_add (buffer, CRLF CRLF, 4);
		cherokee_buffer_move_to_begin (cgi->data, len + end_len);

		return ret_ok;
	}

	return ret_eagain;
}





/*   Library init function
 */

static int _cgi_is_init = 0;

#if 0
#include <signal.h>

static void 
child_finished(int sng)
{
	int status;
	while(waitpid (0, &status, WNOHANG) > 0);
}
#endif

void
cgi_init ()
{
	if (_cgi_is_init) {
		return;
	}

	_cgi_is_init = 1;
#if 0
	signal(SIGCHLD, child_finished);
#endif
}
