/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef __CHEROKEE_COMMON_H__
#define __CHEROKEE_COMMON_H__

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_PTHREAD
#include <pthread.h>
#endif

#ifdef HAVE_DLFCN_H
# include <dlfcn.h>
#endif

#if defined (HAVE_ENDIAN_H)
# include <endian.h>
#elif defined (HAVE_MACHINE_ENDIAN_H)
# include <machine/endian.h>
#elif defined (HAVE_SYS_ENDIAN_H)
# include <sys/endian.h>
#endif

#ifdef HAVE_INTTYPES_H
# include <inttypes.h>
#elif HAVE_STDINT_H
# include <stdint.h>
#else
# error "Can not include inttypes or stdint"
#endif

#ifdef _WIN32
# include "libc_interface.h"
#endif


#define DEFAULT_RECV_SIZE             1024
#define DEFAULT_READ_SIZE             8192
#define MAX_HEADER_LEN                4096
#define MAX_KEEPALIVE                 500
#define MAX_NEW_CONNECTIONS_PER_STEP  50
#define MAX_MMAP_TABLE_ITEMS          15
#define MAX_MMAP_FILE_SIZE            50000

#define EXIT_CANT_CREATE_SERVER_SOCKET4 10
#define EXIT_EPOLL_CREATE_ERROR         20
#define EXIT_EPOLL_CTL_ADD              21
#define EXIT_EPOLL_CTL_MOD              22
#define EXIT_EPOLL_CTL_DEL              23
#define EXIT_EPOLL_CREATE               24
#define EXIT_SERVER_CLEAN               30
#define EXIT_SERVER_READ_CONFIG         31
#define EXIT_SERVER_INIT                32


#define CRLF "\r\n"



typedef enum {
	false = 0,
	true  = 1
} cherokee_boolean_t;

typedef enum {
	ret_nomem          = -3, 
	ret_deny           = -2, 
	ret_error          = -1,
	ret_ok             =  0,
	ret_eof            =  1,
	ret_eof_have_data  =  2,
	ret_not_found      =  3,
	ret_file_not_found =  4,
	ret_eagain         =  5, 
	ret_ok_and_sent    =  6
} ret_t;
	
#define return_if_fail(expr,ret) \
	if (!(expr)) {                                                    \
		fprintf (stderr,                                          \
       		         "file %s: line %d (%s): assertion `%s' failed\n",\
                          __FILE__,                                       \
                          __LINE__,                                       \
                          __PRETTY_FUNCTION__,                            \
                          #expr);                                         \
	        return (ret);                                             \
	}


#define SHOULDNT_HAPPEN \
	do { fprintf (stderr, "file %s: line %d (%s): this shouldn't happend\n", \
			            __FILE__, __LINE__, __PRETTY_FUNCTION__);          \
	} while (0)

#define CHEROKEE_NEW_STRUCT(obj,type)                                                                         \
	cherokee_ ## type ## _t * obj = (cherokee_ ## type ## _t *) malloc (sizeof(cherokee_ ## type ## _t)); \
	return_if_fail (obj != NULL, ret_nomem)

#define CHEROKEE_NEW(obj,type)                   \
	cherokee_ ## type ## _t * obj;           \
	cherokee_ ## type ## _new (& obj );      \
	return_if_fail (obj != NULL, ret_nomem)

#define CHEROKEE_NEW2(obj1,obj2,type)             \
	cherokee_ ## type ## _t * obj1, *obj2;    \
	cherokee_ ## type ## _new (& obj1 );      \
	cherokee_ ## type ## _new (& obj2 );      \
	return_if_fail (obj1 != NULL, ret_nomem); \
	return_if_fail (obj2 != NULL, ret_nomem)

#define CHEROKEE_TEMP(obj, size)                 \
        const unsigned int obj ## _size = size;  \
	char obj[size]

#ifndef MIN
# define MIN(x,y) ((x<y) ? x : y)
#endif

#ifndef MAX
# define MAX(x,y) ((x>y) ? x : y)
#endif


#ifdef HAVE_PTHREAD
# define CHEROKEE_MUTEX_LOCK(m)     pthread_mutex_lock(m)
# define CHEROKEE_MUTEX_UNLOCK(m)   pthread_mutex_unlock(m)
# define CHEROKEE_MUTEX_INIT(m,n)   pthread_mutex_init(m,n)
# define CHEROKEE_MUTEX_DESTROY(m)  pthread_mutex_destroy(m)
# define CHEROKEE_MUTEX_TRY_LOCK(m) pthread_mutex_trylock(m)

# define CHEROKEE_RWLOCK_INIT(m,n)  pthread_rwlock_init(m,n)
# define CHEROKEE_RWLOCK_WRITER(m)  pthread_rwlock_wrlock(m)
# define CHEROKEE_RWLOCK_READER(m)  pthread_rwlock_rdlock(m)
# define CHEROKEE_RWLOCK_UNLOCK(m)  pthread_rwlock_unlock(m)
# define CHEROKEE_RWLOCK_DESTROY(m) pthread_rwlock_destroy(m)
#else
# define CHEROKEE_MUTEX_LOCK(m)
# define CHEROKEE_MUTEX_UNLOCK(m)
# define CHEROKEE_MUTEX_INIT(m,n)  
# define CHEROKEE_MUTEX_DESTROY(m) 
# define CHEROKEE_MUTEX_TRY_LOCK(m) 0

# define CHEROKEE_RWLOCK_INIT(m,n)
# define CHEROKEE_RWLOCK_WRITER(m)
# define CHEROKEE_RWLOCK_READER(m)
# define CHEROKEE_RWLOCK_UNLOCK(m)
# define CHEROKEE_RWLOCK_DESTROY(m)
#endif


void PRINT_ERROR (const char *format, ...);
void PRINT_DEBUG (const char *format, ...);


/* cross compilers should define BYTE_ORDER in CPPFLAGS 
 */
#ifndef BYTE_ORDER

/* Definitions for byte order, according to byte significance from low
 * address to high.
 */
#define LITTLE_ENDIAN  1234    /* LSB first: i386, vax */
#define    BIG_ENDIAN  4321    /* MSB first: 68000, ibm, net */
#define    PDP_ENDIAN  3412    /* LSB first in word, MSW first in long */

/* assume autoconf's AC_C_BIGENDIAN has been ran. If it hasn't, we 
 * assume (maybe falsely) the order is LITTLE ENDIAN
 */
#ifdef WORDS_BIGENDIAN
# define BYTE_ORDER  BIG_ENDIAN
#else
# define BYTE_ORDER  LITTLE_ENDIAN
#endif

#endif /* BYTE_ORDER */



/* Misc functions
 */
int   hexit        (char c);
char *apr_strfsize (size_t size, char *buf);
char *min_string_not_null (char *s1, char *s2);

ret_t get_system_fd_num_limit (int *limit);

#ifndef HAVE_STRSEP
char *strsep (char** str, const char* delims);
#endif

#endif /* __CHEROKEE_COMMON_H__ */
