/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee Benchmarker
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "thread.h"
#include "bench.h"


static ret_t
thread_del_connection (cherokee_bench_thread_t *thread, cherokee_bench_connection_t *conn)
{
	list_del ((list_t *)conn);
	return cherokee_bench_connection_free (conn);
}


static void *
thread_routine (void *data)
{
	ret_t   ret;
	int     r, step;
	list_t *i, *tmp;
	cherokee_bench_thread_t *thread = THREAD(data);
	cherokee_bench_t        *bench  = THREAD_BENCH(thread);

	for (;;) {
		/* Maybe some connections have to connect to server
		 */
		list_for_each_safe (i, tmp, &thread->conns_list) {
			cherokee_bench_connection_t *conn = CONN(i);

			if (SOCKET_STATUS(conn->socket) == socket_closed) {

				/* Get the URL object
				 */
				ret = _cherokee_bench_get_request (bench, &conn->req);
				if (ret < ret_ok) return NULL;
				
				/* Connect to host
				 */
				ret = cherokee_bench_connection_connect (conn);	
				if (ret < ret_ok) {
					cherokee_url_t *url = REQUEST_URL(conn->req);

					PRINT_ERROR ("ERROR: Can not connect to '%s', port %d\n", 
						     URL_HOST(url)->buf, URL_PORT(url));

					thread->error = thread_error;
					goto terminate;
				}
			}
		}

		if (thread->errors.connection > ERRORS_MAX_CONNECTIONS) {
			PRINT_ERROR ("ERRORS_MAX_CONNECTIONS\n");
			goto terminate;
		}


		/* Examine the file descriptors
		 */
		r = cherokee_fdpoll_watch (thread->fdpoll, 1000);
		if (r <= 0) continue;

		
		/* Iterate each connection
		 */
		list_for_each_safe (i, tmp, &thread->conns_list) {
			cherokee_bench_connection_t *conn = CONN(i);
			
			/* Check if the socket can read/write
			 */
			if (cherokee_fdpoll_check (thread->fdpoll, 
						   SOCKET_FD(conn->socket), 
						   SOCKET_STATUS(conn->socket)) == 0) {
				continue;
			}

			/* Do something
			 */
			ret = cherokee_bench_connection_step (conn);

			/* Check if it is finished
			 */
			if (thread->info.transfers >= THREAD_CONNS_TODO(thread)) {
				goto terminate;
			}
		}
	}

terminate:
	return NULL;
}


static void
create_connections (cherokee_bench_thread_t *thread) 
{
	int i;
	cherokee_bench_t *bench = THREAD_BENCH(thread);
	cherokee_bench_connection_t *conn;

	for (i=0; i<BENCH_CONCU(bench); i++) {
		/* Create the new object
		 */
		cherokee_bench_connection_new (&conn);
		list_add ((list_t *)conn, &thread->conns_list);

		/* Set the basic config
		 */
		conn->thread = thread;
	}
}


ret_t 
cherokee_bench_thread_new  (cherokee_bench_thread_t **thread, void *bench, int fd_num)
{
	ret_t ret;

	CHEROKEE_NEW_STRUCT(n,bench_thread);

	/* Init lists
	 */
	INIT_LIST_HEAD((list_t *)&n->list_entry);
	INIT_LIST_HEAD((list_t *)&n->conns_list);
	
	/* File descriptors poll
	 */
	ret = cherokee_fdpoll_new (&n->fdpoll, fd_num);
	if (ret < ret_ok) {
		PRINT_ERROR ("Can not create file descriptors poll\n");
		return ret_error;
	}

	/* Info collector
	 */
	ret = cherokee_bench_info_init (&n->info);
	if (ret < ret_ok) return ret_ok;

	/* Return the object
	 */
	*thread = n;

	/* Init
	 */
	n->conns_todo = 0;
	n->bench      = bench;

	/* Init errors
	 */
	n->error = thread_ok;

	n->errors.connection = 0;
	n->errors.aborted    = 0;
	
	/* Create connections
	 */
	create_connections(n);

	return ret_ok;
}


ret_t 
cherokee_bench_thread_run (cherokee_bench_thread_t *thread)
{
	int re;

	/* Start the thread
	 */
	re = pthread_create (&thread->thread, NULL, thread_routine, thread);
	if (re != 0) return ret_error;

	return ret_ok;
}


ret_t 
cherokee_bench_thread_free (cherokee_bench_thread_t *thread)
{
	list_t *i, *tmp;

	/* Free the connection objects
	 */
	list_for_each_safe (i, tmp, &thread->conns_list) {
		thread_del_connection (thread, CONN(i));
	}

	/* Free the descriptors poll object
	 */
	cherokee_fdpoll_free (thread->fdpoll);

	/* Free "self"
	 */
	free (thread);
	return ret_ok;
}
