/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"

#include <sys/types.h>
#include <limits.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>  

#include "access.h"


typedef enum { 
	ipv4 = AF_INET, 
	ipv6 = AF_INET6
} ip_type_t;

typedef struct {
	union {
		struct in_addr  ip4;
#ifdef HAVE_IPV6
		struct in6_addr ip6;
#endif
	};
} ip_t;

typedef struct {
	struct list_head node;
	
	ip_type_t type;
	ip_t      ip;
} ip_item_t;

typedef struct {
	ip_item_t base;
	ip_t      mask;
} subnet_item_t;

#define IP_NODE(x)     ((ip_item_t *)(x))
#define SUBNET_NODE(x) ((subnet_item_t *)(x))


ip_item_t *
new_ip (void) 
{
	ip_item_t *n = malloc (sizeof(ip_item_t));
	if (n == NULL) return NULL;
	INIT_LIST_HEAD((list_t*)n);
	return n;
}

subnet_item_t *
new_subnet (void) 
{
	subnet_item_t *n = malloc (sizeof(subnet_item_t));
	if (n == NULL) return NULL;
	INIT_LIST_HEAD((list_t*)n);
	return n;
}


ret_t 
cherokee_access_new (cherokee_access_t **entry)
{
	CHEROKEE_NEW_STRUCT (n, access);

	INIT_LIST_HEAD(&n->list_ips);
	INIT_LIST_HEAD(&n->list_subnets);
	
	*entry = n;
	return ret_ok;
}



static void
print_ip (ip_type_t type, ip_t *ip)
{
	CHEROKEE_TEMP(dir,255);

#ifdef HAVE_INET_PTON
# ifdef HAVE_IPV6
	if (type == ipv6) {
		printf (inet_ntop (AF_INET6, ip, dir, dir_size));
		return;
	}
# endif
	if (type == ipv4) {
		printf (inet_ntop (AF_INET, ip, dir, dir_size));
		return;
	}
#else
	printf ("%s", inet_ntoa (ip.ipv4));
#endif
}


ret_t 
cherokee_access_free (cherokee_access_t *entry)
{
	list_t *i, *tmp;
	
	/* Free the IP list items
	 */
	list_for_each_safe (i, tmp, (list_t*)&entry->list_ips) {
		list_del (i);
		free (i);
	}

	/* Free the Subnet list items
	 */
	list_for_each_safe (i, tmp, (list_t*)&entry->list_subnets) {
		list_del (i);
		free (i);
	}

	free (entry);
	return ret_ok;
}


static ret_t
parse_ip (char *ip, ip_item_t *n)
{
	int  ok;

	/* Test if it is a IPv4 or IPv6 connection
	 */
	n->type = ((index (ip, ':') != NULL) || 
		   (strlen(ip) > 15)) ? ipv6 : ipv4;
	
	/* Parse the IP string
	 */
#ifdef HAVE_INET_PTON
	ok = (inet_pton (n->type, ip, &n->ip) > 0);

#ifdef HAVE_IPV6
	if (n->type == ipv6) {
		if (IN6_IS_ADDR_V4MAPPED (&(n->ip).ip6)) {
			PRINT_ERROR ("This IP '%s' is IPv6-mapped IPv6 address.  "
				     "Please, specify IPv4 in a.b.c.d style instead "
				     "of ::ffff:a.b.c.d style\n", ip);
			return ret_error;
		}
	}
#endif /* HAVE_IPV6 */

#else
	ok = (inet_aton (ip, &n->ip) != 0);
#endif

	return (ok) ? ret_ok : ret_error;
}
 

static ret_t
parse_netmask (char *netmask, subnet_item_t *n)
{
	int num;

	/* IPv6 or IPv4 Mask
	 * Eg: 255.255.0.0
	 */
	if ((index (netmask, ':') != NULL) ||
	    (index (netmask, '.') != NULL))
	{
		int ok;

#ifdef HAVE_INET_PTON
		ok = (inet_pton (IP_NODE(n)->type, netmask, &n->mask) > 0);
#else
		ok = (inet_aton (netmask, &n->mask) != 0);
#endif		
		return (ok) ? ret_ok : ret_error;
	}


	/* Lenght mask
	 * Eg: 16
	 */
	if (strlen(netmask) > 3) {
		return ret_error;
	}

	num = strtol(netmask, NULL, 10);
	if (num <= LONG_MIN) 
		return ret_error;

	/* Sanity checks
	 */
	if ((IP_NODE(n)->type == ipv4) && (num >  32)) 
		return ret_error;

	if ((IP_NODE(n)->type == ipv6) && (num > 128)) 
		return ret_error;

#ifdef HAVE_IPV6
	if (num > 128) {
		return ret_error;
	}

	/* Special case
	 */
	if (num == 128) {
		n->mask.ip6.in6_u.u6_addr32[0] = ~0L;
		n->mask.ip6.in6_u.u6_addr32[1] = ~0L;
		n->mask.ip6.in6_u.u6_addr32[2] = ~0L;
		n->mask.ip6.in6_u.u6_addr32[3] = ~0L;

		return ret_ok;
	}

	if (IP_NODE(n)->type == ipv6) {
		int i, j, jj;
		unsigned char mask    = 0;
		unsigned char maskbit = 0x80L;

		j  = (int) num / 8;
		jj = num % 8;

		for (i=0; i<j; i++) {
			n->mask.ip6.in6_u.u6_addr8[i] = 0xFF;
		}

		while (jj--) {
			mask |= maskbit;
			maskbit >>= 1;
		}
		n->mask.ip6.in6_u.u6_addr8[j] = mask;

		return ret_ok;
	} else
#endif
		n->mask.ip4.s_addr = (in_addr_t) htonl(~0L << (32 - num));

	return ret_ok;
}


ret_t 
cherokee_access_add_ip (cherokee_access_t *entry, char *ip)
{
	ret_t ret;
	ip_item_t *n;

	n = new_ip();
	if (n == NULL) return ret_error;

	list_add ((list_t *)n, &entry->list_ips);

	ret = parse_ip (ip, n);
	if (ret < ret_ok) {
		PRINT_ERROR ("IP address '%s' seems to be invalid\n", ip);
	}

	return ret;
}


ret_t 
cherokee_access_add_subnet (cherokee_access_t *entry, char *subnet)
{
	char *slash;
	char *mask;
	ret_t ret;
	subnet_item_t *n;
	CHEROKEE_NEW (ip,buffer);

	/* Split the string
	 */
	slash = strpbrk (subnet, "/\\");
	if (slash == NULL) return ret_error;

	mask = slash +1;
	cherokee_buffer_add (ip, subnet, mask-subnet-1);
	
	/* Create the new list object
	 */
	n = new_subnet();
	if (n == NULL) return ret_error;

	list_add ((list_t *)n, &entry->list_subnets);

	/* Parse the IP
	 */
	ret = parse_ip (ip->buf, IP_NODE(n));
	if (ret < ret_ok) {
		PRINT_ERROR ("IP address '%s' seems to be invalid\n", ip->buf);
		goto error;
	}

	/* Parse the Netmask
	 */
	ret = parse_netmask (mask, n);
	if (ret < ret_ok) {
		PRINT_ERROR ("Netmask '%s' seems to be invalid\n", mask);
		goto error;	
	}

	cherokee_buffer_free (ip);
	return ret_ok;

error:
	cherokee_buffer_free (ip);
	return ret_error;
}


ret_t 
cherokee_access_print_debug (cherokee_access_t *entry)
{
	list_t *i;

	printf ("IPs: ");
	list_for_each (i, (list_t*)&entry->list_ips) {
		print_ip (IP_NODE(i)->type, &IP_NODE(i)->ip);
		printf(" ");
	}
	printf("\n");

	printf ("Subnets: ");
	list_for_each (i, (list_t*)&entry->list_subnets) {
		print_ip (IP_NODE(i)->type, &IP_NODE(i)->ip);
		printf("/");
		print_ip (IP_NODE(i)->type, &SUBNET_NODE(i)->mask);
		printf(" ");
	}
	printf("\n");

	return ret_ok;
}


ret_t 
cherokee_access_ip_match (cherokee_access_t *entry, cherokee_socket_t *sock)
{
	int     re;
	ret_t   ret;
	list_t *i;

	/* Check in the IP list
	 */
	list_for_each (i, (list_t*)&entry->list_ips) {
		if (SOCKET_AF(sock) == IP_NODE(i)->type) {
			switch (IP_NODE(i)->type) {
			case ipv4:
				re = memcmp (&SOCKET_ADDR_IPv4(sock)->sin_addr, &IP_NODE(i)->ip, 4);
				break;
			case ipv6:
				re = memcmp (&SOCKET_ADDR_IPv6(sock)->sin6_addr, &IP_NODE(i)->ip, 16);
				break;
			default:
				SHOULDNT_HAPPEN;
				return ret_error;
			}

			if (re == 0) {
				return ret_ok;
			}
		}
	}
	
	/* Check in the Subnets list
	 */
	list_for_each (i, (list_t*)&entry->list_subnets) {
		int j, re;
		ip_t masqued_remote, masqued_list;
		
		if (SOCKET_AF(sock) == IP_NODE(i)->type) {
			switch (IP_NODE(i)->type) {
			case ipv4:
				printf ("Es IPv4\n");

				masqued_list.ip4.s_addr   = (IP_NODE(i)->ip.ip4.s_addr &
							     SUBNET_NODE(i)->mask.ip4.s_addr);
				masqued_remote.ip4.s_addr = (SOCKET_ADDR_IPv4(sock)->sin_addr.s_addr & 
							     SUBNET_NODE(i)->mask.ip4.s_addr);

				if (masqued_remote.ip4.s_addr == masqued_list.ip4.s_addr) {
					return ret_ok;
				}

				break;
#ifdef HAVE_IPV6
			case ipv6:
			{
				int equal = 1;

//				printf ("remote IPv6: ");
//				print_ip (ipv6, &SOCKET_ADDR_IPv6(sock)->sin6_addr);
//				printf ("\n");

				for (j=0; j<16; j++) {
					masqued_list.ip6.in6_u.u6_addr8[j] = (
						IP_NODE(i)->ip.ip6.in6_u.u6_addr8[j] &
						SUBNET_NODE(i)->mask.ip6.in6_u.u6_addr8[j]);
					masqued_remote.ip6.in6_u.u6_addr8[j] = (
						SOCKET_ADDR_IPv6(sock)->sin6_addr.in6_u.u6_addr8[j] &
						SUBNET_NODE(i)->mask.ip6.in6_u.u6_addr8[j]);

					if (masqued_list.ip6.in6_u.u6_addr8[j] !=
					    masqued_remote.ip6.in6_u.u6_addr8[j])
					{
						equal = 0;
						break;
					}
				}

				if (equal == 1) {
					return ret_ok;
				}
				break;
			}
#endif
			default:
				SHOULDNT_HAPPEN;
				return ret_error;
			}
		}
	}
	
	return ret_not_found;
}
