/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>

#include "mime.h"
#include "handler_file.h"
#include "connection.h"
#include "module.h"
#include "mmap.h"


#if defined(LINUX_SENDFILE_API) || defined(SOLARIS_SENDFILE_API)
# include <sys/sendfile.h>
#elif defined(HPUX_SENDFILE_API)
# include <sys/socket.h>
# include <sys/uio.h>
#elif defined(FREEBSD_SENDFILE_API)
# include <sys/types.h>
# include <sys/socket.h>
# include <sys/uio.h>
#elif defined(LINUX_BROKEN_SENDFILE_API)
 extern int32 sendfile (int out_fd, int in_fd, int32 *offset, uint32 count);
#endif


cherokee_module_info_t cherokee_file_info = {
	cherokee_handler,            /* type     */
	cherokee_handler_file_new    /* new func */
};

ret_t
cherokee_handler_file_new  (cherokee_handler_t **hdl, cherokee_connection_t *cnt, cherokee_table_t *properties)
{
	CHEROKEE_NEW_STRUCT (n, handler_file);
	
	/* Init the base class object
	 */
	cherokee_handler_init_base(HANDLER(n), cnt);

	HANDLER(n)->support = hsupport_length | hsupport_range;

	HANDLER(n)->init        = (handler_func_init_t) cherokee_handler_file_init;
	HANDLER(n)->free        = (handler_func_free_t) cherokee_handler_file_free;
	HANDLER(n)->step        = (handler_func_step_t) cherokee_handler_file_step;
	HANDLER(n)->add_headers = (handler_func_add_headers_t) cherokee_handler_file_add_headers;
	
	/* Init
	 */
	n->offset = 0;
	n->mime   = NULL;

	*hdl = HANDLER(n);
	return ret_ok;
}

static ret_t
check_cached (cherokee_handler_file_t *n)
{
	ret_t ret;
	char *header;
	int   header_len;
	cherokee_connection_t *conn;

	conn = CONN(HANDLER(n)->connection);

		
	/* Based in time
	 */
	ret = cherokee_header_get_unknown (conn->header, "If-Modified-Since", 17, &header, &header_len);
	if (ret == ret_ok)  {
		time_t req_time;
		char  *tmp;
		char  *end = header + header_len;

		tmp = *end;    /* save */
		*end = '\0';   /* set  */

		req_time = tdate_parse (header);			
		if (req_time == -1) {
			cherokee_logger_write_string (
				CONN_VSRV(conn)->logger, 
				"Warning: Unparseable time '%s'",
				header);
		}
		*end = tmp;   /* restore */
		
		if (req_time == -1) {
			return ret_ok;
		}
		
		
		/* The file is cached in the client
		 */
		if (n->info.st_mtime <= req_time) {
			conn->error_code = http_not_modified;
			return ret_error;	
		}
	}
	
	if (conn->header->version == http_version_11) {
		/* Based in ETag
		 */
		ret = cherokee_header_get_unknown (conn->header, "If-None-Match", 13, &header, &header_len);
		if (ret == ret_ok)  {
			int    tmp_len;
			CHEROKEE_TEMP(tmp,100);
			
			tmp_len = snprintf (tmp, tmp_size, "%lx=%lx", n->info.st_mtime, n->info.st_size);

			if (strncmp (header, tmp, tmp_len) == 0) {
				conn->error_code = http_not_modified;
				return ret_error;				
			}
		}
	}

	return ret_ok;
}

ret_t 
cherokee_handler_file_init (cherokee_handler_file_t *n)
{
	int   ret;
	char *ext;
	cherokee_connection_t *conn;

	conn = CONN(HANDLER(n)->connection);

	cherokee_buffer_add (conn->local_directory, conn->request->buf+1, conn->request->len-1); /* (1) */
	n->fd = open (conn->local_directory->buf, O_RDONLY);

	/* Cannot open the file
	 */
	if (n->fd != -1) {
		ret = fstat (n->fd, &n->info);
	} else {
		ret = stat (conn->local_directory->buf, &n->info);
	}

	cherokee_buffer_drop_endding (conn->local_directory, conn->request->len);                /* (1') */


	if (n->fd == -1) {
		if ((ret == 0) && S_ISREG(n->info.st_mode)) {
			conn->error_code = http_access_denied;
		} else {
			conn->error_code = http_not_found;
		}
		return ret_error;
	}


	/* Is dir?
	 */
	if (S_ISDIR(n->info.st_mode)) {
		conn->error_code = http_access_denied;
		return ret_error;		
	}

	/* Is it cached in the client?
	 */
	ret = check_cached(n);
	if (ret != ret_ok) {
		return ret;
	}

	/* Look for the mime type
	 */
	ext = rindex (conn->request->buf, '.');
	if (ext != NULL) {
		cherokee_mime_t *m;

		ret = cherokee_mime_get_default (&m);
		if (ret >= ret_ok) {
			ret = cherokee_mime_get (m, ext+1, &n->mime);
		}
	}

	/* Maybe it is cached in mmap()
	 */
	if (n->info.st_size <= MAX_MMAP_FILE_SIZE) {
		void            *info;
		size_t           info_len;
		cherokee_mmap_t *mmap;

		cherokee_mmap_get_default (&mmap);
		ret = cherokee_mmap_get_fast (mmap, n->fd, &n->info, &info, &info_len);
		if (ret == ret_ok) {
			conn->mmaped     = info;
			conn->mmaped_len = info_len;
		}
	}
	
	/* Range
	 */
	if (conn->range_start != 0) {
		n->offset = conn->range_start;
		lseek (n->fd, n->offset, SEEK_SET);
		conn->error_code = http_partial_content;
	}

	if (conn->range_end == 0) {
		conn->range_end = n->info.st_size;
	}


	/* Maybe use sendfile
	 */
#if HAVE_SENDFILE
	n->using_sendfile = ((conn->mmaped == NULL) &&
			     (n->info.st_size >= 32768) && 
			     (n->info.st_size < 2147483647) &&
			     conn->encoder == NULL);

	if (n->using_sendfile) {
		cherokee_connection_set_cork(conn, 1);
	}
#endif

	return ret_ok;
}


ret_t
cherokee_handler_file_free (cherokee_handler_file_t *fhdl)
{
	if (fhdl->fd != -1) {
		close (fhdl->fd);
		fhdl->fd = -1;
	}

	free (fhdl);
	
	return ret_ok;
}


ret_t
cherokee_handler_file_step (cherokee_handler_file_t *fhdl, 
			    cherokee_buffer_t       *buffer)
{
	int total;

#if HAVE_SENDFILE
	if (fhdl->using_sendfile) {
		ssize_t sent;
		cherokee_connection_t *conn = CONN(HANDLER(fhdl)->connection);
		

#ifdef FREEBSD_SENDFILE_API
		/* FreeBSD sendfile: in_fd and out_fd are reversed
		 *
		 * int
		 * sendfile(int fd, int s, off_t offset, size_t nbytes,
		 * struct sf_hdtr *hdtr, off_t *sbytes, int flags);
		 */
		sent = sendfile (fhdl->fd,                        /* int             fd     */
				 SOCKET_FD(conn->socket),         /* int             s      */
				 &fhdl->offset,                   /* off_t           offset */
				 conn->range_end - fhdl->offset,  /* size_t          nbytes */
				 NULL,                            /* struct sf_hdtr *hdtr   */
				 &fhdl->offset,                   /* off_t          *sbytes */ 
				 0);                              /* int             flags  */
#else
		sent = sendfile (SOCKET_FD(conn->socket),         /* int    out_fd */
				 fhdl->fd,                        /* int    in_fd  */
				 &fhdl->offset,                   /* off_t *offset */
				 conn->range_end - fhdl->offset); /* size_t count  */
#endif

		/* cherokee_handler_file_init() activated the TCP_CORK flags.
		 * After it, the header was sent.  And now, the first
		 * chunk of the file with sendfile().  It's time to turn
		 * off again the TCP_CORK flag
		 */
		if (conn->tcp_cork) {
			cherokee_connection_set_cork(conn, 0);
		}


		if (sent < 0) {
			return ret_error;
		}

		if (fhdl->offset >= conn->range_end) {
			return ret_eof;
		}

		return ret_ok_and_sent;
	}
#endif


	total = read (fhdl->fd, buffer->buf, buffer->size);	
	switch (total) {
	case 0:
		return ret_eof;
	case -1:
		return ret_error;
	default:
		buffer->len = total;
		fhdl->offset += total;
	}	

	/* Maybe it was the last file chunk
	 */
	if (fhdl->offset >= CONN(HANDLER(fhdl)->connection)->range_end) {
		return ret_eof_have_data;
	}

	return ret_ok;
}


ret_t
cherokee_handler_file_add_headers (cherokee_handler_file_t *fhdl,
				   cherokee_buffer_t       *buffer)
{
	int length;
	int len;
	CHEROKEE_TEMP(tmp,100);

	/* Add "Content-length:" header
	 */
	if (fhdl->fd != -1) {
		/* We stat()'ed the file in the handler constructor
		 */
		length = HANDLER_CONN(fhdl)->range_end - HANDLER_CONN(fhdl)->range_start;
		if (length < 0) {
			length = 0;
		}

		if (HANDLER_SUPPORT_LENGTH(fhdl) && 
		    (HANDLER_CONN(fhdl)->encoder == NULL)) 
		{
			cherokee_buffer_add_va (buffer, "Content-length: %d"CRLF, length);
		}

	} else {
		/* Can't use Keep-alive w/o "Content-length:", so
		 */
		HANDLER_CONN(fhdl)->keepalive = 0;
	}

	/* Add MIME related headers: 
	 * "Content-Type:" and "Cache-Control: max-age="
	 */
	if (fhdl->mime != NULL) {
		cherokee_buffer_add (buffer, "Content-Type: ", 14);
		cherokee_buffer_add_buffer (buffer, MIME_ENTRY_NAME(fhdl->mime));
		cherokee_buffer_add (buffer, CRLF, 2);

		if (MIME_ENTRY_AGE(fhdl->mime) != -1) {
			cherokee_buffer_add_va (buffer, "Cache-Control: max-age=%d"CRLF, 
						MIME_ENTRY_AGE(fhdl->mime));
		}
	}

	/* Etag
	 */
	if (HANDLER_CONN(fhdl)->header->version >= http_version_11) { 
		cherokee_buffer_add_va (buffer, "Etag: %lx=%lx"CRLF, fhdl->info.st_mtime, fhdl->info.st_size);
	}

	/* Last-Modified:
	 */
	len = strftime (tmp, tmp_size, "Last-Modified: %a, %d %b %Y %H:%M:%S GMT"CRLF, gmtime(&fhdl->info.st_mtime));
	cherokee_buffer_add (buffer, tmp, len);

	return ret_ok;
}



/*   Library init function
 */
static int _file_is_init = 0;

void
file_init ()
{
	ret_t ret;
	cherokee_mime_t *mime;
	cherokee_mmap_t *mmap_cache;

	/* Init flag
	 */
	if (_file_is_init) {
		return;
	}
	_file_is_init = 1;


	/* Library dependences: mime
	 */
	ret = cherokee_mime_init (&mime);
	if (ret < ret_ok) {
		PRINT_ERROR("Can not init MIME module\n");
		return;
	}

	/* Library dependences: mmap
	 */
	ret = cherokee_mmap_init (&mmap_cache);
	if (ret < ret_ok) {
		PRINT_ERROR("Can not init MMAP module\n");
		return;
	}
}
