/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>

#include "table.h"


typedef struct {
	   char *key;
	   void *value;
} item_t;


static inline int
equal (const void *avl_a, const void *avl_b, void *avl_param)
{
	avl_param = avl_param;

	return strcmp(((item_t *)avl_a)->key, ((item_t *)avl_b)->key);
}


static inline void
del_item (void *avl_item, void *avl_param)
{
	free (((item_t *)avl_item)->key);
}


ret_t
cherokee_table_new (cherokee_table_t **tab)
{
	cherokee_table_t *n = (cherokee_table_t *) malloc (sizeof(cherokee_table_t));
	return_if_fail (n != NULL, ret_nomem);
	
	cherokee_table_init (n);
	
	*tab = n;
	return ret_ok;
}


ret_t
cherokee_table_init (cherokee_table_t *tab)
{
	tab->tree = avl_create (equal, NULL, NULL);
	
	return ret_ok;
}


ret_t
cherokee_table_free (cherokee_table_t *tab)
{
	cherokee_table_clean (tab);

	free (tab);
	return ret_ok;
}


ret_t 
cherokee_table_free2(cherokee_table_t  *tab, cherokee_table_free_item_t free_func)
{
	cherokee_table_clean2 (tab, free_func);

	free (tab);
	return ret_ok;	
}

ret_t
cherokee_table_clean (cherokee_table_t *tab)
{
	if (tab->tree) {
		avl_destroy (tab->tree, del_item);
		tab->tree = NULL;
	}
	
	return ret_ok;
}


ret_t 
cherokee_table_clean2(cherokee_table_t  *tab, cherokee_table_free_item_t free_func)
{
	struct avl_traverser  trav;
	item_t               *item;


	if (tab->tree == NULL) 
		return ret_ok;


	/* We've to visit all the nodes of the tree
	 * to free all the 'value' entries
	 */

	avl_t_init (&trav, tab->tree);

	item = (item_t *) avl_t_first (&trav, tab->tree);
	if (item != NULL) {
		free_func (item->value);
	}
	while ((item = avl_t_next(&trav)) != NULL) {
		free_func (item->value);
	}

	return cherokee_table_clean(tab);
}


ret_t
cherokee_table_add (cherokee_table_t *tab, char *key, void *value)
{
	item_t *n = (item_t *)malloc(sizeof(item_t));
	n->key   = strdup (key);
	n->value = value;
	
	avl_insert (tab->tree, n);
	
	return ret_ok;
}


void*
cherokee_table_get (cherokee_table_t *tab, char *key)
{
	item_t n;
	item_t *found;
	
	n.key = key;
	
	found = avl_find (tab->tree, &n);
	
	if (found)
		return found->value;
	
	return NULL;
}


