/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "mime.h"
#include "handler_file.h"


ret_t
cherokee_handler_file_new  (cherokee_handler_file_t **hdl, char *request)
{
	cherokee_handler_file_t *n = (cherokee_handler_file_t *)malloc(sizeof(cherokee_handler_file_t));
	return_if_fail (n != NULL, ret_nomem);

	/* Init the base class object
	 */
	cherokee_handler_init(HANDLER(n));
	HANDLER(n)->free               = cherokee_handler_file_free;
	HANDLER(n)->step               = cherokee_handler_file_step;
	HANDLER(n)->get_content_length = cherokee_handler_file_get_content_length;
	
	*hdl = n;
	
	/* Init
	 */
	n->f = fopen (request, "r");
	if (n->f == NULL) {
		struct stat info;
		if ((stat (request, &info) == 0) && S_ISREG(info.st_mode)) {
			HANDLER(n)->status = http_access_denied;
		} else {
			HANDLER(n)->status = http_not_found;
		}
		return ret_error;
	}
	
	return ret_ok;
}


static inline void
close_file (cherokee_handler_file_t *hdl)
{
	if (hdl->f != NULL) {
		fclose (hdl->f);
		hdl->f = NULL;
	}
}


ret_t
cherokee_handler_file_free (void *hdl)
{
	close_file (hdl);
	free (hdl);
	
	return ret_ok;
}


ret_t
cherokee_handler_file_step (void *hdl, cherokee_buffer_t *buffer)
{
	int readed;
	cherokee_handler_file_t *fhdl = FHANDLER(hdl);
	
	if (feof(fhdl->f)) {
		return ret_eof;
	}
	
	readed = fread (buffer->buf, 1, buffer->size, fhdl->f);
	buffer->len = readed;
	
	if (readed <= 0) {
		return ret_error;
	}
	
	return ret_ok;
}


int
cherokee_handler_file_get_content_length (void *hdl)
{
	int descriptor;
	struct stat info;
	cherokee_handler_file_t *fhdl = FHANDLER(hdl);
	
	if (fhdl->f == NULL)
		return -1;
	
	
	descriptor = fileno(fhdl->f);
	fstat (descriptor, &info);
	
	return info.st_size;
}
