/*
 * UrlMapperHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 2000-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.3
 * Created by suhler on 00/02/14
 * Last modified by suhler on 04/11/03 08:29:49
 */

package sunlabs.brazil.handler;

import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.regexp.Regexp;
import sunlabs.brazil.util.Format;

import java.io.IOException;

/**
 * Handler for mapping or redirecting URLs.
 * Matches url's against a regexp pattern.  If it matches, the url
 * is either rewritten or redirected.
 * <p>
 * Properties:
 * <dl class=props>
 * <dt>match	<dd>The regexp to match a url.  May contain constructs
 *		of the form ${xxx}, which are replaced by the value of
 *		<code>request.props</code> for the key <i>xxx</i>
 * <dt>replace  <dd>The url to replace it with.  This may contain both
 *		regular expression sub-patterns, such as "\1", or
 *		variables of the form ${..} which are replaced with
 *		the equivalent request properties.
 * <dt>export   <dd>If set, use this as a properties prefix, and set 
 *		    request properties for each sub-expression in "match".
 *                  (E.g. [export]1 [export]2 ...).
 * <dt>redirect <dd>If set, the request is redirected instead of being rewritten
 * <dt>ignoreCase <dd>If set, the case of the expression is ignored.
 * <dt>source
 * <dd>
 * If set, then this string is used instead of the
 * url as the <i>source</i> of the match. Variable substitution
 * using ${xxx} is performed on <i>source</i>, which, if unset, 
 * defaults to "${url}".  If set, the request properties
 * "url", "host" and "user-agent" are set first.
 * The <i>source</i> property
 * is obtained at init time, but evaluated (for ${...}) at every request.
 * <p>
 * As an example, the configuration:<br><code>
 * prefix.source=${user-agent}!${url}<br>
 * prefix.match=Lynx.*!(.*)<br>
 * prefix.replace=/text\\1<br></code>
 * could cause all browsers with "Lynx" in their user agent header
 * to the "text" sub-directory.
 * </dl>
 *
 * @author      Stephen Uhler
 * @version     2.3, 04/11/03
 */

public class UrlMapperHandler implements Handler {
    final static String MATCH = "match";
    final static String SOURCE = "source";
    final static String REPLACE = "replace";
    final static String REDIRECT = "redirect";
    final static String NOCASE = "ignoreCase";
    final static String EXPORT = "export";

    Regexp re = null;		// the expression to match
    String replace;		// the replacement string
    String export;		// the prefix for exported submatches
    String source;		// the source for the match
    boolean redirect = false;	// True to redirect instead of rewrite

    public boolean
    init(Server server, String prefix) {
	boolean noCase = (server.props.getProperty(prefix + NOCASE) != null);
	redirect = (server.props.getProperty(prefix + REDIRECT) != null);
	replace = server.props.getProperty(prefix + REPLACE);
	export = server.props.getProperty(prefix + EXPORT);
	source = server.props.getProperty(prefix + SOURCE);
	if (replace == null) {
	    server.log(Server.LOG_WARNING, prefix, "missing: " + REPLACE);
	    return false;
	}
	String match = server.props.getProperty(prefix + MATCH);
	if (match == null) {
	    server.log(Server.LOG_WARNING, prefix, "missing: " + MATCH);
	    return false;
	}
	match = Format.subst(server.props, match);
	try {
	    re = new Regexp(match, noCase);
	} catch (Exception e) {
	    server.log(Server.LOG_WARNING, prefix, "Bad expression:" + e);
	}
	return (re != null);
    }

    /**
     * If this request matches the expression, rewrite it.
     */

    public boolean
    respond(Request request) throws IOException {
	String formatted;
	if (source != null) {
	    request.props.put("url", request.url);
	    request.props.put("user-agent", request.headers.get("user-agent"));
	    request.props.put("host", request.headers.get("host"));
	    formatted = Format.subst(request.props, source);
	} else {
	    formatted = request.url;
	}
	request.log(Server.LOG_DIAGNOSTIC, null, "Matching " + formatted);
	String sub = re.sub(formatted, replace);
	if (sub == null) {
	    return false;
	}
	sub = Format.subst(request.props, sub);
	if (export != null) {
	    if (!export.endsWith(".")) {
		export += ".";
	    }
	    String[] parts = new String[re.subspecs()];
	    re.match(request.url, parts);
	    for (int i=0; i<parts.length; i++) {
		request.props.put(export + i, parts[i]);
	    }
	}

	request.log(Server.LOG_DIAGNOSTIC, null, "Mapping " +
		formatted + " -> " + sub);
	if (redirect) {
	    request.redirect(sub,null);
	    return true;
	} else {
	    request.url = sub;
	    return false;
	}
    }
}
