/*
 * GenericProxyHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1998-2002 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, rinaldo, suhler.
 *
 * Version:  2.1
 * Created by suhler on 98/09/14
 * Last modified by suhler on 02/10/01 16:36:22
 */

package sunlabs.brazil.handler;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.net.ConnectException;
import java.net.UnknownHostException;
import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.http.HttpRequest;
import sunlabs.brazil.util.http.MimeHeaders;
import sunlabs.brazil.util.Format;
import java.util.StringTokenizer;

/**
 * Handler for implementing a virtual web site.
 * This causes another web site to "appear" inside our document root.
 * This classes is intended to be sub-classed, so some of the methods
 * in this implementation don't do too much.
 *
 * All of the appropriate links in HTML documents on the virtual
 * site are rewritten, so they appear to be local references.
 * This can be used on a firewall in conjunction with
 * {@link AclSwitchHandler}
 * to provide authenticated access to selected web sites.
 * <p>
 * Properties:
 * <dl class=props>
 * <dt>prefix	<dd>URL prefix must match
 * <dt>host	<dd>name of host site to proxy to.
 * <dt>port	<dd>Host port to proxy to (defaults to 80).
 * <dt>proxyHost<dd>Which proxy host to use (if any)
 *		 to contact "host".
 * <dt>proxyPort<dd>The proxy's port (defaults to 80)
 * <dt>headers	<dd>A list of white space delimited tokens that refer to
 *		additional HTTP headers that are added onto the polled
 *		request.  For each token the server properties
 *		<code>[token].name</code> and <code>[token].value</code>
 *		define a new http header.
 * </dl>
 *
 * @author      Stephen Uhler
 * @version	2.1, 02/10/01
 */

public class GenericProxyHandler implements Handler {
//    protected Server server;
    protected String prefix;

    protected String host;	   // The host containing the actual page
    protected int port;		   // The port for above (defaults to 80)
    protected String proxyHost;	   // The proxy server (if any)
    protected int proxyPort;	   // The proxy server's port (defaults to 80)
    protected String urlPrefix;	   // The url prefix that triggers this handler
    protected String requestPrefix;  // The host/port prefix
    protected String tokens;		// our tokens in server.props

    /**
     * Handler configuration property <b>prefix</b>.
     * Only URL's that begin with this string are considered by this handler.
     * The default is (/).
     */
    public static final String PREFIX = "prefix";   // URL prefix for proxy
    /**
     * Handler configuration property <b>host</b>.
     * The actual host site to appear on our site (required)
     */
    public static final String HOST = "host";     // The host to proxy these requests to
    /**
     * Handler configuration property <b>port</b>.
     * The actual port on the host site (defaults to 80).
     */
    public static final String PORT = "port";     // The host port to proxy these requests to
    /**
     * Handler configuration property <b>proxyHost</b>.
     * The name of a proxy to use (if any) to get to the <b>host</b>.
     */
    public static final String PROXY_HOST = "proxyHost";     // The proxy server (if any)
    /**
     * Handler configuration property <b>proxyPort</b>.
     * The proxy port to use to get to the <b>host</b>. defaults to 80.
     */
    public static final String PROXY_PORT = "proxyPort";     // The proxy server port (if any)
    public static final String NL = "\r\n";		// line terminator

    /**
     * Do one-time setup.
     * get and process the handler properties.
     * we can contact the server identified by the <b>host</b> parameter.
     */

    public boolean
    init(Server server, String prefix) {
//	this.server = server;
	this.prefix = prefix;

	if (server.logLevel > server.LOG_DIAGNOSTIC) {
	    MapPage.log = true;
	}

	/* XXX
	 * Need to add tag map entries here for a specific host
	 */

	host = server.props.getProperty(prefix + HOST);
	if (host == null) {
	    server.log(Server.LOG_WARNING, prefix, "no host to proxy to");
	    return false;
	}

	urlPrefix = server.props.getProperty(prefix + PREFIX, "/");
	if (urlPrefix.indexOf('/') != 0) {
	    urlPrefix = "/" + urlPrefix;
	}
	if (!urlPrefix.endsWith("/")) {
	    urlPrefix += "/";
	}

	port = 80;
	try {
	    String str = server.props.getProperty(prefix + PORT);
	    port = Integer.decode(str).intValue();
	} catch (Exception e) {}

	proxyHost = server.props.getProperty(prefix + PROXY_HOST);
	proxyPort = 80;
	try {
	    String str = server.props.getProperty(prefix + PROXY_PORT);
	    proxyPort = Integer.decode(str).intValue();
	} catch (Exception e) {}
	
	if (port == 80) {
	    requestPrefix = "http://" + host;
	} else {
	    requestPrefix = "http://" + host + ":" + port;
	}
	tokens = server.props.getProperty(prefix + "headers");
	return true;
    }

    /**
     * If this is one of "our" url's, fetch the document from
     * the destination server, and return it as if it was local.
     */

    public boolean
    respond(Request request)
    throws IOException {
    	if (!isMine(request)) {
	    return false;
	}	

	String url = request.url.substring(urlPrefix.length());
	if (!url.startsWith("/")) {
	    url = "/" + url;
	}

	if (!request.query.equals("")) {
	    url += "?" + request.query;
	}

	HttpRequest target = new HttpRequest(requestPrefix + url);
	// System.out.println("Fetching: " + requestPrefix + url);
	if (proxyHost != null) {
	    target.setProxy(proxyHost, proxyPort);
	}
	target.setMethod(request.method);

	HttpRequest.removePointToPointHeaders(request.headers, false);
        request.headers.remove("if-modified-since");  // wrong spot XXX
	request.headers.copyTo(target.requestHeaders);

	/* XXX This doesn't belong here! - the proxy should do it */
	target.requestHeaders.remove("host");
	if (tokens != null) {
	    target.addHeaders(tokens, request.props);
	}
	target.requestHeaders.putIfNotPresent("Host",host);

        boolean code=true;
	try {
	    if (request.postData != null) {
		OutputStream out = target.getOutputStream();
		out.write(request.postData);
		out.close();
	    }

	    // Connect to target and read the response headers

	    target.connect();
	    // System.out.println("Got headers: " + target.responseHeaders);
	    HttpRequest.removePointToPointHeaders(target.responseHeaders, true);
	    target.responseHeaders.copyTo(request.responseHeaders);

	    // Now filter the output, writing the header and content if true

	    request.log(Server.LOG_DIAGNOSTIC,
			"  Response headers: " +  target.responseHeaders);
	    if (shouldFilter(request.responseHeaders)) {
		
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		target.getInputStream().copyTo(out);

		request.log(Server.LOG_DIAGNOSTIC,
			"  parsing/modifying " + out.size() + " bytes");
		byte[] content = modifyContent(request, out.toByteArray());

		if (content == null) {	// This is wrong!!
		    request.log(Server.LOG_DIAGNOSTIC,
			    "  null content, returning false");
		    code=false;
		} else {
		    request.sendResponse(content, null);
		}
	    } else {
		request.log(Server.LOG_DIAGNOSTIC, "Delivering normal content");
		request.sendResponse(target.getInputStream(),
			target.getContentLength(), null,
			target.getResponseCode());
	    }
        } catch (InterruptedIOException e) {
            /*
             * Read timeout while reading from the remote side.  We use a
             * read timeout in case the target never responds.
             */
            request.sendError(408, "Timeout / No response");
        } catch (UnknownHostException e) {
            request.sendError(503, urlPrefix +  " Not reachable");
        } catch (ConnectException e) {
            request.sendError(500, "Connection refused");
	} catch (IOException e) {
	    request.sendError(500, "Error retrieving response: " + e);
	    e.printStackTrace();
        } finally {
            target.close();
	    // System.out.println("Finally (proxy): " + code);
        }
	return code;
    }

    /**
     * See if the content needs to be filtered.
     * Return "true" if "modifyContent" should be called
     * @param headers	Vector of mime headers for data to proxy
     */

    protected boolean shouldFilter(MimeHeaders headers) {
	String type = headers.get("Content-Type");
	// System.out.println("Modify?? " + type);
     	return (headers.get("location") != null || 
		(type != null && type.indexOf("text/html") >= 0));
    }

    /**
     * See if this is one of my requests.
     * This method can be overridden to do more sophisticated mappings.
     * @param request		The standard request object
     */

    public boolean isMine(Request request) {
	return  request.url.startsWith(urlPrefix);
    }

    /**
     * Rewrite the links in an html file so they resolve correctly
     * in proxy mode.
     *
     * @param request	The original request to this "proxy"
     * @param headers	The vector of mime headers for the proxy request
     * @return	 	true if the headers and content should be sent to the client, false otherwise
     *			Modifies "headers" as a side effect
     */

    public byte[] modifyContent(Request request, byte[] content) {
        MapPage mapper = new MapPage(urlPrefix);
        addMap(mapper);
	byte[] result;
	result = mapper.convertHtml(new String(content)).getBytes();

	/*
	 * Now fix the content length
	 */

	request.responseHeaders.put("Content-Length", result.length);

	/*
	 * Rewrite the location header, if any
	 */

	String location = request.responseHeaders.get("location");
	if (location != null) {
	    request.setStatus(302);	// XXX doesn't belong here, should copy
						// targets status
	    String fixed = mapper.convertString(location);
	    if (fixed != null) {
		String newLocation = request.serverUrl() + fixed;
		System.out.println("Mapping location: " + location + "->" + newLocation);
		request.responseHeaders.put("Location", newLocation);
	    }
	}

	/*
	 * Fix the domain specifiers on set-cookie requests
	 */

	String cookie = request.responseHeaders.get("set-cookie");
	if (cookie != null) {
	    System.out.println("Need to map: " + cookie);
	}

	return result;
    }

    /**
     * for subclassing
     */

    public void addMap(MapPage mapper) {}
}
