/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.buffer;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.time.Deadline;
import org.apache.flink.configuration.NettyShuffleEnvironmentOptions;
import org.apache.flink.configuration.TaskManagerOptions;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentFactory;
import org.apache.flink.core.memory.MemorySegmentProvider;
import org.apache.flink.runtime.io.AvailabilityProvider;
import org.apache.flink.runtime.io.network.buffer.BufferPool;
import org.apache.flink.runtime.io.network.buffer.BufferPoolFactory;
import org.apache.flink.runtime.io.network.buffer.LocalBufferPool;
import org.apache.flink.util.ExceptionUtils;
import org.apache.flink.util.MathUtils;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NetworkBufferPool
implements BufferPoolFactory,
MemorySegmentProvider,
AvailabilityProvider {
    private static final Logger LOG = LoggerFactory.getLogger(NetworkBufferPool.class);
    private final int totalNumberOfMemorySegments;
    private final int memorySegmentSize;
    private final ArrayDeque<MemorySegment> availableMemorySegments;
    private volatile boolean isDestroyed;
    private final Object factoryLock = new Object();
    private final Set<LocalBufferPool> allBufferPools = new HashSet<LocalBufferPool>();
    private int numTotalRequiredBuffers;
    private final Duration requestSegmentsTimeout;
    private final AvailabilityProvider.AvailabilityHelper availabilityHelper = new AvailabilityProvider.AvailabilityHelper();

    @VisibleForTesting
    public NetworkBufferPool(int numberOfSegmentsToAllocate, int segmentSize) {
        this(numberOfSegmentsToAllocate, segmentSize, Duration.ofMillis(Integer.MAX_VALUE));
    }

    public NetworkBufferPool(int numberOfSegmentsToAllocate, int segmentSize, Duration requestSegmentsTimeout) {
        this.totalNumberOfMemorySegments = numberOfSegmentsToAllocate;
        this.memorySegmentSize = segmentSize;
        Preconditions.checkNotNull((Object)requestSegmentsTimeout);
        Preconditions.checkArgument((requestSegmentsTimeout.toMillis() > 0L ? 1 : 0) != 0, (Object)"The timeout for requesting exclusive buffers should be positive.");
        this.requestSegmentsTimeout = requestSegmentsTimeout;
        long sizeInLong = segmentSize;
        try {
            this.availableMemorySegments = new ArrayDeque(numberOfSegmentsToAllocate);
        }
        catch (OutOfMemoryError err) {
            throw new OutOfMemoryError("Could not allocate buffer queue of length " + numberOfSegmentsToAllocate + " - " + err.getMessage());
        }
        try {
            for (int i = 0; i < numberOfSegmentsToAllocate; ++i) {
                this.availableMemorySegments.add(MemorySegmentFactory.allocateUnpooledOffHeapMemory((int)segmentSize, null));
            }
        }
        catch (OutOfMemoryError err) {
            int allocated = this.availableMemorySegments.size();
            this.availableMemorySegments.clear();
            long requiredMb = sizeInLong * (long)numberOfSegmentsToAllocate >> 20;
            long allocatedMb = sizeInLong * (long)allocated >> 20;
            long missingMb = requiredMb - allocatedMb;
            throw new OutOfMemoryError("Could not allocate enough memory segments for NetworkBufferPool (required (Mb): " + requiredMb + ", allocated (Mb): " + allocatedMb + ", missing (Mb): " + missingMb + "). Cause: " + err.getMessage());
        }
        this.availabilityHelper.resetAvailable();
        long allocatedMb = sizeInLong * (long)this.availableMemorySegments.size() >> 20;
        LOG.info("Allocated {} MB for network buffer pool (number of memory segments: {}, bytes per segment: {}).", new Object[]{allocatedMb, this.availableMemorySegments.size(), segmentSize});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public MemorySegment requestMemorySegment() {
        ArrayDeque<MemorySegment> arrayDeque = this.availableMemorySegments;
        synchronized (arrayDeque) {
            return this.internalRequestMemorySegment();
        }
    }

    public List<MemorySegment> requestMemorySegmentsBlocking(int numberOfSegmentsToRequest) throws IOException {
        return this.internalRequestMemorySegments(numberOfSegmentsToRequest);
    }

    public void recycle(MemorySegment segment) {
        this.internalRecycleMemorySegments(Collections.singleton(Preconditions.checkNotNull((Object)segment)));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<MemorySegment> requestMemorySegments(int numberOfSegmentsToRequest) throws IOException {
        Preconditions.checkArgument((numberOfSegmentsToRequest >= 0 ? 1 : 0) != 0, (Object)"Number of buffers to request must be non-negative.");
        Object object = this.factoryLock;
        synchronized (object) {
            if (this.isDestroyed) {
                throw new IllegalStateException("Network buffer pool has already been destroyed.");
            }
            if (numberOfSegmentsToRequest == 0) {
                return Collections.emptyList();
            }
            this.tryRedistributeBuffers(numberOfSegmentsToRequest);
        }
        return this.internalRequestMemorySegments(numberOfSegmentsToRequest);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private List<MemorySegment> internalRequestMemorySegments(int numberOfSegmentsToRequest) throws IOException {
        ArrayList<MemorySegment> segments = new ArrayList<MemorySegment>(numberOfSegmentsToRequest);
        try {
            Deadline deadline = Deadline.fromNow((Duration)this.requestSegmentsTimeout);
            do {
                MemorySegment segment;
                if (this.isDestroyed) {
                    throw new IllegalStateException("Buffer pool is destroyed.");
                }
                ArrayDeque<MemorySegment> arrayDeque = this.availableMemorySegments;
                synchronized (arrayDeque) {
                    segment = this.internalRequestMemorySegment();
                    if (segment == null) {
                        this.availableMemorySegments.wait(2000L);
                    }
                }
                if (segment != null) {
                    segments.add(segment);
                }
                if (segments.size() >= numberOfSegmentsToRequest) return segments;
            } while (deadline.hasTimeLeft());
            throw new IOException(String.format("Timeout triggered when requesting exclusive buffers: %s,  or you may increase the timeout which is %dms by setting the key '%s'.", this.getConfigDescription(), this.requestSegmentsTimeout.toMillis(), NettyShuffleEnvironmentOptions.NETWORK_EXCLUSIVE_BUFFERS_REQUEST_TIMEOUT_MILLISECONDS.key()));
        }
        catch (Throwable e) {
            this.recycleMemorySegments(segments, numberOfSegmentsToRequest);
            ExceptionUtils.rethrowIOException((Throwable)e);
        }
        return segments;
    }

    @Nullable
    private MemorySegment internalRequestMemorySegment() {
        assert (Thread.holdsLock(this.availableMemorySegments));
        MemorySegment segment = this.availableMemorySegments.poll();
        if (this.availableMemorySegments.isEmpty() && segment != null) {
            this.availabilityHelper.resetUnavailable();
        }
        return segment;
    }

    public void recycleMemorySegments(Collection<MemorySegment> segments) {
        this.recycleMemorySegments(segments, segments.size());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void recycleMemorySegments(Collection<MemorySegment> segments, int size) {
        this.internalRecycleMemorySegments(segments);
        Object object = this.factoryLock;
        synchronized (object) {
            this.numTotalRequiredBuffers -= size;
            this.redistributeBuffers();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void internalRecycleMemorySegments(Collection<MemorySegment> segments) {
        CompletableFuture<?> toNotify = null;
        ArrayDeque<MemorySegment> arrayDeque = this.availableMemorySegments;
        synchronized (arrayDeque) {
            if (this.availableMemorySegments.isEmpty() && !segments.isEmpty()) {
                toNotify = this.availabilityHelper.getUnavailableToResetAvailable();
            }
            this.availableMemorySegments.addAll(segments);
            this.availableMemorySegments.notifyAll();
        }
        if (toNotify != null) {
            toNotify.complete(null);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void destroy() {
        ArrayDeque<MemorySegment> arrayDeque = this.factoryLock;
        synchronized (arrayDeque) {
            this.isDestroyed = true;
        }
        arrayDeque = this.availableMemorySegments;
        synchronized (arrayDeque) {
            MemorySegment segment;
            while ((segment = this.availableMemorySegments.poll()) != null) {
                segment.free();
            }
        }
    }

    public boolean isDestroyed() {
        return this.isDestroyed;
    }

    public int getTotalNumberOfMemorySegments() {
        return this.isDestroyed() ? 0 : this.totalNumberOfMemorySegments;
    }

    public long getTotalMemory() {
        return (long)this.getTotalNumberOfMemorySegments() * (long)this.memorySegmentSize;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getNumberOfAvailableMemorySegments() {
        ArrayDeque<MemorySegment> arrayDeque = this.availableMemorySegments;
        synchronized (arrayDeque) {
            return this.availableMemorySegments.size();
        }
    }

    public long getAvailableMemory() {
        return (long)this.getNumberOfAvailableMemorySegments() * (long)this.memorySegmentSize;
    }

    public int getNumberOfUsedMemorySegments() {
        return this.getTotalNumberOfMemorySegments() - this.getNumberOfAvailableMemorySegments();
    }

    public long getUsedMemory() {
        return (long)this.getNumberOfUsedMemorySegments() * (long)this.memorySegmentSize;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getNumberOfRegisteredBufferPools() {
        Object object = this.factoryLock;
        synchronized (object) {
            return this.allBufferPools.size();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int countBuffers() {
        int buffers = 0;
        Object object = this.factoryLock;
        synchronized (object) {
            for (BufferPool bufferPool : this.allBufferPools) {
                buffers += bufferPool.getNumBuffers();
            }
        }
        return buffers;
    }

    @Override
    public CompletableFuture<?> getAvailableFuture() {
        return this.availabilityHelper.getAvailableFuture();
    }

    @Override
    public BufferPool createBufferPool(int numRequiredBuffers, int maxUsedBuffers) throws IOException {
        return this.internalCreateBufferPool(numRequiredBuffers, maxUsedBuffers, 0, Integer.MAX_VALUE);
    }

    @Override
    public BufferPool createBufferPool(int numRequiredBuffers, int maxUsedBuffers, int numSubpartitions, int maxBuffersPerChannel) throws IOException {
        return this.internalCreateBufferPool(numRequiredBuffers, maxUsedBuffers, numSubpartitions, maxBuffersPerChannel);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private BufferPool internalCreateBufferPool(int numRequiredBuffers, int maxUsedBuffers, int numSubpartitions, int maxBuffersPerChannel) throws IOException {
        Object object = this.factoryLock;
        synchronized (object) {
            if (this.isDestroyed) {
                throw new IllegalStateException("Network buffer pool has already been destroyed.");
            }
            if (this.numTotalRequiredBuffers + numRequiredBuffers > this.totalNumberOfMemorySegments) {
                throw new IOException(String.format("Insufficient number of network buffers: required %d, but only %d available. %s.", numRequiredBuffers, this.totalNumberOfMemorySegments - this.numTotalRequiredBuffers, this.getConfigDescription()));
            }
            this.numTotalRequiredBuffers += numRequiredBuffers;
            LocalBufferPool localBufferPool = new LocalBufferPool(this, numRequiredBuffers, maxUsedBuffers, numSubpartitions, maxBuffersPerChannel);
            this.allBufferPools.add(localBufferPool);
            this.redistributeBuffers();
            return localBufferPool;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void destroyBufferPool(BufferPool bufferPool) {
        if (!(bufferPool instanceof LocalBufferPool)) {
            throw new IllegalArgumentException("bufferPool is no LocalBufferPool");
        }
        Object object = this.factoryLock;
        synchronized (object) {
            if (this.allBufferPools.remove(bufferPool)) {
                this.numTotalRequiredBuffers -= bufferPool.getNumberOfRequiredMemorySegments();
                this.redistributeBuffers();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void destroyAllBufferPools() {
        Object object = this.factoryLock;
        synchronized (object) {
            LocalBufferPool[] poolsCopy;
            for (LocalBufferPool pool : poolsCopy = this.allBufferPools.toArray(new LocalBufferPool[this.allBufferPools.size()])) {
                pool.lazyDestroy();
            }
            if (this.allBufferPools.size() > 0 || this.numTotalRequiredBuffers > 0) {
                throw new IllegalStateException("NetworkBufferPool is not empty after destroying all LocalBufferPools");
            }
        }
    }

    private void tryRedistributeBuffers(int numberOfSegmentsToRequest) throws IOException {
        assert (Thread.holdsLock(this.factoryLock));
        if (this.numTotalRequiredBuffers + numberOfSegmentsToRequest > this.totalNumberOfMemorySegments) {
            throw new IOException(String.format("Insufficient number of network buffers: required %d, but only %d available. %s.", numberOfSegmentsToRequest, this.totalNumberOfMemorySegments - this.numTotalRequiredBuffers, this.getConfigDescription()));
        }
        this.numTotalRequiredBuffers += numberOfSegmentsToRequest;
        try {
            this.redistributeBuffers();
        }
        catch (Throwable t) {
            this.numTotalRequiredBuffers -= numberOfSegmentsToRequest;
            this.redistributeBuffers();
            ExceptionUtils.rethrow((Throwable)t);
        }
    }

    private void redistributeBuffers() {
        assert (Thread.holdsLock(this.factoryLock));
        int numAvailableMemorySegment = this.totalNumberOfMemorySegments - this.numTotalRequiredBuffers;
        if (numAvailableMemorySegment == 0) {
            for (LocalBufferPool bufferPool : this.allBufferPools) {
                bufferPool.setNumBuffers(bufferPool.getNumberOfRequiredMemorySegments());
            }
            return;
        }
        long totalCapacity = 0L;
        for (LocalBufferPool bufferPool : this.allBufferPools) {
            int excessMax = bufferPool.getMaxNumberOfMemorySegments() - bufferPool.getNumberOfRequiredMemorySegments();
            totalCapacity += (long)Math.min(numAvailableMemorySegment, excessMax);
        }
        if (totalCapacity == 0L) {
            return;
        }
        int memorySegmentsToDistribute = MathUtils.checkedDownCast((long)Math.min((long)numAvailableMemorySegment, totalCapacity));
        long totalPartsUsed = 0L;
        int numDistributedMemorySegment = 0;
        for (LocalBufferPool bufferPool : this.allBufferPools) {
            int excessMax = bufferPool.getMaxNumberOfMemorySegments() - bufferPool.getNumberOfRequiredMemorySegments();
            if (excessMax == 0) continue;
            int mySize = MathUtils.checkedDownCast((long)((long)memorySegmentsToDistribute * (totalPartsUsed += (long)Math.min(numAvailableMemorySegment, excessMax)) / totalCapacity - (long)numDistributedMemorySegment));
            numDistributedMemorySegment += mySize;
            bufferPool.setNumBuffers(bufferPool.getNumberOfRequiredMemorySegments() + mySize);
        }
        assert (totalPartsUsed == totalCapacity);
        assert (numDistributedMemorySegment == memorySegmentsToDistribute);
    }

    private String getConfigDescription() {
        return String.format("The total number of network buffers is currently set to %d of %d bytes each. You can increase this number by setting the configuration keys '%s', '%s', and '%s'", this.totalNumberOfMemorySegments, this.memorySegmentSize, TaskManagerOptions.NETWORK_MEMORY_FRACTION.key(), TaskManagerOptions.NETWORK_MEMORY_MIN.key(), TaskManagerOptions.NETWORK_MEMORY_MAX.key());
    }
}

