/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.elasticjob.cloud.scheduler.mesos;

import com.google.common.util.concurrent.Service;
import org.apache.mesos.SchedulerDriver;
import org.apache.shardingsphere.elasticjob.cloud.console.ConsoleBootstrap;
import org.apache.shardingsphere.elasticjob.cloud.scheduler.config.app.CloudAppConfigurationListener;
import org.apache.shardingsphere.elasticjob.cloud.scheduler.config.job.CloudJobConfigurationListener;
import org.apache.shardingsphere.elasticjob.cloud.scheduler.env.BootstrapEnvironment;
import org.apache.shardingsphere.elasticjob.cloud.scheduler.env.FrameworkConfiguration;
import org.apache.shardingsphere.elasticjob.cloud.scheduler.producer.ProducerManager;
import org.apache.shardingsphere.elasticjob.cloud.scheduler.state.disable.app.CloudAppDisableListener;
import org.apache.shardingsphere.elasticjob.cloud.scheduler.state.disable.job.CloudJobDisableListener;
import org.apache.shardingsphere.elasticjob.cloud.scheduler.statistics.StatisticManager;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InOrder;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import static org.mockito.Mockito.inOrder;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.when;

@ExtendWith(MockitoExtension.class)
class SchedulerServiceTest {
    
    @Mock
    private BootstrapEnvironment env;
    
    @Mock
    private CloudJobConfigurationListener cloudJobConfigurationListener;
    
    @Mock
    private FacadeService facadeService;
    
    @Mock
    private SchedulerDriver schedulerDriver;
    
    @Mock
    private ProducerManager producerManager;
    
    @Mock
    private StatisticManager statisticManager;
    
    @Mock
    private Service taskLaunchScheduledService;
    
    @Mock
    private ConsoleBootstrap consoleBootstrap;
    
    @Mock
    private ReconcileService reconcileService;
    
    @Mock
    private CloudJobDisableListener cloudJobDisableListener;
    
    @Mock
    private CloudAppConfigurationListener cloudAppConfigurationListener;
    
    @Mock
    private CloudAppDisableListener cloudAppDisableListener;
    
    private SchedulerService schedulerService;
    
    @BeforeEach
    void setUp() {
        schedulerService = new SchedulerService(env, facadeService, schedulerDriver,
                producerManager, statisticManager, cloudJobConfigurationListener,
                taskLaunchScheduledService, consoleBootstrap, reconcileService, cloudJobDisableListener,
                cloudAppConfigurationListener, cloudAppDisableListener);
    }
    
    @Test
    void assertStart() {
        setReconcileEnabled(true);
        schedulerService.start();
        InOrder inOrder = getInOrder();
        inOrder.verify(facadeService).start();
        inOrder.verify(producerManager).startup();
        inOrder.verify(statisticManager).startup();
        inOrder.verify(cloudJobConfigurationListener).start();
        inOrder.verify(taskLaunchScheduledService).startAsync();
        inOrder.verify(consoleBootstrap).start();
        inOrder.verify(schedulerDriver).start();
        inOrder.verify(reconcileService).startAsync();
    }
    
    @Test
    void assertStartWithoutReconcile() {
        setReconcileEnabled(false);
        schedulerService.start();
        InOrder inOrder = getInOrder();
        inOrder.verify(facadeService).start();
        inOrder.verify(producerManager).startup();
        inOrder.verify(statisticManager).startup();
        inOrder.verify(cloudJobConfigurationListener).start();
        inOrder.verify(taskLaunchScheduledService).startAsync();
        inOrder.verify(consoleBootstrap).start();
        inOrder.verify(schedulerDriver).start();
        inOrder.verify(reconcileService, never()).stopAsync();
    }
    
    @Test
    void assertStop() {
        setReconcileEnabled(true);
        schedulerService.stop();
        InOrder inOrder = getInOrder();
        inOrder.verify(consoleBootstrap).stop();
        inOrder.verify(taskLaunchScheduledService).stopAsync();
        inOrder.verify(cloudJobConfigurationListener).stop();
        inOrder.verify(statisticManager).shutdown();
        inOrder.verify(producerManager).shutdown();
        inOrder.verify(schedulerDriver).stop(true);
        inOrder.verify(facadeService).stop();
        inOrder.verify(reconcileService).stopAsync();
    }
    
    @Test
    void assertStopWithoutReconcile() {
        setReconcileEnabled(false);
        schedulerService.stop();
        InOrder inOrder = getInOrder();
        inOrder.verify(consoleBootstrap).stop();
        inOrder.verify(taskLaunchScheduledService).stopAsync();
        inOrder.verify(cloudJobConfigurationListener).stop();
        inOrder.verify(statisticManager).shutdown();
        inOrder.verify(producerManager).shutdown();
        inOrder.verify(schedulerDriver).stop(true);
        inOrder.verify(facadeService).stop();
        inOrder.verify(reconcileService, never()).stopAsync();
    }
    
    private InOrder getInOrder() {
        return inOrder(facadeService, schedulerDriver, producerManager,
                statisticManager, cloudJobConfigurationListener, taskLaunchScheduledService, consoleBootstrap, reconcileService);
    }
    
    private void setReconcileEnabled(final boolean isEnabled) {
        FrameworkConfiguration frameworkConfiguration = mock(FrameworkConfiguration.class);
        when(frameworkConfiguration.isEnabledReconcile()).thenReturn(isEnabled);
        when(env.getFrameworkConfiguration()).thenReturn(frameworkConfiguration);
    }
}
