// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.
//! Euclidean DistanceExt generic implementation
//!
//! Ported (and contains copied code) from `geo::algorithm::line_measures::metric_spaces::euclidean::distance`:
//! <https://github.com/georust/geo/blob/5d667f844716a3d0a17aa60bc0a58528cb5808c3/geo/src/algorithm/line_measures/metric_spaces/euclidean/distance.rs>.
//! Original code is dual-licensed under Apache-2.0 or MIT; used here under Apache-2.0.
use crate::{CoordFloat, GeoFloat, Point};
use num_traits::{Bounded, Float};
use std::borrow::Borrow;

// Import all the utility functions from utils module
use super::utils::{
    distance_coord_to_line_generic,
    // Symmetric distance functions generated by macro
    distance_line_to_line_generic,
    distance_line_to_linestring_generic,
    distance_line_to_polygon_generic,
    distance_linestring_to_polygon_generic,
    distance_point_to_linestring_generic,
    distance_point_to_point_generic,
    distance_point_to_polygon_generic,
    distance_polygon_to_polygon_generic,
};

// ┌──────────────────────────────────────────────────────────┐
// │ Generic Trait Distance Extension                         │
// └──────────────────────────────────────────────────────────┘

use sedona_geo_traits_ext::*;

/// Extension trait for generic geometry types to calculate distances directly
/// using Euclidean metric space without conversion overhead
/// Supports both same-type and cross-type distance calculations
pub trait DistanceExt<F: CoordFloat, Rhs = Self> {
    /// Calculate Euclidean distance using generic traits without conversion overhead
    fn distance_ext(&self, other: &Rhs) -> F;
}

// ┌──────────────────────────────────────────────────────────┐
// │ Generic trait macro implementations                      │
// └──────────────────────────────────────────────────────────┘

/// Generic trait version of polygon-like geometry distance implementation
/// Follows the same pattern as impl_euclidean_distance_for_polygonlike_geometry!
macro_rules! impl_distance_ext_for_polygonlike_geometry_trait {
    ($polygonlike_trait:ident, $polygonlike_tag:ident, [$(($geometry_trait:ident, $geometry_tag:ident)),*]) => {
        // Self-to-self distance implementation
        impl<F, LHS, RHS> GenericDistanceTrait<F, $polygonlike_tag, $polygonlike_tag, RHS> for LHS
        where
            F: GeoFloat,
            LHS: $polygonlike_trait<T = F>,
            RHS: $polygonlike_trait<T = F>,
        {
            fn generic_distance_trait(&self, rhs: &RHS) -> F {
                let poly1 = self.to_polygon();
                let poly2 = rhs.to_polygon();
                distance_polygon_to_polygon_generic(&poly1, &poly2)
            }
        }
    };
}

// Separate macro to generate individual implementations for each geometry type
macro_rules! impl_polygonlike_to_geometry_distance {
    ($polygonlike_trait:ident, $polygonlike_tag:ident, $geometry_trait:ident, $geometry_tag:ident) => {
        impl<F, PL, G> GenericDistanceTrait<F, $polygonlike_tag, $geometry_tag, G> for PL
        where
            F: GeoFloat,
            PL: $polygonlike_trait<T = F>,
            G: $geometry_trait<T = F>,
        {
            fn generic_distance_trait(&self, rhs: &G) -> F {
                let poly = self.to_polygon();
                impl_polygonlike_to_geometry_distance!(@call_distance poly, rhs, $geometry_tag)
            }
        }
    };

    (@call_distance $poly:expr, $rhs:expr, PointTag) => {
        distance_point_to_polygon_generic($rhs, &$poly)
    };
    (@call_distance $poly:expr, $rhs:expr, LineTag) => {
        distance_line_to_polygon_generic($rhs, &$poly)
    };
    (@call_distance $poly:expr, $rhs:expr, LineStringTag) => {
        distance_linestring_to_polygon_generic($rhs, &$poly)
    };
    (@call_distance $poly:expr, $rhs:expr, PolygonTag) => {
        distance_polygon_to_polygon_generic(&$poly, $rhs)
    };
    (@call_distance $poly:expr, $rhs:expr, RectTag) => {
        {
            let poly2 = $rhs.to_polygon();
            distance_polygon_to_polygon_generic(&$poly, &poly2)
        }
    };
    (@call_distance $poly:expr, $rhs:expr, TriangleTag) => {
        {
            let poly2 = $rhs.to_polygon();
            distance_polygon_to_polygon_generic(&$poly, &poly2)
        }
    };
    (@call_distance $poly:expr, $rhs:expr, MultiPointTag) => {
        {
            let mut min_dist: F = Bounded::max_value();
            for coord in $rhs.coord_iter() {
                let point = Point::from(coord);
                let dist = distance_point_to_polygon_generic(&point, &$poly);
                min_dist = min_dist.min(dist);
            }
            if min_dist == Bounded::max_value() {
                F::zero()
            } else {
                min_dist
            }
        }
    };
    (@call_distance $poly:expr, $rhs:expr, MultiLineStringTag) => {
        {
            let mut min_dist: F = Bounded::max_value();
            for line_string in $rhs.line_strings_ext() {
                let dist = distance_linestring_to_polygon_generic(&line_string, &$poly);
                min_dist = min_dist.min(dist);
            }
            if min_dist == Bounded::max_value() {
                F::zero()
            } else {
                min_dist
            }
        }
    };
    (@call_distance $poly:expr, $rhs:expr, MultiPolygonTag) => {
        {
            let mut min_dist: F = Bounded::max_value();
            for polygon in $rhs.polygons_ext() {
                let dist = distance_polygon_to_polygon_generic(&$poly, &polygon);
                min_dist = min_dist.min(dist);
            }
            if min_dist == Bounded::max_value() {
                F::zero()
            } else {
                min_dist
            }
        }
    };
}

/// Generic trait version of multi-geometry distance implementation
/// Follows the same pattern as impl_euclidean_distance_for_iter_geometry!
macro_rules! impl_distance_ext_for_iter_geometry_trait {
    ($iter_trait:ident, $iter_tag:ident, $member_method:ident) => {
        impl<F, LHS, RHS> GenericDistanceTrait<F, $iter_tag, $iter_tag, RHS> for LHS
        where
            F: GeoFloat,
            LHS: $iter_trait<T = F>,
            RHS: $iter_trait<T = F>,
        {
            fn generic_distance_trait(&self, rhs: &RHS) -> F {
                let mut min_dist: F = Float::max_value();
                for member1 in self.$member_method() {
                    for member2 in rhs.$member_method() {
                        let dist = member1.distance_ext(&member2);
                        min_dist = min_dist.min(dist);
                    }
                }
                if min_dist == Float::max_value() {
                    F::zero()
                } else {
                    min_dist
                }
            }
        }
    };
}

// Array-based macro for systematic implementation generation
macro_rules! impl_cross_type_array {
    // Generate multi-geometry self-implementations
    (self_multi_geometries: [$(($trait:ident, $tag:ident, $method:ident)),+]) => {
        $(
            impl_distance_ext_for_iter_geometry_trait!($trait, $tag, $method);
        )+
    };

    // Generate single-to-multi implementations with Point to MultiPoint skip
    (single_to_multi: $single_trait:ident, $single_tag:ident => [$(($multi_trait:ident, $multi_tag:ident, $method:ident)),+]) => {
        $(
            impl_cross_type_array!(@single_to_multi_check $single_trait, $single_tag, $multi_trait, $multi_tag, $method);
        )+
    };

    // Skip Point to MultiPoint (special implementation exists)
    (@single_to_multi_check PointTraitExt, PointTag, MultiPointTraitExt, MultiPointTag, $method:ident) => {};

    // Generate for all other combinations
    (@single_to_multi_check $single_trait:ident, $single_tag:ident, $multi_trait:ident, $multi_tag:ident, $method:ident) => {
        impl_single_to_multi_geometry_distance!($single_trait, $single_tag, $multi_trait, $multi_tag, $method);
    };

    // Generate symmetric implementations for single-to-multi
    (symmetric_single_to_multi: $single_trait:ident, $single_tag:ident => [$(($multi_trait:ident, $multi_tag:ident)),+]) => {
        $(
            symmetric_distance_ext_trait_impl!(GeoFloat, $multi_trait, $multi_tag, $single_trait, $single_tag);
        )+
    };
}

/// Macro for implementing cross-type distance calculations from single geometry to multi-geometry types
macro_rules! impl_single_to_multi_geometry_distance {
    ($single_trait:ident, $single_tag:ident, $multi_trait:ident, $multi_tag:ident, $member_method:ident) => {
        impl<F, S, M> GenericDistanceTrait<F, $single_tag, $multi_tag, M> for S
        where
            F: GeoFloat,
            S: $single_trait<T = F>,
            M: $multi_trait<T = F>,
        {
            fn generic_distance_trait(&self, rhs: &M) -> F {
                let mut min_dist: F = Bounded::max_value();
                for member in rhs.$member_method() {
                    let dist = self.distance_ext(&member);
                    min_dist = min_dist.min(dist);
                }
                if min_dist == Bounded::max_value() {
                    F::zero()
                } else {
                    min_dist
                }
            }
        }
    };
}

// Implementation of DistanceExt for cross-type generic trait geometries using the two type-tag pattern
impl<F, LHS, RHS> DistanceExt<F, RHS> for LHS
where
    F: GeoFloat,
    LHS: GeoTraitExtWithTypeTag,
    RHS: GeoTraitExtWithTypeTag,
    LHS: GenericDistanceTrait<F, LHS::Tag, RHS::Tag, RHS>,
{
    fn distance_ext(&self, other: &RHS) -> F {
        self.generic_distance_trait(other)
    }
}

// ┌────────────────────────────────────────────────────────────┐
// │ Internal trait for cross-type distance calculations       │
// └────────────────────────────────────────────────────────────┘

// Internal trait for cross-type distance calculations without conversion
trait GenericDistanceTrait<F, LhsTag: GeoTypeTag, RhsTag: GeoTypeTag, Rhs = Self>
where
    F: GeoFloat,
{
    fn generic_distance_trait(&self, rhs: &Rhs) -> F;
}
macro_rules! symmetric_distance_ext_trait_impl {
    ($num_type:ident, $lhs_type:ident, $lhs_tag:ident, $rhs_type:ident, $rhs_tag:ident) => {
        impl<F, LHS, RHS> GenericDistanceTrait<F, $lhs_tag, $rhs_tag, RHS> for LHS
        where
            F: $num_type,
            LHS: $lhs_type<T = F>,
            RHS: $rhs_type<T = F>,
        {
            fn generic_distance_trait(&self, rhs: &RHS) -> F {
                rhs.generic_distance_trait(self)
            }
        }
    };
}

// ┌────────────────────────────────────────────────────────────┐
// │ Implementations for Coord  (generic traits)                │
// └────────────────────────────────────────────────────────────┘

// Coord-to-Coord direct distance implementation
impl<F, LHS, RHS> GenericDistanceTrait<F, CoordTag, CoordTag, RHS> for LHS
where
    F: GeoFloat,
    LHS: CoordTraitExt<T = F>,
    RHS: CoordTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &RHS) -> F {
        let delta = self.geo_coord() - rhs.geo_coord();
        delta.x.hypot(delta.y)
    }
}

// Coord-to-Point distance implementation
// The other side (Point-to-Coord) is handled via a symmetric impl or blanket impl
impl<F, LHS, RHS> GenericDistanceTrait<F, CoordTag, PointTag, RHS> for LHS
where
    F: GeoFloat,
    LHS: CoordTraitExt<T = F>,
    RHS: PointTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &RHS) -> F {
        if let Some(rhs_coord) = rhs.coord_ext() {
            let delta = self.geo_coord() - rhs_coord.geo_coord();
            delta.x.hypot(delta.y)
        } else {
            F::zero()
        }
    }
}

// Coord-to-Line distance implementation
impl<F, LHS, RHS> GenericDistanceTrait<F, CoordTag, LineTag, RHS> for LHS
where
    F: GeoFloat,
    LHS: CoordTraitExt<T = F>,
    RHS: LineTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &RHS) -> F {
        distance_coord_to_line_generic(self, rhs)
    }
}

// ┌────────────────────────────────────────────────────────────┐
// │ Implementations for Point (generic traits)                 │
// └────────────────────────────────────────────────────────────┘

// Point-to-Point direct distance implementation
impl<F, LHS, RHS> GenericDistanceTrait<F, PointTag, PointTag, RHS> for LHS
where
    F: GeoFloat,
    LHS: PointTraitExt<T = F>,
    RHS: PointTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &RHS) -> F {
        distance_point_to_point_generic(self, rhs)
    }
}

// Point-to-Line distance implementation
impl<F, P, L> GenericDistanceTrait<F, PointTag, LineTag, L> for P
where
    F: GeoFloat,
    P: PointTraitExt<T = F>,
    L: LineTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &L) -> F {
        if let Some(coord) = self.coord_ext() {
            distance_coord_to_line_generic(&coord, rhs)
        } else {
            F::zero()
        }
    }
}

// Point-to-LineString distance implementation
impl<F, P, LS> GenericDistanceTrait<F, PointTag, LineStringTag, LS> for P
where
    F: GeoFloat,
    P: PointTraitExt<T = F>,
    LS: LineStringTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &LS) -> F {
        distance_point_to_linestring_generic(self, rhs)
    }
}

// Point-to-Polygon distance implementation
impl<F, P, Poly> GenericDistanceTrait<F, PointTag, PolygonTag, Poly> for P
where
    F: GeoFloat,
    P: PointTraitExt<T = F>,
    Poly: PolygonTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &Poly) -> F {
        distance_point_to_polygon_generic(self, rhs)
    }
}

// Point to MultiPoint distance implementation
impl<F, P, MP> GenericDistanceTrait<F, PointTag, MultiPointTag, MP> for P
where
    F: GeoFloat,
    P: PointTraitExt<T = F>,
    MP: MultiPointTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &MP) -> F {
        if let Some(point_coord) = self.geo_coord() {
            let mut min_dist: F = Bounded::max_value();
            for coord in rhs.coord_iter() {
                let dist = point_coord.distance_ext(&coord);
                min_dist = min_dist.min(dist);
            }
            if min_dist == Bounded::max_value() {
                F::zero()
            } else {
                min_dist
            }
        } else {
            F::zero()
        }
    }
}

// ┌────────────────────────────────────────────────────────────┐
// │ Implementations for Line (generic traits)                  │
// └────────────────────────────────────────────────────────────┘

// Symmetric Line distance implementations
symmetric_distance_ext_trait_impl!(GeoFloat, LineTraitExt, LineTag, CoordTraitExt, CoordTag);
symmetric_distance_ext_trait_impl!(GeoFloat, LineTraitExt, LineTag, PointTraitExt, PointTag);

// Line-to-Line direct distance implementation
impl<F, LHS, RHS> GenericDistanceTrait<F, LineTag, LineTag, RHS> for LHS
where
    F: GeoFloat,
    LHS: LineTraitExt<T = F>,
    RHS: LineTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &RHS) -> F {
        distance_line_to_line_generic(self, rhs)
    }
}

// Line-to-LineString distance implementation
impl<F, L, LS> GenericDistanceTrait<F, LineTag, LineStringTag, LS> for L
where
    F: GeoFloat,
    L: LineTraitExt<T = F>,
    LS: LineStringTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &LS) -> F {
        distance_line_to_linestring_generic(self, rhs)
    }
}

// Line-to-Polygon distance implementation
impl<F, L, Poly> GenericDistanceTrait<F, LineTag, PolygonTag, Poly> for L
where
    F: GeoFloat,
    L: LineTraitExt<T = F>,
    Poly: PolygonTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &Poly) -> F {
        distance_line_to_polygon_generic(self, rhs)
    }
}

// ┌────────────────────────────────────────────────────────────┐
// │ Implementations for LineString (generic traits)            │
// └────────────────────────────────────────────────────────────┘

// Symmetric LineString distance implementations
// LineString-to-Point (symmetric to Point-to-LineString)
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    LineStringTraitExt,
    LineStringTag,
    PointTraitExt,
    PointTag
);

// LineString-to-Line (symmetric to Line-to-LineString)
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    LineStringTraitExt,
    LineStringTag,
    LineTraitExt,
    LineTag
);

// LineString-to-LineString distance implementation
// This general implementation supports both same-type (LS to LS) and different-type (LS1 to LS2)
impl<F, LS1, LS2> GenericDistanceTrait<F, LineStringTag, LineStringTag, LS2> for LS1
where
    F: GeoFloat,
    LS1: LineStringTraitExt<T = F>,
    LS2: LineStringTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &LS2) -> F {
        let mut min_dist: F = Float::max_value();
        for line1 in self.lines() {
            for line2 in rhs.lines() {
                // Line-to-line distance using endpoints
                let d1 = distance_coord_to_line_generic(&line1.start_coord(), &line2);
                let d2 = distance_coord_to_line_generic(&line1.end_coord(), &line2);
                let d3 = distance_coord_to_line_generic(&line2.start_coord(), &line1);
                let d4 = distance_coord_to_line_generic(&line2.end_coord(), &line1);
                let line_dist = d1.min(d2).min(d3).min(d4);
                min_dist = min_dist.min(line_dist);
            }
        }
        if min_dist == Float::max_value() {
            F::zero()
        } else {
            min_dist
        }
    }
}

// LineString-to-Polygon distance implementation
impl<F, LS, Poly> GenericDistanceTrait<F, LineStringTag, PolygonTag, Poly> for LS
where
    F: GeoFloat,
    LS: LineStringTraitExt<T = F>,
    Poly: PolygonTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &Poly) -> F {
        distance_linestring_to_polygon_generic(self, rhs)
    }
}

// ┌────────────────────────────────────────────────────────────┐
// │ Implementations for Polygon (generic traits)              │
// └────────────────────────────────────────────────────────────┘

// Symmetric Polygon distance implementations
// Polygon-to-Point (symmetric to Point-to-Polygon)
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    PolygonTraitExt,
    PolygonTag,
    PointTraitExt,
    PointTag
);

// Polygon-to-Line (symmetric to Line-to-Polygon)
symmetric_distance_ext_trait_impl!(GeoFloat, PolygonTraitExt, PolygonTag, LineTraitExt, LineTag);

// Polygon-to-LineString (symmetric to LineString-to-Polygon)
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    PolygonTraitExt,
    PolygonTag,
    LineStringTraitExt,
    LineStringTag
);

// Polygon-to-Polygon distance implementation
// This general implementation supports both same-type (P to P) and different-type (P1 to P2)
impl<F, P1, P2> GenericDistanceTrait<F, PolygonTag, PolygonTag, P2> for P1
where
    F: GeoFloat,
    P1: PolygonTraitExt<T = F>,
    P2: PolygonTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &P2) -> F {
        distance_polygon_to_polygon_generic(self, rhs)
    }
}

// ┌────────────────────────────────────────────────────────────┐
// │ Implementations for Rect and Triangle (generic traits)     │
// └────────────────────────────────────────────────────────────┘

// Triangle implementations
impl_distance_ext_for_polygonlike_geometry_trait!(TriangleTraitExt, TriangleTag, []);
impl_polygonlike_to_geometry_distance!(TriangleTraitExt, TriangleTag, PointTraitExt, PointTag);
impl_polygonlike_to_geometry_distance!(TriangleTraitExt, TriangleTag, LineTraitExt, LineTag);
impl_polygonlike_to_geometry_distance!(
    TriangleTraitExt,
    TriangleTag,
    LineStringTraitExt,
    LineStringTag
);
impl_polygonlike_to_geometry_distance!(TriangleTraitExt, TriangleTag, PolygonTraitExt, PolygonTag);
impl_polygonlike_to_geometry_distance!(TriangleTraitExt, TriangleTag, RectTraitExt, RectTag);
impl_polygonlike_to_geometry_distance!(
    TriangleTraitExt,
    TriangleTag,
    MultiPointTraitExt,
    MultiPointTag
);
impl_polygonlike_to_geometry_distance!(
    TriangleTraitExt,
    TriangleTag,
    MultiLineStringTraitExt,
    MultiLineStringTag
);
impl_polygonlike_to_geometry_distance!(
    TriangleTraitExt,
    TriangleTag,
    MultiPolygonTraitExt,
    MultiPolygonTag
);

// Symmetric implementations for Triangle
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    PointTraitExt,
    PointTag,
    TriangleTraitExt,
    TriangleTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    LineTraitExt,
    LineTag,
    TriangleTraitExt,
    TriangleTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    LineStringTraitExt,
    LineStringTag,
    TriangleTraitExt,
    TriangleTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    PolygonTraitExt,
    PolygonTag,
    TriangleTraitExt,
    TriangleTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    RectTraitExt,
    RectTag,
    TriangleTraitExt,
    TriangleTag
);

// Rect implementations
impl_distance_ext_for_polygonlike_geometry_trait!(RectTraitExt, RectTag, []);
impl_polygonlike_to_geometry_distance!(RectTraitExt, RectTag, PointTraitExt, PointTag);
impl_polygonlike_to_geometry_distance!(RectTraitExt, RectTag, LineTraitExt, LineTag);
impl_polygonlike_to_geometry_distance!(RectTraitExt, RectTag, LineStringTraitExt, LineStringTag);
impl_polygonlike_to_geometry_distance!(RectTraitExt, RectTag, PolygonTraitExt, PolygonTag);
impl_polygonlike_to_geometry_distance!(RectTraitExt, RectTag, MultiPointTraitExt, MultiPointTag);
impl_polygonlike_to_geometry_distance!(
    RectTraitExt,
    RectTag,
    MultiLineStringTraitExt,
    MultiLineStringTag
);
impl_polygonlike_to_geometry_distance!(
    RectTraitExt,
    RectTag,
    MultiPolygonTraitExt,
    MultiPolygonTag
);

// Symmetric implementations for Rect (excluding Triangle which is already handled above)
symmetric_distance_ext_trait_impl!(GeoFloat, PointTraitExt, PointTag, RectTraitExt, RectTag);
symmetric_distance_ext_trait_impl!(GeoFloat, LineTraitExt, LineTag, RectTraitExt, RectTag);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    LineStringTraitExt,
    LineStringTag,
    RectTraitExt,
    RectTag
);
symmetric_distance_ext_trait_impl!(GeoFloat, PolygonTraitExt, PolygonTag, RectTraitExt, RectTag);

// ┌────────────────────────────────────────────────────────────┐
// │ Implementations for multi-geometry types (generic traits)  │
// └────────────────────────────────────────────────────────────┘

// Multi-geometry self-implementations
impl_cross_type_array!(self_multi_geometries: [
    (MultiPointTraitExt, MultiPointTag, points_ext),
    (MultiLineStringTraitExt, MultiLineStringTag, line_strings_ext),
    (MultiPolygonTraitExt, MultiPolygonTag, polygons_ext)
]);

// Single-geometry to multi-geometry implementations
impl_cross_type_array!(single_to_multi: PointTraitExt, PointTag => [
    (MultiPointTraitExt, MultiPointTag, points_ext),
    (MultiLineStringTraitExt, MultiLineStringTag, line_strings_ext),
    (MultiPolygonTraitExt, MultiPolygonTag, polygons_ext)
]);

impl_cross_type_array!(single_to_multi: LineTraitExt, LineTag => [
    (MultiPointTraitExt, MultiPointTag, points_ext),
    (MultiLineStringTraitExt, MultiLineStringTag, line_strings_ext),
    (MultiPolygonTraitExt, MultiPolygonTag, polygons_ext)
]);

impl_cross_type_array!(single_to_multi: LineStringTraitExt, LineStringTag => [
    (MultiPointTraitExt, MultiPointTag, points_ext),
    (MultiLineStringTraitExt, MultiLineStringTag, line_strings_ext),
    (MultiPolygonTraitExt, MultiPolygonTag, polygons_ext)
]);

impl_cross_type_array!(single_to_multi: PolygonTraitExt, PolygonTag => [
    (MultiPointTraitExt, MultiPointTag, points_ext),
    (MultiLineStringTraitExt, MultiLineStringTag, line_strings_ext),
    (MultiPolygonTraitExt, MultiPolygonTag, polygons_ext)
]);

// Multi-geometry to multi-geometry implementations
impl_single_to_multi_geometry_distance!(
    MultiPointTraitExt,
    MultiPointTag,
    MultiLineStringTraitExt,
    MultiLineStringTag,
    line_strings_ext
);
impl_single_to_multi_geometry_distance!(
    MultiPointTraitExt,
    MultiPointTag,
    MultiPolygonTraitExt,
    MultiPolygonTag,
    polygons_ext
);
impl_single_to_multi_geometry_distance!(
    MultiLineStringTraitExt,
    MultiLineStringTag,
    MultiPointTraitExt,
    MultiPointTag,
    points_ext
);
impl_single_to_multi_geometry_distance!(
    MultiLineStringTraitExt,
    MultiLineStringTag,
    MultiPolygonTraitExt,
    MultiPolygonTag,
    polygons_ext
);
impl_single_to_multi_geometry_distance!(
    MultiPolygonTraitExt,
    MultiPolygonTag,
    MultiPointTraitExt,
    MultiPointTag,
    points_ext
);
impl_single_to_multi_geometry_distance!(
    MultiPolygonTraitExt,
    MultiPolygonTag,
    MultiLineStringTraitExt,
    MultiLineStringTag,
    line_strings_ext
);

// Symmetric implementations
impl_cross_type_array!(symmetric_single_to_multi: PointTraitExt, PointTag => [
    (MultiPointTraitExt, MultiPointTag),
    (MultiLineStringTraitExt, MultiLineStringTag),
    (MultiPolygonTraitExt, MultiPolygonTag)
]);

impl_cross_type_array!(symmetric_single_to_multi: LineTraitExt, LineTag => [
    (MultiPointTraitExt, MultiPointTag),
    (MultiLineStringTraitExt, MultiLineStringTag),
    (MultiPolygonTraitExt, MultiPolygonTag)
]);

impl_cross_type_array!(symmetric_single_to_multi: LineStringTraitExt, LineStringTag => [
    (MultiPointTraitExt, MultiPointTag),
    (MultiLineStringTraitExt, MultiLineStringTag),
    (MultiPolygonTraitExt, MultiPolygonTag)
]);

impl_cross_type_array!(symmetric_single_to_multi: PolygonTraitExt, PolygonTag => [
    (MultiPointTraitExt, MultiPointTag),
    (MultiLineStringTraitExt, MultiLineStringTag),
    (MultiPolygonTraitExt, MultiPolygonTag)
]);

impl_cross_type_array!(symmetric_single_to_multi: RectTraitExt, RectTag => [
    (MultiPointTraitExt, MultiPointTag),
    (MultiLineStringTraitExt, MultiLineStringTag),
    (MultiPolygonTraitExt, MultiPolygonTag)
]);

impl_cross_type_array!(symmetric_single_to_multi: TriangleTraitExt, TriangleTag => [
    (MultiPointTraitExt, MultiPointTag),
    (MultiLineStringTraitExt, MultiLineStringTag),
    (MultiPolygonTraitExt, MultiPolygonTag)
]);

// ┌────────────────────────────────────────────────────────────┐
// │ Implementation for GeometryCollection (generic traits)     │
// └────────────────────────────────────────────────────────────┘

// Generate implementations for GeometryCollection by delegating to the Geometry implementation
macro_rules! impl_distance_geometry_collection_from_geometry {
    ($rhs_type:ident, $rhs_tag:ident) => {
        impl<F, LHS, RHS> GenericDistanceTrait<F, GeometryCollectionTag, $rhs_tag, RHS> for LHS
        where
            F: GeoFloat,
            LHS: GeometryCollectionTraitExt<T = F>,
            RHS: $rhs_type<T = F>,
        {
            fn generic_distance_trait(&self, rhs: &RHS) -> F {
                use num_traits::Bounded;

                // Use distance_ext which will route through the appropriate implementations
                // The key insight is that this works for all geometry types except GeometryCollection,
                // where we need special handling to avoid infinite recursion
                self.geometries_ext()
                    .map(|geom| geom.distance_ext(rhs))
                    .fold(Bounded::max_value(), |acc, dist| acc.min(dist))
            }
        }
    };
}

impl_distance_geometry_collection_from_geometry!(PointTraitExt, PointTag);
impl_distance_geometry_collection_from_geometry!(LineTraitExt, LineTag);
impl_distance_geometry_collection_from_geometry!(LineStringTraitExt, LineStringTag);
impl_distance_geometry_collection_from_geometry!(PolygonTraitExt, PolygonTag);
impl_distance_geometry_collection_from_geometry!(MultiPointTraitExt, MultiPointTag);
impl_distance_geometry_collection_from_geometry!(MultiLineStringTraitExt, MultiLineStringTag);
impl_distance_geometry_collection_from_geometry!(MultiPolygonTraitExt, MultiPolygonTag);
impl_distance_geometry_collection_from_geometry!(RectTraitExt, RectTag);
impl_distance_geometry_collection_from_geometry!(TriangleTraitExt, TriangleTag);

impl<F, LHS, RHS> GenericDistanceTrait<F, GeometryCollectionTag, GeometryCollectionTag, RHS> for LHS
where
    F: GeoFloat,
    LHS: GeometryCollectionTraitExt<T = F>,
    RHS: GeometryCollectionTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &RHS) -> F {
        let mut min_distance = <F as Bounded>::max_value();
        for lhs_geom in self.geometries_ext() {
            for rhs_geom in rhs.geometries_ext() {
                let distance = lhs_geom.distance_ext(&rhs_geom);
                min_distance = min_distance.min(distance);

                // Early exit optimization
                if distance == F::zero() {
                    return F::zero();
                }
            }
        }

        min_distance
    }
}

symmetric_distance_ext_trait_impl!(
    GeoFloat,
    PointTraitExt,
    PointTag,
    GeometryCollectionTraitExt,
    GeometryCollectionTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    LineTraitExt,
    LineTag,
    GeometryCollectionTraitExt,
    GeometryCollectionTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    LineStringTraitExt,
    LineStringTag,
    GeometryCollectionTraitExt,
    GeometryCollectionTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    PolygonTraitExt,
    PolygonTag,
    GeometryCollectionTraitExt,
    GeometryCollectionTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    MultiPointTraitExt,
    MultiPointTag,
    GeometryCollectionTraitExt,
    GeometryCollectionTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    MultiLineStringTraitExt,
    MultiLineStringTag,
    GeometryCollectionTraitExt,
    GeometryCollectionTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    MultiPolygonTraitExt,
    MultiPolygonTag,
    GeometryCollectionTraitExt,
    GeometryCollectionTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    RectTraitExt,
    RectTag,
    GeometryCollectionTraitExt,
    GeometryCollectionTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    TriangleTraitExt,
    TriangleTag,
    GeometryCollectionTraitExt,
    GeometryCollectionTag
);

// ┌────────────────────────────────────────────────────────────┐
// │ Implementation for Geometry (generic traits)               │
// └────────────────────────────────────────────────────────────┘

// Generate implementations for Geometry with other types using conversion
macro_rules! impl_distance_geometry_to_type {
    ($rhs_type:ident, $rhs_tag:ident) => {
        impl<F, LHS, RHS> GenericDistanceTrait<F, GeometryTag, $rhs_tag, RHS> for LHS
        where
            F: GeoFloat,
            LHS: GeometryTraitExt<T = F>,
            RHS: $rhs_type<T = F>,
        {
            fn generic_distance_trait(&self, rhs: &RHS) -> F {
                if self.is_collection() {
                    let mut min_distance = <F as Bounded>::max_value();
                    for lhs_geom in self.geometries_ext() {
                        let lhs_geom = lhs_geom.borrow();
                        let distance = lhs_geom.generic_distance_trait(rhs);
                        min_distance = min_distance.min(distance);

                        // Early exit optimization
                        if distance == F::zero() {
                            return F::zero();
                        }
                    }
                    min_distance
                } else {
                    match self.as_type_ext() {
                        sedona_geo_traits_ext::GeometryTypeExt::Point(g) => {
                            g.generic_distance_trait(rhs)
                        }
                        sedona_geo_traits_ext::GeometryTypeExt::Line(g) => {
                            g.generic_distance_trait(rhs)
                        }
                        sedona_geo_traits_ext::GeometryTypeExt::LineString(g) => {
                            g.generic_distance_trait(rhs)
                        }
                        sedona_geo_traits_ext::GeometryTypeExt::Polygon(g) => {
                            g.generic_distance_trait(rhs)
                        }
                        sedona_geo_traits_ext::GeometryTypeExt::MultiPoint(g) => {
                            g.generic_distance_trait(rhs)
                        }
                        sedona_geo_traits_ext::GeometryTypeExt::MultiLineString(g) => {
                            g.generic_distance_trait(rhs)
                        }
                        sedona_geo_traits_ext::GeometryTypeExt::MultiPolygon(g) => {
                            g.generic_distance_trait(rhs)
                        }
                        sedona_geo_traits_ext::GeometryTypeExt::Rect(g) => {
                            g.generic_distance_trait(rhs)
                        }
                        sedona_geo_traits_ext::GeometryTypeExt::Triangle(g) => {
                            g.generic_distance_trait(rhs)
                        }
                    }
                }
            }
        }
    };
}

impl_distance_geometry_to_type!(PointTraitExt, PointTag);
impl_distance_geometry_to_type!(LineTraitExt, LineTag);
impl_distance_geometry_to_type!(LineStringTraitExt, LineStringTag);
impl_distance_geometry_to_type!(PolygonTraitExt, PolygonTag);
impl_distance_geometry_to_type!(MultiPointTraitExt, MultiPointTag);
impl_distance_geometry_to_type!(MultiLineStringTraitExt, MultiLineStringTag);
impl_distance_geometry_to_type!(MultiPolygonTraitExt, MultiPolygonTag);
impl_distance_geometry_to_type!(RectTraitExt, RectTag);
impl_distance_geometry_to_type!(TriangleTraitExt, TriangleTag);
impl_distance_geometry_to_type!(GeometryCollectionTraitExt, GeometryCollectionTag);

symmetric_distance_ext_trait_impl!(
    GeoFloat,
    PointTraitExt,
    PointTag,
    GeometryTraitExt,
    GeometryTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    LineTraitExt,
    LineTag,
    GeometryTraitExt,
    GeometryTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    LineStringTraitExt,
    LineStringTag,
    GeometryTraitExt,
    GeometryTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    PolygonTraitExt,
    PolygonTag,
    GeometryTraitExt,
    GeometryTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    MultiPointTraitExt,
    MultiPointTag,
    GeometryTraitExt,
    GeometryTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    MultiLineStringTraitExt,
    MultiLineStringTag,
    GeometryTraitExt,
    GeometryTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    MultiPolygonTraitExt,
    MultiPolygonTag,
    GeometryTraitExt,
    GeometryTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    RectTraitExt,
    RectTag,
    GeometryTraitExt,
    GeometryTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    TriangleTraitExt,
    TriangleTag,
    GeometryTraitExt,
    GeometryTag
);
symmetric_distance_ext_trait_impl!(
    GeoFloat,
    GeometryCollectionTraitExt,
    GeometryCollectionTag,
    GeometryTraitExt,
    GeometryTag
);

impl<F, LHS, RHS> GenericDistanceTrait<F, GeometryTag, GeometryTag, RHS> for LHS
where
    F: GeoFloat,
    LHS: GeometryTraitExt<T = F>,
    RHS: GeometryTraitExt<T = F>,
{
    fn generic_distance_trait(&self, rhs: &RHS) -> F {
        if self.is_collection() {
            let mut min_distance = <F as Bounded>::max_value();
            for lhs_geom in self.geometries_ext() {
                let lhs_geom = lhs_geom.borrow();
                let distance = lhs_geom.generic_distance_trait(rhs);
                min_distance = min_distance.min(distance);

                // Early exit optimization
                if distance == F::zero() {
                    return F::zero();
                }
            }
            min_distance
        } else {
            match self.as_type_ext() {
                sedona_geo_traits_ext::GeometryTypeExt::Point(g) => g.generic_distance_trait(rhs),
                sedona_geo_traits_ext::GeometryTypeExt::Line(g) => g.generic_distance_trait(rhs),
                sedona_geo_traits_ext::GeometryTypeExt::LineString(g) => {
                    g.generic_distance_trait(rhs)
                }
                sedona_geo_traits_ext::GeometryTypeExt::Polygon(g) => g.generic_distance_trait(rhs),
                sedona_geo_traits_ext::GeometryTypeExt::MultiPoint(g) => {
                    g.generic_distance_trait(rhs)
                }
                sedona_geo_traits_ext::GeometryTypeExt::MultiLineString(g) => {
                    g.generic_distance_trait(rhs)
                }
                sedona_geo_traits_ext::GeometryTypeExt::MultiPolygon(g) => {
                    g.generic_distance_trait(rhs)
                }
                sedona_geo_traits_ext::GeometryTypeExt::Rect(g) => g.generic_distance_trait(rhs),
                sedona_geo_traits_ext::GeometryTypeExt::Triangle(g) => {
                    g.generic_distance_trait(rhs)
                }
            }
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::{Line, LineString, MultiLineString, MultiPoint, MultiPolygon, Point, Polygon};
    use geo::orient::{Direction, Orient};
    use geo_types::{coord, polygon, private_utils::line_segment_distance};

    mod distance_cross_validation_tests {
        use geo::{Coord, Distance, Euclidean, Geometry, GeometryCollection, Rect, Triangle};

        use super::*;

        #[test]
        fn line_segment_distance_test() {
            let o1 = Point::new(8.0, 0.0);
            let o2 = Point::new(5.5, 0.0);
            let o3 = Point::new(5.0, 0.0);
            let o4 = Point::new(4.5, 1.5);

            let p1 = Point::new(7.2, 2.0);
            let p2 = Point::new(6.0, 1.0);

            // Test original implementation
            let dist = line_segment_distance(o1, p1, p2);
            let dist2 = line_segment_distance(o2, p1, p2);
            let dist3 = line_segment_distance(o3, p1, p2);
            let dist4 = line_segment_distance(o4, p1, p2);
            // Results agree with Shapely
            assert_relative_eq!(dist, 2.0485900789263356);
            assert_relative_eq!(dist2, 1.118033988749895);
            assert_relative_eq!(dist3, std::f64::consts::SQRT_2); // workaround clippy::correctness error approx_constant (1.4142135623730951)
            assert_relative_eq!(dist4, 1.5811388300841898);
            // Point is on the line
            let zero_dist = line_segment_distance(p1, p1, p2);
            assert_relative_eq!(zero_dist, 0.0);

            // Test generic implementation
            if let (Some(p1_coord), Some(p2_coord)) = (p1.coord_ext(), p2.coord_ext()) {
                let line_seg = Line::new(p1_coord, p2_coord);

                if let Some(o1_coord) = o1.coord_ext() {
                    let generic_dist = distance_coord_to_line_generic(&o1_coord, &line_seg);
                    assert_relative_eq!(generic_dist, 2.0485900789263356);
                    assert_relative_eq!(dist, generic_dist);
                }
                if let Some(o2_coord) = o2.coord_ext() {
                    let generic_dist2 = distance_coord_to_line_generic(&o2_coord, &line_seg);
                    assert_relative_eq!(generic_dist2, 1.118033988749895);
                    assert_relative_eq!(dist2, generic_dist2);
                }
                if let Some(o3_coord) = o3.coord_ext() {
                    let generic_dist3 = distance_coord_to_line_generic(&o3_coord, &line_seg);
                    assert_relative_eq!(generic_dist3, std::f64::consts::SQRT_2);
                    assert_relative_eq!(dist3, generic_dist3);
                }
                if let Some(o4_coord) = o4.coord_ext() {
                    let generic_dist4 = distance_coord_to_line_generic(&o4_coord, &line_seg);
                    assert_relative_eq!(generic_dist4, 1.5811388300841898);
                    assert_relative_eq!(dist4, generic_dist4);
                }
                if let Some(p1_coord_zero) = p1.coord_ext() {
                    let generic_zero_dist =
                        distance_coord_to_line_generic(&p1_coord_zero, &line_seg);
                    assert_relative_eq!(generic_zero_dist, 0.0);
                    assert_relative_eq!(zero_dist, generic_zero_dist);
                }
            }
        }
        #[test]
        // Point to Polygon, outside point
        fn point_polygon_distance_outside_test() {
            // an octagon
            let points = vec![
                (5., 1.),
                (4., 2.),
                (4., 3.),
                (5., 4.),
                (6., 4.),
                (7., 3.),
                (7., 2.),
                (6., 1.),
                (5., 1.),
            ];
            let ls = LineString::from(points);
            let poly = Polygon::new(ls, vec![]);
            // A Random point outside the octagon
            let p = Point::new(2.5, 0.5);

            // Test original implementation
            let dist = Euclidean.distance(&p, &poly);
            assert_relative_eq!(dist, 2.1213203435596424);

            // Test generic implementation
            let generic_dist = p.distance_ext(&poly);
            assert_relative_eq!(generic_dist, 2.1213203435596424);

            // Ensure both implementations agree
            assert_relative_eq!(dist, generic_dist);
        }
        #[test]
        // Point to Polygon, inside point
        fn point_polygon_distance_inside_test() {
            // an octagon
            let points = vec![
                (5., 1.),
                (4., 2.),
                (4., 3.),
                (5., 4.),
                (6., 4.),
                (7., 3.),
                (7., 2.),
                (6., 1.),
                (5., 1.),
            ];
            let ls = LineString::from(points);
            let poly = Polygon::new(ls, vec![]);
            // A Random point inside the octagon
            let p = Point::new(5.5, 2.1);

            // Test original implementation
            let dist = Euclidean.distance(&p, &poly);
            assert_relative_eq!(dist, 0.0);

            // Test generic implementation
            let generic_dist = p.distance_ext(&poly);
            assert_relative_eq!(generic_dist, 0.0);

            // Ensure both implementations agree
            assert_relative_eq!(dist, generic_dist);
        }
        #[test]
        // Point to Polygon, on boundary
        fn point_polygon_distance_boundary_test() {
            // an octagon
            let points = vec![
                (5., 1.),
                (4., 2.),
                (4., 3.),
                (5., 4.),
                (6., 4.),
                (7., 3.),
                (7., 2.),
                (6., 1.),
                (5., 1.),
            ];
            let ls = LineString::from(points);
            let poly = Polygon::new(ls, vec![]);
            // A point on the octagon
            let p = Point::new(5.0, 1.0);

            // Test original implementation
            let dist = Euclidean.distance(&p, &poly);
            assert_relative_eq!(dist, 0.0);

            // Test generic implementation
            let generic_dist = p.distance_ext(&poly);
            assert_relative_eq!(generic_dist, 0.0);

            // Ensure both implementations agree
            assert_relative_eq!(dist, generic_dist);
        }
        #[test]
        // Point to Polygon, on boundary
        fn point_polygon_boundary_test2() {
            let exterior = LineString::from(vec![
                (0., 0.),
                (0., 0.0004),
                (0.0004, 0.0004),
                (0.0004, 0.),
                (0., 0.),
            ]);

            let poly = Polygon::new(exterior, vec![]);
            let bugged_point = Point::new(0.0001, 0.);

            // Test original implementation
            let distance = Euclidean.distance(&poly, &bugged_point);
            assert_relative_eq!(distance, 0.);

            // Test generic implementation
            let generic_distance = poly.distance_ext(&bugged_point);
            assert_relative_eq!(generic_distance, 0.);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }
        #[test]
        // Point to Polygon, empty Polygon
        fn point_polygon_empty_test() {
            // an empty Polygon
            let points = vec![];
            let ls = LineString::new(points);
            let poly = Polygon::new(ls, vec![]);
            // A point on the octagon
            let p = Point::new(2.5, 0.5);

            // Test original implementation
            let dist = Euclidean.distance(&p, &poly);
            assert_relative_eq!(dist, 0.0);

            // Test generic implementation
            let generic_dist = p.distance_ext(&poly);
            assert_relative_eq!(generic_dist, 0.0);

            // Ensure both implementations agree
            assert_relative_eq!(dist, generic_dist);
        }
        #[test]
        // Point to Polygon with an interior ring
        fn point_polygon_interior_cutout_test() {
            // an octagon
            let ext_points = vec![
                (4., 1.),
                (5., 2.),
                (5., 3.),
                (4., 4.),
                (3., 4.),
                (2., 3.),
                (2., 2.),
                (3., 1.),
                (4., 1.),
            ];
            // cut out a triangle inside octagon
            let int_points = vec![(3.5, 3.5), (4.4, 1.5), (2.6, 1.5), (3.5, 3.5)];
            let ls_ext = LineString::from(ext_points);
            let ls_int = LineString::from(int_points);
            let poly = Polygon::new(ls_ext, vec![ls_int]);
            // A point inside the cutout triangle
            let p = Point::new(3.5, 2.5);

            // Test original implementation
            let dist = Euclidean.distance(&p, &poly);
            // 0.41036467732879783 <-- Shapely
            assert_relative_eq!(dist, 0.41036467732879767);

            // Test generic implementation
            let generic_dist = p.distance_ext(&poly);
            assert_relative_eq!(generic_dist, 0.41036467732879767);

            // Ensure both implementations agree
            assert_relative_eq!(dist, generic_dist);
        }

        #[test]
        fn line_distance_multipolygon_do_not_intersect_test() {
            // checks that the distance from the multipolygon
            // is equal to the distance from the closest polygon
            // taken in isolation, whatever that distance is
            let ls1 = LineString::from(vec![
                (0.0, 0.0),
                (10.0, 0.0),
                (10.0, 10.0),
                (5.0, 15.0),
                (0.0, 10.0),
                (0.0, 0.0),
            ]);
            let ls2 = LineString::from(vec![
                (0.0, 30.0),
                (0.0, 25.0),
                (10.0, 25.0),
                (10.0, 30.0),
                (0.0, 30.0),
            ]);
            let ls3 = LineString::from(vec![
                (15.0, 30.0),
                (15.0, 25.0),
                (20.0, 25.0),
                (20.0, 30.0),
                (15.0, 30.0),
            ]);
            let pol1 = Polygon::new(ls1, vec![]);
            let pol2 = Polygon::new(ls2, vec![]);
            let pol3 = Polygon::new(ls3, vec![]);
            let mp = MultiPolygon::new(vec![pol1.clone(), pol2, pol3]);
            let pnt1 = Point::new(0.0, 15.0);
            let pnt2 = Point::new(10.0, 20.0);
            let ln = Line::new(pnt1.0, pnt2.0);

            // Test original implementation
            let dist_mp_ln = Euclidean.distance(&ln, &mp);
            let dist_pol1_ln = Euclidean.distance(&ln, &pol1);
            assert_relative_eq!(dist_mp_ln, dist_pol1_ln);

            // Test generic implementation - compare line to polygon
            let generic_dist_pol1_ln = ln.distance_ext(&pol1);
            assert_relative_eq!(generic_dist_pol1_ln, dist_pol1_ln);

            // Ensure both implementations agree for the single polygon case
            assert_relative_eq!(dist_pol1_ln, generic_dist_pol1_ln);
        }

        #[test]
        fn point_distance_multipolygon_test() {
            let ls1 = LineString::from(vec![(0.0, 0.0), (1.0, 10.0), (2.0, 0.0), (0.0, 0.0)]);
            let ls2 = LineString::from(vec![(3.0, 0.0), (4.0, 10.0), (5.0, 0.0), (3.0, 0.0)]);
            let p1 = Polygon::new(ls1, vec![]);
            let p2 = Polygon::new(ls2, vec![]);
            let mp = MultiPolygon::new(vec![p1.clone(), p2.clone()]);
            let p = Point::new(50.0, 50.0);

            // Test original implementation
            let distance = Euclidean.distance(&p, &mp);
            assert_relative_eq!(distance, 60.959002616512684);

            // Test generic implementation
            let generic_dist = mp.distance_ext(&p);
            assert_relative_eq!(generic_dist, 60.959002616512684);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_dist);
        }
        #[test]
        // Point to LineString
        fn point_linestring_distance_test() {
            // like an octagon, but missing the lowest horizontal segment
            let points = vec![
                (5., 1.),
                (4., 2.),
                (4., 3.),
                (5., 4.),
                (6., 4.),
                (7., 3.),
                (7., 2.),
                (6., 1.),
            ];
            let ls = LineString::from(points);
            // A Random point "inside" the LineString
            let p = Point::new(5.5, 2.1);

            // Test original implementation
            let dist = Euclidean.distance(&p, &ls);
            assert_relative_eq!(dist, 1.1313708498984762);

            // Test generic implementation
            let generic_dist = p.distance_ext(&ls);
            assert_relative_eq!(generic_dist, 1.1313708498984762);

            // Ensure both implementations agree
            assert_relative_eq!(dist, generic_dist);
        }
        #[test]
        // Point to LineString, point lies on the LineString
        fn point_linestring_contains_test() {
            // like an octagon, but missing the lowest horizontal segment
            let points = vec![
                (5., 1.),
                (4., 2.),
                (4., 3.),
                (5., 4.),
                (6., 4.),
                (7., 3.),
                (7., 2.),
                (6., 1.),
            ];
            let ls = LineString::from(points);
            // A point which lies on the LineString
            let p = Point::new(5.0, 4.0);

            // Test original implementation
            let dist = Euclidean.distance(&p, &ls);
            assert_relative_eq!(dist, 0.0);

            // Test generic implementation
            let generic_dist = p.distance_ext(&ls);
            assert_relative_eq!(generic_dist, 0.0);

            // Ensure both implementations agree
            assert_relative_eq!(dist, generic_dist);
        }
        #[test]
        // Point to LineString, closed triangle
        fn point_linestring_triangle_test() {
            let points = vec![(3.5, 3.5), (4.4, 2.0), (2.6, 2.0), (3.5, 3.5)];
            let ls = LineString::from(points);
            let p = Point::new(3.5, 2.5);

            // Test original implementation
            let dist = Euclidean.distance(&p, &ls);
            assert_relative_eq!(dist, 0.5);

            // Test generic implementation
            let generic_dist = p.distance_ext(&ls);
            assert_relative_eq!(generic_dist, 0.5);

            // Ensure both implementations agree
            assert_relative_eq!(dist, generic_dist);
        }
        #[test]
        // Point to LineString, empty LineString
        fn point_linestring_empty_test() {
            let points = vec![];
            let ls = LineString::new(points);
            let p = Point::new(5.0, 4.0);

            // Test original implementation
            let dist = Euclidean.distance(&p, &ls);
            assert_relative_eq!(dist, 0.0);

            // Test generic implementation
            let generic_dist = p.distance_ext(&ls);
            assert_relative_eq!(generic_dist, 0.0);

            // Ensure both implementations agree
            assert_relative_eq!(dist, generic_dist);
        }
        #[test]
        fn distance_multilinestring_test() {
            let v1 = LineString::from(vec![(0.0, 0.0), (1.0, 10.0)]);
            let v2 = LineString::from(vec![(1.0, 10.0), (2.0, 0.0), (3.0, 1.0)]);
            let mls = MultiLineString::new(vec![v1.clone(), v2.clone()]);
            let p = Point::new(50.0, 50.0);

            // Test original implementation
            let distance = Euclidean.distance(&p, &mls);
            assert_relative_eq!(distance, 63.25345840347388);

            // Test generic implementation
            let generic_dist = p.distance_ext(&mls);
            assert_relative_eq!(generic_dist, 63.25345840347388);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_dist);
        }
        #[test]
        fn distance1_test() {
            let p1 = Point::new(0., 0.);
            let p2 = Point::new(1., 0.);

            // Test original implementation
            let distance = Euclidean.distance(&p1, &p2);
            assert_relative_eq!(distance, 1.);

            // Test generic implementation
            let generic_distance = distance_point_to_point_generic(&p1, &p2);
            assert_relative_eq!(generic_distance, 1.);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }
        #[test]
        fn distance2_test() {
            let p1 = Point::new(-72.1235, 42.3521);
            let p2 = Point::new(72.1260, 70.612);

            // Test original implementation
            let dist = Euclidean.distance(&p1, &p2);
            assert_relative_eq!(dist, 146.99163308930207);

            // Test generic implementation
            let generic_dist = distance_point_to_point_generic(&p1, &p2);
            assert_relative_eq!(generic_dist, 146.99163308930207);

            // Ensure both implementations agree
            assert_relative_eq!(dist, generic_dist);
        }
        #[test]
        fn distance_multipoint_test() {
            let v = vec![
                Point::new(0.0, 10.0),
                Point::new(1.0, 1.0),
                Point::new(10.0, 0.0),
                Point::new(1.0, -1.0),
                Point::new(0.0, -10.0),
                Point::new(-1.0, -1.0),
                Point::new(-10.0, 0.0),
                Point::new(-1.0, 1.0),
                Point::new(0.0, 10.0),
            ];
            let mp = MultiPoint::new(v.clone());
            let p = Point::new(50.0, 50.0);

            // Test original implementation
            let distance = Euclidean.distance(&p, &mp);
            assert_relative_eq!(distance, 64.03124237432849);

            let generic_dist = mp.distance_ext(&p);
            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_dist);
        }
        #[test]
        fn distance_line_test() {
            let line0 = Line::from([(0., 0.), (5., 0.)]);
            let p0 = Point::new(2., 3.);
            let p1 = Point::new(3., 0.);
            let p2 = Point::new(6., 0.);

            // Test original implementation
            let dist_line_p0 = Euclidean.distance(&line0, &p0);
            let dist_p0_line = Euclidean.distance(&p0, &line0);
            assert_relative_eq!(dist_line_p0, 3.);
            assert_relative_eq!(dist_p0_line, 3.);

            let dist_line_p1 = Euclidean.distance(&line0, &p1);
            let dist_p1_line = Euclidean.distance(&p1, &line0);
            assert_relative_eq!(dist_line_p1, 0.);
            assert_relative_eq!(dist_p1_line, 0.);

            let dist_line_p2 = Euclidean.distance(&line0, &p2);
            let dist_p2_line = Euclidean.distance(&p2, &line0);
            assert_relative_eq!(dist_line_p2, 1.);
            assert_relative_eq!(dist_p2_line, 1.);

            // Test generic implementation
            let generic_dist_p0 = if let Some(coord) = p0.coord_ext() {
                distance_coord_to_line_generic(&coord, &line0)
            } else {
                0.0
            };
            let generic_dist_p1 = if let Some(coord) = p1.coord_ext() {
                distance_coord_to_line_generic(&coord, &line0)
            } else {
                0.0
            };
            let generic_dist_p2 = if let Some(coord) = p2.coord_ext() {
                distance_coord_to_line_generic(&coord, &line0)
            } else {
                0.0
            };

            assert_relative_eq!(generic_dist_p0, 3.);
            assert_relative_eq!(generic_dist_p1, 0.);
            assert_relative_eq!(generic_dist_p2, 1.);

            // Ensure both implementations agree
            assert_relative_eq!(dist_line_p0, generic_dist_p0);
            assert_relative_eq!(dist_p0_line, generic_dist_p0);
            assert_relative_eq!(dist_line_p1, generic_dist_p1);
            assert_relative_eq!(dist_p1_line, generic_dist_p1);
            assert_relative_eq!(dist_line_p2, generic_dist_p2);
            assert_relative_eq!(dist_p2_line, generic_dist_p2);
        }
        #[test]
        fn distance_line_line_test() {
            let line0 = Line::from([(0., 0.), (5., 0.)]);
            let line1 = Line::from([(2., 1.), (7., 2.)]);

            // Test original implementation
            let distance01 = Euclidean.distance(&line0, &line1);
            let distance10 = Euclidean.distance(&line1, &line0);
            assert_relative_eq!(distance01, 1.);
            assert_relative_eq!(distance10, 1.);

            // Test generic implementation
            let generic_distance01 = line0.distance_ext(&line1);
            let generic_distance10 = line1.distance_ext(&line0);
            assert_relative_eq!(generic_distance01, 1.);
            assert_relative_eq!(generic_distance10, 1.);

            // Ensure both implementations agree
            assert_relative_eq!(distance01, generic_distance01);
            assert_relative_eq!(distance10, generic_distance10);
        }
        #[test]
        // See https://github.com/georust/geo/issues/476
        fn distance_line_polygon_test() {
            let line = Line::new(
                coord! {
                    x: -0.17084137691985102,
                    y: 0.8748085493016657,
                },
                coord! {
                    x: -0.17084137691985102,
                    y: 0.09858870312437906,
                },
            );
            let poly: Polygon<f64> = polygon![
                coord! {
                    x: -0.10781391405721802,
                    y: -0.15433610862574643,
                },
                coord! {
                    x: -0.7855276236615211,
                    y: 0.23694208404779793,
                },
                coord! {
                    x: -0.7855276236615214,
                    y: -0.5456143012992907,
                },
                coord! {
                    x: -0.10781391405721802,
                    y: -0.15433610862574643,
                },
            ];

            // Test original implementation
            let distance = Euclidean.distance(&line, &poly);
            assert_eq!(distance, 0.18752558079168907);

            // Test generic implementation
            let generic_distance = line.distance_ext(&poly);
            assert_relative_eq!(generic_distance, 0.18752558079168907);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }
        #[test]
        // test edge-vertex minimum distance
        fn test_minimum_polygon_distance() {
            let points_raw = [
                (126., 232.),
                (126., 212.),
                (112., 202.),
                (97., 204.),
                (87., 215.),
                (87., 232.),
                (100., 246.),
                (118., 247.),
            ];
            let points = points_raw
                .iter()
                .map(|e| Point::new(e.0, e.1))
                .collect::<Vec<_>>();
            let poly1 = Polygon::new(LineString::from(points), vec![]);

            let points_raw_2 = [
                (188., 231.),
                (189., 207.),
                (174., 196.),
                (164., 196.),
                (147., 220.),
                (158., 242.),
                (177., 242.),
            ];
            let points2 = points_raw_2
                .iter()
                .map(|e| Point::new(e.0, e.1))
                .collect::<Vec<_>>();
            let poly2 = Polygon::new(LineString::from(points2), vec![]);

            // Test generic implementation
            let generic_dist = poly1.exterior().distance_ext(poly2.exterior());
            assert_relative_eq!(generic_dist, 21.0);
        }
        #[test]
        // test vertex-vertex minimum distance
        fn test_minimum_polygon_distance_2() {
            let points_raw = [
                (118., 200.),
                (153., 179.),
                (106., 155.),
                (88., 190.),
                (118., 200.),
            ];
            let points = points_raw
                .iter()
                .map(|e| Point::new(e.0, e.1))
                .collect::<Vec<_>>();
            let poly1 = Polygon::new(LineString::from(points), vec![]);

            let points_raw_2 = [
                (242., 186.),
                (260., 146.),
                (182., 175.),
                (216., 193.),
                (242., 186.),
            ];
            let points2 = points_raw_2
                .iter()
                .map(|e| Point::new(e.0, e.1))
                .collect::<Vec<_>>();
            let poly2 = Polygon::new(LineString::from(points2), vec![]);

            // Test generic implementation
            let generic_dist = poly1.exterior().distance_ext(poly2.exterior());
            assert_relative_eq!(generic_dist, 29.274562336608895);
        }
        #[test]
        // test edge-edge minimum distance
        fn test_minimum_polygon_distance_3() {
            let points_raw = [
                (182., 182.),
                (182., 168.),
                (138., 160.),
                (136., 193.),
                (182., 182.),
            ];
            let points = points_raw
                .iter()
                .map(|e| Point::new(e.0, e.1))
                .collect::<Vec<_>>();
            let poly1 = Polygon::new(LineString::from(points), vec![]);

            let points_raw_2 = [
                (232., 196.),
                (234., 150.),
                (194., 165.),
                (194., 191.),
                (232., 196.),
            ];
            let points2 = points_raw_2
                .iter()
                .map(|e| Point::new(e.0, e.1))
                .collect::<Vec<_>>();
            let poly2 = Polygon::new(LineString::from(points2), vec![]);

            // Test generic implementation
            let generic_dist = poly1.exterior().distance_ext(poly2.exterior());
            assert_relative_eq!(generic_dist, 12.0);
        }
        #[test]
        fn test_large_polygon_distance() {
            let ls = sedona_testing::fixtures::norway_main::<f64>();
            let poly1 = Polygon::new(ls, vec![]);
            let vec2 = vec![
                (4.921875, 66.33750501996518),
                (3.69140625, 65.21989393613207),
                (6.15234375, 65.07213008560697),
                (4.921875, 66.33750501996518),
            ];
            let poly2 = Polygon::new(vec2.into(), vec![]);

            // Test original implementation
            let distance = Euclidean.distance(&poly1, &poly2);
            // GEOS says 2.2864896295566055
            assert_relative_eq!(distance, 2.2864896295566055);

            // Test generic implementation
            let generic_distance = poly1.distance_ext(&poly2);
            assert_relative_eq!(generic_distance, 2.2864896295566055);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }
        #[test]
        // A polygon inside another polygon's ring; they're disjoint in the DE-9IM sense:
        // FF2FF1212
        fn test_poly_in_ring() {
            let shell = sedona_testing::fixtures::shell::<f64>();
            let ring = sedona_testing::fixtures::ring::<f64>();
            let poly_in_ring = sedona_testing::fixtures::poly_in_ring::<f64>();
            // inside is "inside" outside's ring, but they are disjoint
            let outside = Polygon::new(shell, vec![ring]);
            let inside = Polygon::new(poly_in_ring, vec![]);

            // Test original implementation
            let distance = Euclidean.distance(&outside, &inside);
            assert_relative_eq!(distance, 5.992772737231033);

            // Test generic implementation
            let generic_distance = outside.distance_ext(&inside);
            assert_relative_eq!(generic_distance, 5.992772737231033);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }
        #[test]
        // two ring LineStrings; one encloses the other but they neither touch nor intersect
        fn test_linestring_distance() {
            let ring = sedona_testing::fixtures::ring::<f64>();
            let poly_in_ring = sedona_testing::fixtures::poly_in_ring::<f64>();

            // Test original implementation
            let distance = Euclidean.distance(&ring, &poly_in_ring);
            assert_relative_eq!(distance, 5.992772737231033);

            // Test generic implementation
            let generic_distance = ring.distance_ext(&poly_in_ring);
            assert_relative_eq!(generic_distance, 5.992772737231033);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }
        #[test]
        // Line-Polygon test: closest point on Polygon is NOT nearest to a Line end-point
        fn test_line_polygon_simple() {
            let line = Line::from([(0.0, 0.0), (0.0, 3.0)]);
            let v = vec![(5.0, 1.0), (5.0, 2.0), (0.25, 1.5), (5.0, 1.0)];
            let poly = Polygon::new(v.into(), vec![]);

            // Test original implementation
            let distance = Euclidean.distance(&line, &poly);
            assert_relative_eq!(distance, 0.25);

            // Test generic implementation
            let generic_distance = line.distance_ext(&poly);
            assert_relative_eq!(generic_distance, 0.25);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }
        #[test]
        // Line-Polygon test: Line intersects Polygon
        fn test_line_polygon_intersects() {
            let line = Line::from([(0.5, 0.0), (0.0, 3.0)]);
            let v = vec![(5.0, 1.0), (5.0, 2.0), (0.25, 1.5), (5.0, 1.0)];
            let poly = Polygon::new(v.into(), vec![]);

            // Test original implementation
            let distance = Euclidean.distance(&line, &poly);
            assert_relative_eq!(distance, 0.0);

            // Test generic implementation
            let generic_distance = line.distance_ext(&poly);
            assert_relative_eq!(generic_distance, 0.0);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }
        #[test]
        // Line-Polygon test: Line contained by interior ring
        fn test_line_polygon_inside_ring() {
            let line = Line::from([(4.4, 1.5), (4.45, 1.5)]);
            let v = vec![(5.0, 1.0), (5.0, 2.0), (0.25, 1.0), (5.0, 1.0)];
            let v2 = vec![(4.5, 1.2), (4.5, 1.8), (3.5, 1.2), (4.5, 1.2)];
            let poly = Polygon::new(v.into(), vec![v2.into()]);

            // Test original implementation
            let distance = Euclidean.distance(&line, &poly);
            assert_relative_eq!(distance, 0.04999999999999982);

            // Test generic implementation
            let generic_distance = line.distance_ext(&poly);
            assert_relative_eq!(generic_distance, 0.04999999999999982);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }
        #[test]
        // LineString-Line test
        fn test_linestring_line_distance() {
            let line = Line::from([(0.0, 0.0), (0.0, 2.0)]);
            let ls: LineString<_> = vec![(3.0, 0.0), (1.0, 1.0), (3.0, 2.0)].into();

            // Test original implementation
            let distance = Euclidean.distance(&ls, &line);
            assert_relative_eq!(distance, 1.0);

            // Test generic implementation
            let generic_distance = ls.distance_ext(&line);
            assert_relative_eq!(generic_distance, 1.0);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }

        #[test]
        // Triangle-Point test: point on vertex
        fn test_triangle_point_on_vertex_distance() {
            let triangle = Triangle::from([(0.0, 0.0), (2.0, 0.0), (2.0, 2.0)]);
            let point = Point::new(0.0, 0.0);

            // Test original implementation
            let distance = Euclidean.distance(&triangle, &point);
            assert_relative_eq!(distance, 0.0);

            // Test generic implementation
            let generic_distance = triangle.distance_ext(&point);
            assert_relative_eq!(generic_distance, 0.0);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }

        #[test]
        // Triangle-Point test: point on edge
        fn test_triangle_point_on_edge_distance() {
            let triangle = Triangle::from([(0.0, 0.0), (2.0, 0.0), (2.0, 2.0)]);
            let point = Point::new(1.5, 0.0);

            // Test original implementation
            let distance = Euclidean.distance(&triangle, &point);
            assert_relative_eq!(distance, 0.0);

            // Test generic implementation
            let generic_distance = triangle.distance_ext(&point);
            assert_relative_eq!(generic_distance, 0.0);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }

        #[test]
        // Triangle-Point test
        fn test_triangle_point_distance() {
            let triangle = Triangle::from([(0.0, 0.0), (2.0, 0.0), (2.0, 2.0)]);
            let point = Point::new(2.0, 3.0);

            // Test original implementation
            let distance = Euclidean.distance(&triangle, &point);
            assert_relative_eq!(distance, 1.0);

            // Test generic implementation
            let generic_distance = triangle.distance_ext(&point);
            assert_relative_eq!(generic_distance, 1.0);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }

        #[test]
        // Triangle-Point test: point within triangle
        fn test_triangle_point_inside_distance() {
            let triangle = Triangle::from([(0.0, 0.0), (2.0, 0.0), (2.0, 2.0)]);
            let point = Point::new(1.0, 0.5);

            // Test original implementation
            let distance = Euclidean.distance(&triangle, &point);
            assert_relative_eq!(distance, 0.0);

            // Test generic implementation
            let generic_distance = triangle.distance_ext(&point);
            assert_relative_eq!(generic_distance, 0.0);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }

        #[test]
        fn convex_and_nearest_neighbour_comparison() {
            let ls1: LineString<f64> = vec![
                Coord::from((57.39453770777941, 307.60533608924663)),
                Coord::from((67.1800355576469, 309.6654408997451)),
                Coord::from((84.89693692793338, 225.5101593908847)),
                Coord::from((75.1114390780659, 223.45005458038628)),
                Coord::from((57.39453770777941, 307.60533608924663)),
            ]
            .into();
            let first_polygon: Polygon<f64> = Polygon::new(ls1, vec![]);
            let ls2: LineString<f64> = vec![
                Coord::from((138.11769866645008, -45.75134112915392)),
                Coord::from((130.50230476949187, -39.270154833870336)),
                Coord::from((184.94426964987397, 24.699153900578573)),
                Coord::from((192.55966354683218, 18.217967605294987)),
                Coord::from((138.11769866645008, -45.75134112915392)),
            ]
            .into();
            let second_polygon = Polygon::new(ls2, vec![]);

            // Test original implementation
            let distance = Euclidean.distance(&first_polygon, &second_polygon);
            assert_relative_eq!(distance, 224.35357967013238);

            // Test generic implementation
            let generic_distance = first_polygon.distance_ext(&second_polygon);
            assert_relative_eq!(generic_distance, 224.35357967013238);

            // Ensure both implementations agree
            assert_relative_eq!(distance, generic_distance);
        }
        #[test]
        fn fast_path_regression() {
            // this test will fail if the fast path algorithm is reintroduced without being fixed
            let p1 = polygon!(
                (x: 0_f64, y: 0_f64),
                (x: 300_f64, y: 0_f64),
                (x: 300_f64, y: 100_f64),
                (x: 0_f64, y: 100_f64),
            )
            .orient(Direction::Default);
            let p2 = polygon!(
                (x: 100_f64, y: 150_f64),
                (x: 150_f64, y: 200_f64),
                (x: 50_f64, y: 200_f64),
            )
            .orient(Direction::Default);
            let p3 = polygon!(
                (x: 0_f64, y: 0_f64),
                (x: 300_f64, y: 0_f64),
                (x: 300_f64, y: 100_f64),
                (x: 0_f64, y: 100_f64),
            )
            .orient(Direction::Reversed);
            let p4 = polygon!(
                (x: 100_f64, y: 150_f64),
                (x: 150_f64, y: 200_f64),
                (x: 50_f64, y: 200_f64),
            )
            .orient(Direction::Reversed);

            // Test original implementation
            let distance_p1_p2 = Euclidean.distance(&p1, &p2);
            let distance_p3_p4 = Euclidean.distance(&p3, &p4);
            let distance_p1_p4 = Euclidean.distance(&p1, &p4);
            let distance_p2_p3 = Euclidean.distance(&p2, &p3);
            assert_eq!(distance_p1_p2, 50.0f64);
            assert_eq!(distance_p3_p4, 50.0f64);
            assert_eq!(distance_p1_p4, 50.0f64);
            assert_eq!(distance_p2_p3, 50.0f64);

            // Test generic implementation
            let generic_distance_p1_p2 = p1.distance_ext(&p2);
            let generic_distance_p3_p4 = p3.distance_ext(&p4);
            let generic_distance_p1_p4 = p1.distance_ext(&p4);
            let generic_distance_p2_p3 = p2.distance_ext(&p3);
            assert_relative_eq!(generic_distance_p1_p2, 50.0f64);
            assert_relative_eq!(generic_distance_p3_p4, 50.0f64);
            assert_relative_eq!(generic_distance_p1_p4, 50.0f64);
            assert_relative_eq!(generic_distance_p2_p3, 50.0f64);

            // Ensure both implementations agree
            assert_relative_eq!(distance_p1_p2, generic_distance_p1_p2);
            assert_relative_eq!(distance_p3_p4, generic_distance_p3_p4);
            assert_relative_eq!(distance_p1_p4, generic_distance_p1_p4);
            assert_relative_eq!(distance_p2_p3, generic_distance_p2_p3);
        }
        #[test]
        fn rect_to_polygon_distance_test() {
            // Test that Rect to Polygon distance works
            let rect = Rect::new((0.0, 0.0), (2.0, 2.0));
            let poly_points = vec![(3., 0.), (5., 0.), (5., 2.), (3., 2.), (3., 0.)];
            let poly = Polygon::new(LineString::from(poly_points), vec![]);

            // Test original implementation (both directions)
            let dist1 = Euclidean.distance(&rect, &poly);
            let dist2 = Euclidean.distance(&poly, &rect);
            assert_relative_eq!(dist1, 1.0);
            assert_relative_eq!(dist2, 1.0);
            assert_relative_eq!(dist1, dist2); // Verify symmetry

            // Test generic implementation
            let rect_as_poly = rect.to_polygon();
            let generic_dist1 = rect_as_poly.distance_ext(&poly);
            let generic_dist2 = poly.distance_ext(&rect_as_poly);
            assert_relative_eq!(generic_dist1, 1.0);
            assert_relative_eq!(generic_dist2, 1.0);

            // Ensure both implementations agree
            assert_relative_eq!(dist1, generic_dist1);
            assert_relative_eq!(dist2, generic_dist2);
        }

        #[test]
        fn all_types_geometry_collection_test() {
            let p = Point::new(0.0, 0.0);
            let line = Line::from([(-1.0, -1.0), (-2.0, -2.0)]);
            let ls = LineString::from(vec![(0.0, 0.0), (1.0, 10.0), (2.0, 0.0)]);
            let poly = Polygon::new(
                LineString::from(vec![(0.0, 0.0), (1.0, 10.0), (2.0, 0.0), (0.0, 0.0)]),
                vec![],
            );
            let tri = Triangle::from([(0.0, 0.0), (1.0, 10.0), (2.0, 0.0)]);
            let rect = Rect::new((0.0, 0.0), (-1.0, -1.0));

            let ls1 = LineString::from(vec![(0.0, 0.0), (1.0, 10.0), (2.0, 0.0), (0.0, 0.0)]);
            let ls2 = LineString::from(vec![(3.0, 0.0), (4.0, 10.0), (5.0, 0.0), (3.0, 0.0)]);
            let p1 = Polygon::new(ls1, vec![]);
            let p2 = Polygon::new(ls2, vec![]);
            let mpoly = MultiPolygon::new(vec![p1, p2]);

            let v = vec![
                Point::new(0.0, 10.0),
                Point::new(1.0, 1.0),
                Point::new(10.0, 0.0),
                Point::new(1.0, -1.0),
                Point::new(0.0, -10.0),
                Point::new(-1.0, -1.0),
                Point::new(-10.0, 0.0),
                Point::new(-1.0, 1.0),
                Point::new(0.0, 10.0),
            ];
            let mpoint = MultiPoint::new(v);

            let v1 = LineString::from(vec![(0.0, 0.0), (1.0, 10.0)]);
            let v2 = LineString::from(vec![(1.0, 10.0), (2.0, 0.0), (3.0, 1.0)]);
            let mls = MultiLineString::new(vec![v1, v2]);

            let gc = GeometryCollection(vec![
                Geometry::Point(p),
                Geometry::Line(line),
                Geometry::LineString(ls),
                Geometry::Polygon(poly),
                Geometry::MultiPoint(mpoint),
                Geometry::MultiLineString(mls),
                Geometry::MultiPolygon(mpoly),
                Geometry::Triangle(tri),
                Geometry::Rect(rect),
            ]);

            // Test original implementations
            let test_p = Point::new(50., 50.);
            let distance_p_gc = Euclidean.distance(&test_p, &gc);
            assert_relative_eq!(distance_p_gc, 60.959002616512684);

            let test_multipoint = MultiPoint::new(vec![test_p]);
            let distance_mp_gc = Euclidean.distance(&test_multipoint, &gc);
            assert_relative_eq!(distance_mp_gc, 60.959002616512684);

            let test_line = Line::from([(50., 50.), (60., 60.)]);
            let distance_line_gc = Euclidean.distance(&test_line, &gc);
            assert_relative_eq!(distance_line_gc, 60.959002616512684);

            let test_ls = LineString::from(vec![(50., 50.), (60., 60.), (70., 70.)]);
            let distance_ls_gc = Euclidean.distance(&test_ls, &gc);
            assert_relative_eq!(distance_ls_gc, 60.959002616512684);

            let test_mls = MultiLineString::new(vec![test_ls]);
            let distance_mls_gc = Euclidean.distance(&test_mls, &gc);
            assert_relative_eq!(distance_mls_gc, 60.959002616512684);

            let test_poly = Polygon::new(
                LineString::from(vec![
                    (50., 50.),
                    (60., 50.),
                    (60., 60.),
                    (55., 55.),
                    (50., 50.),
                ]),
                vec![],
            );
            let distance_poly_gc = Euclidean.distance(&test_poly, &gc);
            assert_relative_eq!(distance_poly_gc, 60.959002616512684);

            let test_multipoly = MultiPolygon::new(vec![test_poly]);
            let distance_multipoly_gc = Euclidean.distance(&test_multipoly, &gc);
            assert_relative_eq!(distance_multipoly_gc, 60.959002616512684);

            let test_tri = Triangle::from([(50., 50.), (60., 50.), (55., 55.)]);
            let distance_tri_gc = Euclidean.distance(&test_tri, &gc);
            assert_relative_eq!(distance_tri_gc, 60.959002616512684);

            let test_rect = Rect::new(coord! { x: 50., y: 50. }, coord! { x: 60., y: 60. });
            let distance_rect_gc = Euclidean.distance(&test_rect, &gc);
            assert_relative_eq!(distance_rect_gc, 60.959002616512684);

            let test_gc = GeometryCollection(vec![Geometry::Rect(test_rect)]);
            let distance_gc_gc = Euclidean.distance(&test_gc, &gc);
            assert_relative_eq!(distance_gc_gc, 60.959002616512684);
        }

        #[test]
        fn test_original_issue_verification() {
            let point = Point::new(0.0, 0.0);
            let linestring = LineString::from(vec![(0.0, 0.0), (1.0, 1.0)]);

            let gc1 = GeometryCollection(vec![
                Geometry::Point(point),
                Geometry::LineString(linestring.clone()),
            ]);

            let gc2 = GeometryCollection(vec![
                Geometry::Point(point),
                Geometry::LineString(linestring),
            ]);

            // Test the concrete Distance API
            let distance = Euclidean.distance(&gc1, &gc2);
            assert_eq!(
                distance, 0.0,
                "Distance between identical GeometryCollections should be 0"
            );

            // Test the generic distance_ext API directly
            use crate::line_measures::DistanceExt;
            let distance_ext = gc1.distance_ext(&gc2);
            assert_eq!(distance_ext, 0.0, "Generic distance should also be 0");
        }

        #[test]
        fn test_force_generic_trait_recursion() {
            let point = Point::new(0.0, 0.0);
            let linestring = LineString::from(vec![(0.0, 0.0), (1.0, 1.0)]);

            let gc1 = GeometryCollection(vec![
                Geometry::Point(point),
                Geometry::LineString(linestring.clone()),
            ]);

            let gc2 = GeometryCollection(vec![
                Geometry::Point(point),
                Geometry::LineString(linestring),
            ]);

            let distance_result = gc1.distance_ext(&gc2);
            assert_eq!(distance_result, 0.0);

            let geom_gc1 = Geometry::GeometryCollection(gc1.clone());
            let geom_gc2 = Geometry::GeometryCollection(gc2.clone());
            let distance_result = geom_gc1.distance_ext(&geom_gc2);
            assert_eq!(distance_result, 0.0);

            let distance_result = geom_gc1.distance_ext(&gc2);
            assert_eq!(distance_result, 0.0);

            let distance_result = gc1.distance_ext(&geom_gc2);
            assert_eq!(distance_result, 0.0);
        }
    }
}
