/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.sedona.viz.core.Serde;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.Serializer;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import org.apache.log4j.Logger;
import org.apache.sedona.viz.utils.Pixel;

/**
 * This serializer aims at serializing Pixel object. In the generated ByteArray, we put attributes as follows: X, Y, ResolutionX,
 * ResolutionY, isDuplicate, currentPartitionId.
 */
public class PixelSerializer
        extends Serializer<Pixel>
{
    final static Logger log = Logger.getLogger(PixelSerializer.class);

    @Override
    public void write(Kryo kryo, Output output, Pixel object)
    {
        log.debug("Serializing Pixel...");
        output.writeInt((int) object.getX());
        output.writeInt((int) object.getY());
        output.writeInt(object.getResolutionX());
        output.writeInt(object.getResolutionY());
        output.writeBoolean(object.isDuplicate());
        output.writeInt(object.getCurrentPartitionId());
    }

    @Override
    public Pixel read(Kryo kryo, Input input, Class<Pixel> type)
    {
        log.debug("De-serializing Pixel...");
        int X = input.readInt();
        int Y = input.readInt();
        int ResolutionX = input.readInt();
        int ResolutionY = input.readInt();
        boolean isDuplicate = input.readBoolean();
        int currentPartitionId = input.readInt();
        Pixel pixel = new Pixel(X, Y, ResolutionX, ResolutionY, isDuplicate, currentPartitionId);
        return pixel;
    }

    public Pixel readPixel(byte[] inputArray)
    {
        Kryo kryo = new Kryo();
        Input input = new Input(inputArray);
        return read(kryo, input, Pixel.class);
    }
}
