/*
 * Decompiled with CFR 0.152.
 */
package org.apache.twill.discovery;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.hash.Hashing;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nullable;
import org.apache.twill.common.Cancellable;
import org.apache.twill.common.Threads;
import org.apache.twill.discovery.DefaultServiceDiscovered;
import org.apache.twill.discovery.Discoverable;
import org.apache.twill.discovery.DiscoverableAdapter;
import org.apache.twill.discovery.DiscoveryService;
import org.apache.twill.discovery.DiscoveryServiceClient;
import org.apache.twill.discovery.ServiceDiscovered;
import org.apache.twill.zookeeper.NodeChildren;
import org.apache.twill.zookeeper.NodeData;
import org.apache.twill.zookeeper.OperationFuture;
import org.apache.twill.zookeeper.ZKClient;
import org.apache.twill.zookeeper.ZKClients;
import org.apache.twill.zookeeper.ZKOperations;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKDiscoveryService
implements DiscoveryService,
DiscoveryServiceClient,
AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(ZKDiscoveryService.class);
    private static final long RETRY_MILLIS = 1000L;
    private final AtomicBoolean closed = new AtomicBoolean();
    private final Multimap<Discoverable, DiscoveryCancellable> discoverables = HashMultimap.create();
    private final Lock lock = new ReentrantLock();
    private final LoadingCache<String, ServiceDiscoveredCacheEntry> services;
    private final ZKClient zkClient;
    private final ScheduledExecutorService retryExecutor = Executors.newSingleThreadScheduledExecutor(Threads.createDaemonThreadFactory("zk-discovery-retry"));
    private final Cancellable watcherCancellable;

    public ZKDiscoveryService(ZKClient zkClient) {
        this(zkClient, "/discoverable");
    }

    public ZKDiscoveryService(ZKClient zkClient, String namespace) {
        this.zkClient = namespace == null ? zkClient : ZKClients.namespace(zkClient, namespace);
        this.services = CacheBuilder.newBuilder().removalListener((RemovalListener)new RemovalListener<String, ServiceDiscoveredCacheEntry>(){

            public void onRemoval(RemovalNotification<String, ServiceDiscoveredCacheEntry> notification) {
                ServiceDiscoveredCacheEntry entry = (ServiceDiscoveredCacheEntry)notification.getValue();
                if (entry != null) {
                    entry.cancel();
                }
            }
        }).build(this.createServiceLoader());
        this.watcherCancellable = this.zkClient.addConnectionWatcher(this.createConnectionWatcher());
    }

    @Override
    public Cancellable register(final Discoverable discoverable) {
        if (this.closed.get()) {
            throw new IllegalStateException("Cannot register discoverable through a closed ZKDiscoveryService");
        }
        final SettableFuture future = SettableFuture.create();
        final DiscoveryCancellable cancellable = new DiscoveryCancellable(discoverable);
        Futures.addCallback(this.doRegister(discoverable), (FutureCallback)new FutureCallback<String>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void onSuccess(String result) {
                cancellable.setPath(result);
                ZKDiscoveryService.this.lock.lock();
                try {
                    if (!ZKDiscoveryService.this.closed.get()) {
                        ZKDiscoveryService.this.discoverables.put((Object)discoverable, (Object)cancellable);
                    } else {
                        cancellable.asyncCancel();
                    }
                }
                finally {
                    ZKDiscoveryService.this.lock.unlock();
                }
                LOG.debug("Service registered: {} {}", (Object)discoverable, (Object)result);
                future.set((Object)result);
            }

            public void onFailure(Throwable t) {
                if (t instanceof KeeperException.NodeExistsException) {
                    ZKDiscoveryService.this.handleRegisterFailure(discoverable, (SettableFuture<String>)future, (FutureCallback<String>)this, t);
                } else {
                    LOG.warn("Failed to register: {}", (Object)discoverable, (Object)t);
                    future.setException(t);
                }
            }
        }, (Executor)Threads.SAME_THREAD_EXECUTOR);
        Futures.getUnchecked((Future)future);
        return cancellable;
    }

    @Override
    public ServiceDiscovered discover(String service) {
        if (this.closed.get()) {
            throw new IllegalStateException("Cannot discover through a closed ZKDiscoveryService");
        }
        return (ServiceDiscovered)this.services.getUnchecked((Object)service);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        if (!this.closed.compareAndSet(false, true)) {
            return;
        }
        this.retryExecutor.shutdownNow();
        this.watcherCancellable.cancel();
        ArrayList futures = new ArrayList();
        this.lock.lock();
        try {
            for (Map.Entry entry : this.discoverables.entries()) {
                LOG.debug("Un-registering service {} - {}", (Object)((Discoverable)entry.getKey()).getName(), (Object)((Discoverable)entry.getKey()).getSocketAddress());
                futures.add(((DiscoveryCancellable)entry.getValue()).asyncCancel());
            }
        }
        finally {
            this.lock.unlock();
        }
        try {
            Futures.successfulAsList(futures).get();
            LOG.debug("All services unregistered");
        }
        catch (Exception e) {
            LOG.warn("Unexpected exception when waiting for all services to get unregistered", (Throwable)e);
        }
        this.services.invalidateAll();
    }

    private void handleRegisterFailure(final Discoverable discoverable, final SettableFuture<String> completion, final FutureCallback<String> creationCallback, final Throwable failureCause) {
        if (this.closed.get()) {
            return;
        }
        final String path = this.getNodePath(discoverable);
        Futures.addCallback(this.zkClient.exists(path), (FutureCallback)new FutureCallback<Stat>(){

            public void onSuccess(@Nullable Stat result) {
                if (result == null) {
                    LOG.info("Node {} is gone. Retry registration for {}.", (Object)path, (Object)discoverable);
                    ZKDiscoveryService.this.retryRegister(discoverable, (FutureCallback<String>)creationCallback);
                    return;
                }
                long ephemeralOwner = result.getEphemeralOwner();
                if (ephemeralOwner == 0L) {
                    LOG.error("Node {} already exists and is not an ephemeral node. Discoverable registration failed: {}.", (Object)path, (Object)discoverable);
                    completion.setException(failureCause);
                    return;
                }
                Long sessionId = ZKDiscoveryService.this.zkClient.getSessionId();
                if (sessionId == null || ephemeralOwner != sessionId) {
                    LOG.info("Owner of {} is different. Retry registration for {}.", (Object)path, (Object)discoverable);
                    ZKDiscoveryService.this.retryRegister(discoverable, (FutureCallback<String>)creationCallback);
                } else {
                    completion.set((Object)path);
                }
            }

            public void onFailure(Throwable t) {
                LOG.warn("Error when getting stats on {}. Retry registration for {}.", (Object)path, (Object)discoverable);
                ZKDiscoveryService.this.retryRegister(discoverable, (FutureCallback<String>)creationCallback);
            }
        }, (Executor)Threads.SAME_THREAD_EXECUTOR);
    }

    private OperationFuture<String> doRegister(Discoverable discoverable) {
        byte[] discoverableBytes = DiscoverableAdapter.encode(discoverable);
        return this.zkClient.create(this.getNodePath(discoverable), discoverableBytes, CreateMode.EPHEMERAL, true);
    }

    private void retryRegister(final Discoverable discoverable, final FutureCallback<String> creationCallback) {
        if (this.closed.get()) {
            return;
        }
        this.retryExecutor.schedule(new Runnable(){

            @Override
            public void run() {
                if (!ZKDiscoveryService.this.closed.get()) {
                    Futures.addCallback((ListenableFuture)ZKDiscoveryService.this.doRegister(discoverable), (FutureCallback)creationCallback, (Executor)Threads.SAME_THREAD_EXECUTOR);
                }
            }
        }, 1000L, TimeUnit.MILLISECONDS);
    }

    private String getNodePath(Discoverable discoverable) {
        InetSocketAddress socketAddress = discoverable.getSocketAddress();
        String node = Hashing.md5().newHasher().putBytes(socketAddress.getAddress().getAddress()).putInt(socketAddress.getPort()).hash().toString();
        return String.format("/%s/%s", discoverable.getName(), node);
    }

    private Watcher createConnectionWatcher() {
        return new Watcher(){
            private boolean expired;

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void process(WatchedEvent event) {
                if (event.getState() == Watcher.Event.KeeperState.Expired) {
                    LOG.warn("ZK Session expired: {}", (Object)ZKDiscoveryService.this.zkClient.getConnectString());
                    this.expired = true;
                } else if (event.getState() == Watcher.Event.KeeperState.SyncConnected && this.expired) {
                    LOG.info("Reconnected after expiration: {}", (Object)ZKDiscoveryService.this.zkClient.getConnectString());
                    this.expired = false;
                    ZKDiscoveryService.this.lock.lock();
                    try {
                        for (final Map.Entry entry : ZKDiscoveryService.this.discoverables.entries()) {
                            if (ZKDiscoveryService.this.closed.get()) {
                                ((DiscoveryCancellable)entry.getValue()).asyncCancel();
                                continue;
                            }
                            LOG.info("Re-registering service: {}", entry.getKey());
                            Futures.addCallback((ListenableFuture)ZKDiscoveryService.this.doRegister((Discoverable)entry.getKey()), (FutureCallback)new FutureCallback<String>(){

                                public void onSuccess(String result) {
                                    ((DiscoveryCancellable)entry.getValue()).setPath(result);
                                    LOG.debug("Service re-registered: {} {}", entry.getKey(), (Object)result);
                                }

                                public void onFailure(Throwable t) {
                                    ((DiscoveryCancellable)entry.getValue()).setPath(null);
                                    LOG.error("Failed to re-register service: {}", entry.getKey(), (Object)t);
                                }
                            }, (Executor)Threads.SAME_THREAD_EXECUTOR);
                        }
                    }
                    finally {
                        ZKDiscoveryService.this.lock.unlock();
                    }
                }
            }
        };
    }

    private CacheLoader<String, ServiceDiscoveredCacheEntry> createServiceLoader() {
        return new CacheLoader<String, ServiceDiscoveredCacheEntry>(){

            public ServiceDiscoveredCacheEntry load(String service) throws Exception {
                final DefaultServiceDiscovered serviceDiscovered = new DefaultServiceDiscovered(service);
                final String pathBase = "/" + service;
                Cancellable cancellable = ZKOperations.watchChildren(ZKDiscoveryService.this.zkClient, pathBase, new ZKOperations.ChildrenCallback(){

                    @Override
                    public void updated(NodeChildren nodeChildren) {
                        List<String> children = nodeChildren.getChildren();
                        ArrayList dataFutures = Lists.newArrayListWithCapacity((int)children.size());
                        for (String child : children) {
                            dataFutures.add(ZKDiscoveryService.this.zkClient.getData(pathBase + "/" + child));
                        }
                        final ListenableFuture fetchFuture = Futures.successfulAsList((Iterable)dataFutures);
                        fetchFuture.addListener(new Runnable(){

                            @Override
                            public void run() {
                                ImmutableSet.Builder builder = ImmutableSet.builder();
                                for (NodeData nodeData : (List)Futures.getUnchecked((Future)fetchFuture)) {
                                    Discoverable discoverable;
                                    if (nodeData == null || (discoverable = DiscoverableAdapter.decode(nodeData.getData())) == null) continue;
                                    builder.add((Object)discoverable);
                                }
                                serviceDiscovered.setDiscoverables((Set<Discoverable>)builder.build());
                            }
                        }, Threads.SAME_THREAD_EXECUTOR);
                    }
                });
                return new ServiceDiscoveredCacheEntry(serviceDiscovered, cancellable);
            }
        };
    }

    private static final class ServiceDiscoveredCacheEntry
    implements Cancellable,
    ServiceDiscovered {
        private final ServiceDiscovered serviceDiscovered;
        private final Cancellable cancellable;

        private ServiceDiscoveredCacheEntry(ServiceDiscovered serviceDiscovered, Cancellable cancellable) {
            this.serviceDiscovered = serviceDiscovered;
            this.cancellable = cancellable;
        }

        @Override
        public void cancel() {
            this.cancellable.cancel();
        }

        @Override
        public String getName() {
            return this.serviceDiscovered.getName();
        }

        @Override
        public Cancellable watchChanges(ServiceDiscovered.ChangeListener listener, Executor executor) {
            return this.serviceDiscovered.watchChanges(listener, executor);
        }

        @Override
        public boolean contains(Discoverable discoverable) {
            return this.serviceDiscovered.contains(discoverable);
        }

        @Override
        public Iterator<Discoverable> iterator() {
            return this.serviceDiscovered.iterator();
        }
    }

    private final class DiscoveryCancellable
    implements Cancellable {
        private final Discoverable discoverable;
        private final AtomicBoolean cancelled;
        private volatile String path;

        DiscoveryCancellable(Discoverable discoverable) {
            this.discoverable = discoverable;
            this.cancelled = new AtomicBoolean();
        }

        void setPath(String path) {
            this.path = path;
            if (this.cancelled.get() && path != null) {
                ZKDiscoveryService.this.zkClient.delete(path);
            }
        }

        @Override
        public void cancel() {
            Futures.getUnchecked(this.asyncCancel());
            LOG.debug("Service unregistered: {} {}", (Object)this.discoverable, (Object)this.path);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        ListenableFuture<?> asyncCancel() {
            if (!this.cancelled.compareAndSet(false, true)) {
                return Futures.immediateFuture(null);
            }
            String path = this.path;
            if (path == null) {
                return Futures.immediateFuture(null);
            }
            ZKDiscoveryService.this.lock.lock();
            try {
                ZKDiscoveryService.this.discoverables.remove((Object)this.discoverable, (Object)this);
            }
            finally {
                ZKDiscoveryService.this.lock.unlock();
            }
            try {
                return ZKOperations.ignoreError(ZKDiscoveryService.this.zkClient.delete(path), KeeperException.NoNodeException.class, path);
            }
            catch (Exception e) {
                return Futures.immediateFailedFuture((Throwable)e);
            }
        }
    }
}

