/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tephra.distributed;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.inject.Guice;
import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Module;
import com.google.inject.name.Named;
import java.io.InputStream;
import java.lang.management.ManagementFactory;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import org.apache.hadoop.conf.Configuration;
import org.apache.tephra.InvalidTruncateTimeException;
import org.apache.tephra.Transaction;
import org.apache.tephra.TransactionConflictException;
import org.apache.tephra.TransactionCouldNotTakeSnapshotException;
import org.apache.tephra.TransactionFailureException;
import org.apache.tephra.TransactionNotInProgressException;
import org.apache.tephra.TransactionSizeException;
import org.apache.tephra.TransactionSystemClient;
import org.apache.tephra.distributed.CloseableThriftClient;
import org.apache.tephra.distributed.RetryNTimes;
import org.apache.tephra.distributed.RetryStrategy;
import org.apache.tephra.distributed.RetryStrategyProvider;
import org.apache.tephra.distributed.RetryWithBackoff;
import org.apache.tephra.distributed.ThriftClientProvider;
import org.apache.tephra.distributed.TransactionServiceThriftClient;
import org.apache.tephra.runtime.ConfigModule;
import org.apache.tephra.runtime.DiscoveryModules;
import org.apache.tephra.runtime.TransactionClientModule;
import org.apache.tephra.runtime.TransactionModules;
import org.apache.tephra.runtime.ZKModule;
import org.apache.tephra.util.ConfigurationFactory;
import org.apache.thrift.TException;
import org.apache.twill.zookeeper.ZKClientService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransactionServiceClient
implements TransactionSystemClient {
    private static final Logger LOG = LoggerFactory.getLogger(TransactionServiceClient.class);
    private ThriftClientProvider clientProvider;
    private final RetryStrategyProvider retryStrategyProvider;
    private final String clientId;
    private final int changeSetCountLimit;
    private final int changeSetCountThreshold;
    private final long changeSetSizeLimit;
    private final long changeSetSizeThreshold;

    public static void main(String[] args) throws Exception {
        if (args.length > 1 || args.length == 1 && !"-v".equals(args[0])) {
            System.out.println("USAGE: TransactionServiceClient [-v]");
        }
        boolean verbose = false;
        if (args.length == 1 && "-v".equals(args[0])) {
            verbose = true;
        }
        TransactionServiceClient.doMain(verbose, new ConfigurationFactory().get());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public static void doMain(boolean verbose, Configuration conf) throws Exception {
        LOG.info("Starting tx server client test.");
        Injector injector = Guice.createInjector((Module[])new Module[]{new ConfigModule(conf), new ZKModule(), new DiscoveryModules().getDistributedModules(), new TransactionModules().getDistributedModules(), new TransactionClientModule()});
        ZKClientService zkClient = (ZKClientService)injector.getInstance(ZKClientService.class);
        zkClient.startAndWait();
        try {
            TransactionServiceClient client = (TransactionServiceClient)injector.getInstance(TransactionServiceClient.class);
            LOG.info("Starting tx...");
            Transaction tx = client.startShort();
            if (verbose) {
                LOG.info("Started tx details: " + tx.toString());
            } else {
                LOG.info("Started tx: " + tx.getTransactionId() + ", readPointer: " + tx.getReadPointer() + ", invalids: " + tx.getInvalids().length + ", inProgress: " + tx.getInProgress().length);
            }
            try {
                LOG.info("Checking if canCommit tx...");
                client.canCommitOrThrow(tx, Collections.emptyList());
                LOG.info("canCommit: success");
                LOG.info("Committing tx...");
                client.commitOrThrow(tx);
                LOG.info("Committed tx: success");
            }
            catch (TransactionConflictException e) {
                LOG.info("Aborting tx...");
                client.abort(tx);
                LOG.info("Aborted tx...");
            }
        }
        finally {
            zkClient.stopAndWait();
        }
    }

    public TransactionServiceClient(Configuration config, ThriftClientProvider clientProvider) {
        this(config, clientProvider, ManagementFactory.getRuntimeMXBean().getName());
    }

    @Inject
    public TransactionServiceClient(Configuration config, ThriftClientProvider clientProvider, @Named(value="tephra.client.id") String clientId) {
        String retryStrat = config.get("data.tx.client.retry.strategy", "backoff");
        if ("backoff".equals(retryStrat)) {
            this.retryStrategyProvider = new RetryWithBackoff.Provider();
        } else if ("n-times".equals(retryStrat)) {
            this.retryStrategyProvider = new RetryNTimes.Provider();
        } else {
            try {
                this.retryStrategyProvider = (RetryStrategyProvider)Class.forName(retryStrat).newInstance();
            }
            catch (ClassNotFoundException | IllegalAccessException | InstantiationException e) {
                throw new IllegalArgumentException(String.format("Unable to instantiate RetryStrategyProvider '%s'", retryStrat), e);
            }
        }
        this.retryStrategyProvider.configure(config);
        LOG.debug("Retry strategy is " + this.retryStrategyProvider);
        this.clientProvider = clientProvider;
        this.clientId = clientId;
        this.changeSetCountLimit = config.getInt("data.tx.changeset.count.limit", Integer.MAX_VALUE);
        this.changeSetCountThreshold = config.getInt("data.tx.changeset.count.warn.threshold", Integer.MAX_VALUE);
        this.changeSetSizeLimit = config.getLong("data.tx.changeset.size.limit", Long.MAX_VALUE);
        this.changeSetSizeThreshold = config.getLong("data.tx.changeset.size.warn.threshold", Long.MAX_VALUE);
    }

    private <T> T execute(Operation<T> operation) throws Exception {
        return this.execute(operation, null);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private <T> T execute(Operation<T> operation, ThriftClientProvider provider) throws Exception {
        RetryStrategy retryStrategy = this.retryStrategyProvider.newRetryStrategy();
        while (true) {
            if (provider == null) {
                provider = this.clientProvider;
            }
            try (CloseableThriftClient closeable = provider.getCloseableClient();){
                T t = operation.execute(closeable.getThriftClient());
                return t;
            }
            catch (TException te) {
                boolean retry = retryStrategy.failOnce();
                if (!retry) {
                    String message = "Thrift error for " + operation + ": " + te.getMessage();
                    LOG.error(message);
                    LOG.debug(message, (Throwable)te);
                    throw te;
                }
                retryStrategy.beforeRetry();
                String msg = "Retrying " + operation.getName() + " after Thrift error: " + te.getMessage();
                LOG.info(msg);
                LOG.debug(msg, (Throwable)te);
                continue;
            }
            break;
        }
    }

    @Override
    public Transaction startLong() {
        try {
            return this.execute(new Operation<Transaction>("startLong"){

                @Override
                public Transaction execute(TransactionServiceThriftClient client) throws TException {
                    return client.startLong(TransactionServiceClient.this.clientId);
                }
            });
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public Transaction startShort() {
        try {
            return this.execute(new Operation<Transaction>("startShort"){

                @Override
                public Transaction execute(TransactionServiceThriftClient client) throws TException {
                    return client.startShort(TransactionServiceClient.this.clientId);
                }
            });
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public Transaction startShort(final int timeout) {
        try {
            return this.execute(new Operation<Transaction>("startShort"){

                @Override
                public Transaction execute(TransactionServiceThriftClient client) throws TException {
                    return client.startShort(TransactionServiceClient.this.clientId, timeout);
                }
            });
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public boolean canCommit(Transaction tx, Collection<byte[]> changeIds) throws TransactionNotInProgressException {
        try {
            this.canCommitOrThrow(tx, changeIds);
            return true;
        }
        catch (TransactionFailureException e) {
            return false;
        }
    }

    @Override
    public void canCommitOrThrow(final Transaction tx, final Collection<byte[]> changeIds) throws TransactionFailureException {
        if (changeIds.size() > this.changeSetCountLimit) {
            throw new TransactionSizeException(String.format("Change set for transaction %d has %d entries and exceeds the limit of %d", tx.getTransactionId(), changeIds.size(), this.changeSetCountLimit));
        }
        if (changeIds.size() > this.changeSetCountThreshold) {
            LOG.warn("Change set for transaction {} has {} entries. It is recommended to limit the number of changes to {}, or to use a long-running transaction. ", new Object[]{tx.getTransactionId(), changeIds.size(), this.changeSetCountThreshold});
        }
        long byteCount = 0L;
        for (byte[] change : changeIds) {
            byteCount += (long)change.length;
        }
        if (byteCount > this.changeSetSizeLimit) {
            throw new TransactionSizeException(String.format("Change set for transaction %d has total size of %d bytes and exceeds the limit of %d bytes", tx.getTransactionId(), byteCount, this.changeSetSizeLimit));
        }
        if (byteCount > this.changeSetSizeThreshold) {
            LOG.warn("Change set for transaction {} has total size of {} bytes. It is recommended to limit the total size to {} bytes, or to use a long-running transaction. ", new Object[]{tx.getTransactionId(), byteCount, this.changeSetSizeThreshold});
        }
        try {
            this.execute(new Operation<Void>("canCommit"){

                @Override
                public Void execute(TransactionServiceThriftClient client) throws Exception {
                    client.canCommit(tx, changeIds);
                    return null;
                }
            });
        }
        catch (TransactionConflictException | TransactionNotInProgressException | TransactionSizeException e) {
            throw e;
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public boolean commit(Transaction tx) throws TransactionNotInProgressException {
        try {
            this.commitOrThrow(tx);
            return true;
        }
        catch (TransactionFailureException e) {
            return false;
        }
    }

    @Override
    public void commitOrThrow(final Transaction tx) throws TransactionFailureException {
        try {
            this.execute(new Operation<Void>("commit"){

                @Override
                public Void execute(TransactionServiceThriftClient client) throws Exception {
                    client.commit(tx.getTransactionId(), tx.getWritePointer());
                    return null;
                }
            });
        }
        catch (TransactionConflictException | TransactionNotInProgressException e) {
            throw e;
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public void abort(final Transaction tx) {
        try {
            this.execute(new Operation<Boolean>("abort"){

                @Override
                public Boolean execute(TransactionServiceThriftClient client) throws TException {
                    client.abort(tx);
                    return true;
                }
            });
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public boolean invalidate(final long tx) {
        try {
            return this.execute(new Operation<Boolean>("invalidate"){

                @Override
                public Boolean execute(TransactionServiceThriftClient client) throws TException {
                    return client.invalidate(tx);
                }
            });
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public Transaction checkpoint(final Transaction tx) throws TransactionNotInProgressException {
        try {
            return this.execute(new Operation<Transaction>("checkpoint"){

                @Override
                Transaction execute(TransactionServiceThriftClient client) throws Exception {
                    return client.checkpoint(tx);
                }
            });
        }
        catch (TransactionNotInProgressException e) {
            throw e;
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public InputStream getSnapshotInputStream() throws TransactionCouldNotTakeSnapshotException {
        try {
            return this.execute(new Operation<InputStream>("takeSnapshot"){

                @Override
                public InputStream execute(TransactionServiceThriftClient client) throws Exception {
                    return client.getSnapshotStream();
                }
            });
        }
        catch (TransactionCouldNotTakeSnapshotException e) {
            throw e;
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public String status() {
        try {
            return this.execute(new Operation<String>("status"){

                @Override
                public String execute(TransactionServiceThriftClient client) throws Exception {
                    return client.status();
                }
            });
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public void resetState() {
        try {
            this.execute(new Operation<Boolean>("resetState"){

                @Override
                public Boolean execute(TransactionServiceThriftClient client) throws TException {
                    client.resetState();
                    return true;
                }
            });
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public boolean truncateInvalidTx(final Set<Long> invalidTxIds) {
        try {
            return this.execute(new Operation<Boolean>("truncateInvalidTx"){

                @Override
                public Boolean execute(TransactionServiceThriftClient client) throws TException {
                    return client.truncateInvalidTx(invalidTxIds);
                }
            });
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public boolean truncateInvalidTxBefore(final long time) throws InvalidTruncateTimeException {
        try {
            return this.execute(new Operation<Boolean>("truncateInvalidTxBefore"){

                @Override
                public Boolean execute(TransactionServiceThriftClient client) throws Exception {
                    return client.truncateInvalidTxBefore(time);
                }
            });
        }
        catch (InvalidTruncateTimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public int getInvalidSize() {
        try {
            return this.execute(new Operation<Integer>("getInvalidSize"){

                @Override
                public Integer execute(TransactionServiceThriftClient client) throws TException {
                    return client.getInvalidSize();
                }
            });
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public void pruneNow() {
        try {
            this.execute(new Operation<Void>("pruneNow"){

                @Override
                public Void execute(TransactionServiceThriftClient client) throws TException {
                    client.pruneNow();
                    return null;
                }
            });
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    static abstract class Operation<T> {
        String name;

        Operation(String name) {
            this.name = name;
        }

        String getName() {
            return this.name;
        }

        abstract T execute(TransactionServiceThriftClient var1) throws Exception;
    }
}

