/****************************************************************************
 * binfmt/libelf/libelf_dtors.c
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <inttypes.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <assert.h>
#include <debug.h>

#include <nuttx/kmalloc.h>
#include <nuttx/binfmt/elf.h>

#include "libelf.h"

#ifdef CONFIG_BINFMT_CONSTRUCTORS

/****************************************************************************
 * Pre-processor Definitions
 ****************************************************************************/

/****************************************************************************
 * Private Types
 ****************************************************************************/

/****************************************************************************
 * Private Constant Data
 ****************************************************************************/

/****************************************************************************
 * Private Functions
 ****************************************************************************/

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * Name: elf_loaddtors
 *
 * Description:
 *  Load pointers to static destructors into an in-memory array.
 *
 * Input Parameters:
 *   loadinfo - Load state information
 *
 * Returned Value:
 *   0 (OK) is returned on success and a negated errno is returned on
 *   failure.
 *
 ****************************************************************************/

int elf_loaddtors(FAR struct elf_loadinfo_s *loadinfo)
{
  FAR Elf_Shdr *shdr;
  size_t dtorsize;
  int dtoridx;
  int ret;
  int i;

  DEBUGASSERT(loadinfo->dtors == NULL);

  /* Find the index to the section named ".dtors."  NOTE:  On old ABI system,
   * .dtors is the name of the section containing the list of destructors;
   * On newer systems, the similar section is called .fini_array.  It is
   * expected that the linker script will force the section name to be
   * ".dtors" in either case.
   */

  dtoridx = elf_findsection(loadinfo, ".dtors");
  if (dtoridx < 0)
    {
      /* This may not be a failure.  -ENOENT indicates that the file has no
       * static destructor section.
       */

      binfo("elf_findsection .dtors section failed: %d\n", dtoridx);
      return dtoridx == -ENOENT ? OK : dtoridx;
    }

  /* Now we can get a pointer to the .dtor section in the section header
   * table.
   */

  shdr = &loadinfo->shdr[dtoridx];

  /* Get the size of the .dtor section and the number of destructors that
   * will need to be called.
   */

  dtorsize         = shdr->sh_size;
  loadinfo->ndtors = dtorsize / sizeof(binfmt_dtor_t);

  binfo("dtoridx=%d dtorsize=%d sizeof(binfmt_dtor_t)=%d ndtors=%d\n",
        dtoridx, dtorsize, sizeof(binfmt_dtor_t), loadinfo->ndtors);

  /* Check if there are any destructors.  It is not an error if there
   * are none.
   */

  if (loadinfo->ndtors > 0)
    {
      /* Check an assumption that we made above */

      DEBUGASSERT(shdr->sh_size == loadinfo->ndtors * sizeof(binfmt_dtor_t));

      /* In the old ABI, the .dtors section is not allocated.  In that case,
       * we need to allocate memory to hold the .dtors and then copy the
       * from the file into the allocated memory.
       *
       * SHF_ALLOC indicates that the section requires memory during
       * execution.
       */

      if ((shdr->sh_flags & SHF_ALLOC) == 0)
        {
          /* Allocate memory to hold a copy of the .dtor section */

          loadinfo->dtoralloc = kumm_malloc(dtorsize);
          if (!loadinfo->dtoralloc)
            {
              berr("Failed to allocate memory for .dtors\n");
              return -ENOMEM;
            }

          loadinfo->dtors = (binfmt_dtor_t *)loadinfo->dtoralloc;

          /* Read the section header table into memory */

          ret = elf_read(loadinfo, (FAR uint8_t *)loadinfo->dtors, dtorsize,
                         shdr->sh_offset);
          if (ret < 0)
            {
              berr("Failed to allocate .dtors: %d\n", ret);
              return ret;
            }

          /* Fix up all of the .dtor addresses.  Since the addresses
           * do not lie in allocated memory, there will be no relocation
           * section for them.
           */

          for (i = 0; i < loadinfo->ndtors; i++)
            {
              FAR uintptr_t *ptr = (FAR uintptr_t *)
                  ((FAR void *)(&loadinfo->dtors)[i]);

              binfo("dtor %d: "
                    "%08" PRIxPTR " + %08" PRIxPTR " = %08" PRIxPTR "\n", i,
                    *ptr, loadinfo->textalloc, (*ptr + loadinfo->textalloc));

              *ptr += loadinfo->textalloc;
            }
        }
      else
        {
          /* Save the address of the .dtors (actually, .init_array) where
           * it was loaded into memory.  Since the .dtors lie in allocated
           * memory, they will be relocated via the normal mechanism.
           */

          loadinfo->dtors = (binfmt_dtor_t *)shdr->sh_addr;
        }
    }

  return OK;
}

#endif /* CONFIG_BINFMT_CONSTRUCTORS */
