/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.web.security.knox;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSObject;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.RSASSAVerifier;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.SignedJWT;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.web.security.InvalidAuthenticationException;
import org.apache.nifi.web.security.knox.KnoxConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KnoxService {
    private static final Logger logger = LoggerFactory.getLogger(KnoxService.class);
    private KnoxConfiguration configuration;
    private JWSVerifier verifier;
    private String knoxUrl;
    private Set<String> audiences;

    public KnoxService(KnoxConfiguration configuration) {
        this.configuration = configuration;
        if (configuration.isKnoxEnabled()) {
            this.knoxUrl = configuration.getKnoxUrl();
            if (StringUtils.isBlank((CharSequence)this.knoxUrl)) {
                throw new RuntimeException("Knox URL is required when Apache Knox SSO support is enabled.");
            }
            if (StringUtils.isBlank((CharSequence)configuration.getKnoxCookieName())) {
                throw new RuntimeException("Knox Cookie Name is required when Apache Knox SSO support is enabled.");
            }
            this.verifier = new RSASSAVerifier(configuration.getKnoxPublicKey());
            this.audiences = configuration.getAudiences();
        }
    }

    public boolean isKnoxEnabled() {
        return this.configuration.isKnoxEnabled();
    }

    public String getKnoxUrl() {
        if (!this.configuration.isKnoxEnabled()) {
            throw new IllegalStateException("Apache Knox SSO is not enabled.");
        }
        return this.knoxUrl;
    }

    public String getAuthenticationFromToken(String jwt) throws ParseException, JOSEException {
        if (!this.configuration.isKnoxEnabled()) {
            throw new IllegalStateException("Apache Knox SSO is not enabled.");
        }
        SignedJWT signedJwt = SignedJWT.parse((String)jwt);
        if (this.validateToken(signedJwt)) {
            JWTClaimsSet claimsSet = signedJwt.getJWTClaimsSet();
            if (claimsSet == null) {
                logger.info("Claims set is missing from Knox JWT.");
                throw new InvalidAuthenticationException("The Knox JWT token is not valid.");
            }
            return claimsSet.getSubject();
        }
        throw new InvalidAuthenticationException("The Knox JWT token is not valid.");
    }

    private boolean validateToken(SignedJWT jwtToken) throws JOSEException, ParseException {
        boolean validSignature = this.validateSignature(jwtToken);
        boolean validAudience = this.validateAudience(jwtToken);
        boolean notExpired = this.validateExpiration(jwtToken);
        return validSignature && validAudience && notExpired;
    }

    private boolean validateSignature(SignedJWT jwtToken) throws JOSEException {
        boolean valid = false;
        if (JWSObject.State.SIGNED.equals((Object)jwtToken.getState()) && jwtToken.getSignature() != null) {
            valid = jwtToken.verify(this.verifier);
        }
        if (!valid) {
            logger.error("The Knox JWT has an invalid signature.");
        }
        return valid;
    }

    private boolean validateAudience(SignedJWT jwtToken) throws ParseException {
        if (this.audiences == null) {
            return true;
        }
        JWTClaimsSet claimsSet = jwtToken.getJWTClaimsSet();
        if (claimsSet == null) {
            logger.error("Claims set is missing from Knox JWT.");
            return false;
        }
        List tokenAudiences = claimsSet.getAudience();
        if (tokenAudiences == null) {
            logger.error("Audience is missing from the Knox JWT.");
            return false;
        }
        boolean valid = false;
        for (String tokenAudience : tokenAudiences) {
            if (!this.audiences.contains(tokenAudience)) continue;
            valid = true;
            break;
        }
        if (!valid) {
            logger.error(String.format("The Knox JWT does not have the required audience(s). Required one of [%s]. Present in JWT [%s].", StringUtils.join(this.audiences, (String)", "), StringUtils.join((Iterable)tokenAudiences, (String)", ")));
        }
        return valid;
    }

    private boolean validateExpiration(SignedJWT jwtToken) throws ParseException {
        boolean valid = false;
        JWTClaimsSet claimsSet = jwtToken.getJWTClaimsSet();
        if (claimsSet == null) {
            logger.error("Claims set is missing from Knox JWT.");
            return false;
        }
        Date now = new Date();
        Date expiration = claimsSet.getExpirationTime();
        if (expiration == null || now.before(expiration)) {
            valid = true;
        }
        if (!valid) {
            logger.error("The Knox JWT is expired.");
        }
        return valid;
    }
}

