/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.web.security.jwt.key.service;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.nifi.components.state.Scope;
import org.apache.nifi.components.state.StateManager;
import org.apache.nifi.components.state.StateMap;
import org.apache.nifi.web.security.jwt.key.service.VerificationKey;
import org.apache.nifi.web.security.jwt.key.service.VerificationKeyService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StandardVerificationKeyService
implements VerificationKeyService {
    private static final Logger LOGGER = LoggerFactory.getLogger(StandardVerificationKeyService.class);
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper().registerModule((Module)new JavaTimeModule());
    private static final Scope SCOPE = Scope.LOCAL;
    private final StateManager stateManager;

    public StandardVerificationKeyService(StateManager stateManager) {
        this.stateManager = stateManager;
    }

    @Override
    public Optional<Key> findById(String id) {
        Optional<String> serializedKey = this.findSerializedKey(id);
        return serializedKey.map(this::getVerificationKey).map(this::getKey);
    }

    @Override
    public synchronized void deleteExpired() {
        Map state = this.getStateMap().toMap();
        Instant now = Instant.now();
        Map<String, String> updatedState = state.values().stream().map(this::getVerificationKey).filter(verificationKey -> verificationKey.getExpiration().isAfter(now)).collect(Collectors.toMap(VerificationKey::getId, this::serializeVerificationKey));
        if (updatedState.equals(state)) {
            LOGGER.debug("Expired Verification Keys not found");
        } else {
            try {
                this.stateManager.setState(updatedState, SCOPE);
            }
            catch (IOException e) {
                throw new UncheckedIOException("Delete Expired Verification Keys Failed", e);
            }
            LOGGER.debug("Delete Expired Verification Keys Completed: Keys Before [{}] Keys After [{}]", (Object)state.size(), (Object)updatedState.size());
        }
    }

    @Override
    public void save(String id, Key key, Instant expiration) {
        VerificationKey verificationKey = new VerificationKey();
        verificationKey.setId(id);
        verificationKey.setEncoded(key.getEncoded());
        verificationKey.setAlgorithm(key.getAlgorithm());
        verificationKey.setExpiration(expiration);
        this.setVerificationKey(verificationKey);
    }

    @Override
    public void setExpiration(String id, Instant expiration) {
        Optional<String> serializedKey = this.findSerializedKey(id);
        if (serializedKey.isPresent()) {
            VerificationKey verificationKey = this.getVerificationKey(serializedKey.get());
            verificationKey.setExpiration(expiration);
            this.setVerificationKey(verificationKey);
        }
    }

    private synchronized void setVerificationKey(VerificationKey verificationKey) {
        try {
            String serialized = this.serializeVerificationKey(verificationKey);
            HashMap<String, String> state = new HashMap<String, String>(this.getStateMap().toMap());
            state.put(verificationKey.getId(), serialized);
            this.stateManager.setState(state, SCOPE);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Set Verification Key State Failed", e);
        }
        LOGGER.debug("Stored Verification Key [{}] Expiration [{}]", (Object)verificationKey.getId(), (Object)verificationKey.getExpiration());
    }

    private Optional<String> findSerializedKey(String id) {
        StateMap stateMap = this.getStateMap();
        return Optional.ofNullable(stateMap.get(id));
    }

    private String serializeVerificationKey(VerificationKey verificationKey) {
        try {
            return OBJECT_MAPPER.writeValueAsString((Object)verificationKey);
        }
        catch (JsonProcessingException e) {
            throw new UncheckedIOException("Serialize Verification Key Failed", (IOException)((Object)e));
        }
    }

    private VerificationKey getVerificationKey(String serialized) {
        try {
            return (VerificationKey)OBJECT_MAPPER.readValue(serialized, VerificationKey.class);
        }
        catch (JsonProcessingException e) {
            throw new UncheckedIOException("Read Verification Key Failed", (IOException)((Object)e));
        }
    }

    private Key getKey(VerificationKey verificationKey) {
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(verificationKey.getEncoded());
        String algorithm = verificationKey.getAlgorithm();
        try {
            KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
            return keyFactory.generatePublic(keySpec);
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            String message = String.format("Parsing Encoded Key [%s] Algorithm [%s] Failed", verificationKey.getId(), algorithm);
            throw new IllegalStateException(message, e);
        }
    }

    private StateMap getStateMap() {
        try {
            return this.stateManager.getState(SCOPE);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Get State Failed", e);
        }
    }
}

