/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.orchestra.requestParameterProvider;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.myfaces.orchestra.lib._UrlMatcher;

import java.io.IOException;

/**
 * This filter wraps each {@link HttpServletResponse} in a {@link RequestParameterResponseWrapper},
 * meaning that every call to response.encodeURL() gets forwarded to the RequestParameterProviderManager.
 */
public class RequestParameterServletFilter implements Filter
{
    public final static String REQUEST_PARAM_FILTER_CALLED =
        RequestParameterServletFilter.class.getName() + ".CALLED";
    public final static String REQUEST_PARAM_RESPONSE_WRAPPED =
        RequestParameterServletFilter.class.getName() + ".WRAPPED";

    private _UrlMatcher urlMatcher;

    public RequestParameterServletFilter()
    {
    }

    public void init(FilterConfig filterConfig)
    {
        urlMatcher = new _UrlMatcher(filterConfig);
    }

    public void doFilter(ServletRequest servletRequest,
                         ServletResponse servletResponse,
                         FilterChain filterChain) throws IOException, ServletException
    {
        if (urlMatcher.accept(servletRequest) && servletResponse instanceof HttpServletResponse)
        {
            HttpServletRequest httpServletRequest = (HttpServletRequest) servletRequest;
            if (!Boolean.TRUE.equals(httpServletRequest.getAttribute(REQUEST_PARAM_FILTER_CALLED)))
            {
                httpServletRequest.setAttribute(REQUEST_PARAM_FILTER_CALLED, Boolean.TRUE);
                servletResponse = new RequestParameterResponseWrapper((HttpServletResponse) servletResponse);
            }
        }

        filterChain.doFilter(servletRequest, servletResponse);
    }

    public void destroy()
    {
    }
}
