/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.memory;

import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.common.collect.Lists;
import org.elasticsearch.common.collect.Maps;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.engine.EngineClosedException;
import org.elasticsearch.index.engine.FlushNotAllowedEngineException;
import org.elasticsearch.index.service.IndexService;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.service.IndexShard;
import org.elasticsearch.index.shard.service.InternalIndexShard;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.indices.IndicesLifecycle;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.threadpool.ThreadPool;

public class IndexingMemoryController
extends AbstractLifecycleComponent<IndexingMemoryController> {
    private final ThreadPool threadPool;
    private final IndicesService indicesService;
    private final ByteSizeValue indexingBuffer;
    private final ByteSizeValue minShardIndexBufferSize;
    private final ByteSizeValue maxShardIndexBufferSize;
    private final ByteSizeValue translogBuffer;
    private final ByteSizeValue minShardTranslogBufferSize;
    private final ByteSizeValue maxShardTranslogBufferSize;
    private final TimeValue inactiveTime;
    private final TimeValue interval;
    private final AtomicBoolean shardsCreatedOrDeleted = new AtomicBoolean();
    private final Listener listener = new Listener();
    private final Map<ShardId, ShardIndexingStatus> shardsIndicesStatus = Maps.newHashMap();
    private volatile ScheduledFuture scheduler;
    private final Object mutex = new Object();

    @Inject
    public IndexingMemoryController(Settings settings, ThreadPool threadPool, IndicesService indicesService) {
        super(settings);
        ByteSizeValue translogBuffer;
        ByteSizeValue indexingBuffer;
        this.threadPool = threadPool;
        this.indicesService = indicesService;
        String indexingBufferSetting = this.componentSettings.get("index_buffer_size", "10%");
        if (indexingBufferSetting.endsWith("%")) {
            double percent = Double.parseDouble(indexingBufferSetting.substring(0, indexingBufferSetting.length() - 1));
            indexingBuffer = new ByteSizeValue((long)((double)JvmInfo.jvmInfo().mem().heapMax().bytes() * (percent / 100.0)));
            ByteSizeValue minIndexingBuffer = this.componentSettings.getAsBytesSize("min_index_buffer_size", new ByteSizeValue(48L, ByteSizeUnit.MB));
            ByteSizeValue maxIndexingBuffer = this.componentSettings.getAsBytesSize("max_index_buffer_size", null);
            if (indexingBuffer.bytes() < minIndexingBuffer.bytes()) {
                indexingBuffer = minIndexingBuffer;
            }
            if (maxIndexingBuffer != null && indexingBuffer.bytes() > maxIndexingBuffer.bytes()) {
                indexingBuffer = maxIndexingBuffer;
            }
        } else {
            indexingBuffer = ByteSizeValue.parseBytesSizeValue(indexingBufferSetting, null);
        }
        this.indexingBuffer = indexingBuffer;
        this.minShardIndexBufferSize = this.componentSettings.getAsBytesSize("min_shard_index_buffer_size", new ByteSizeValue(4L, ByteSizeUnit.MB));
        this.maxShardIndexBufferSize = this.componentSettings.getAsBytesSize("max_shard_index_buffer_size", new ByteSizeValue(512L, ByteSizeUnit.MB));
        String translogBufferSetting = this.componentSettings.get("translog_buffer_size", "1%");
        if (translogBufferSetting.endsWith("%")) {
            double percent = Double.parseDouble(translogBufferSetting.substring(0, translogBufferSetting.length() - 1));
            translogBuffer = new ByteSizeValue((long)((double)JvmInfo.jvmInfo().mem().heapMax().bytes() * (percent / 100.0)));
            ByteSizeValue minTranslogBuffer = this.componentSettings.getAsBytesSize("min_translog_buffer_size", new ByteSizeValue(256L, ByteSizeUnit.KB));
            ByteSizeValue maxTranslogBuffer = this.componentSettings.getAsBytesSize("max_translog_buffer_size", null);
            if (translogBuffer.bytes() < minTranslogBuffer.bytes()) {
                translogBuffer = minTranslogBuffer;
            }
            if (maxTranslogBuffer != null && translogBuffer.bytes() > maxTranslogBuffer.bytes()) {
                translogBuffer = maxTranslogBuffer;
            }
        } else {
            translogBuffer = ByteSizeValue.parseBytesSizeValue(translogBufferSetting, null);
        }
        this.translogBuffer = translogBuffer;
        this.minShardTranslogBufferSize = this.componentSettings.getAsBytesSize("min_shard_translog_buffer_size", new ByteSizeValue(2L, ByteSizeUnit.KB));
        this.maxShardTranslogBufferSize = this.componentSettings.getAsBytesSize("max_shard_translog_buffer_size", new ByteSizeValue(64L, ByteSizeUnit.KB));
        this.inactiveTime = this.componentSettings.getAsTime("shard_inactive_time", TimeValue.timeValueMinutes(30L));
        this.interval = this.componentSettings.getAsTime("interval", TimeValue.timeValueSeconds(30L));
        this.logger.debug("using index_buffer_size [{}], with min_shard_index_buffer_size [{}], max_shard_index_buffer_size [{}], shard_inactive_time [{}]", this.indexingBuffer, this.minShardIndexBufferSize, this.maxShardIndexBufferSize, this.inactiveTime);
    }

    @Override
    protected void doStart() throws ElasticsearchException {
        this.indicesService.indicesLifecycle().addListener(this.listener);
        this.scheduler = this.threadPool.scheduleWithFixedDelay(new ShardsIndicesStatusChecker(), this.interval);
    }

    @Override
    protected void doStop() throws ElasticsearchException {
        this.indicesService.indicesLifecycle().removeListener(this.listener);
        if (this.scheduler != null) {
            this.scheduler.cancel(false);
            this.scheduler = null;
        }
    }

    @Override
    protected void doClose() throws ElasticsearchException {
    }

    private void calcAndSetShardBuffers(String reason) {
        ByteSizeValue shardTranslogBufferSize;
        int shardsCount = this.countShards();
        if (shardsCount == 0) {
            return;
        }
        ByteSizeValue shardIndexingBufferSize = new ByteSizeValue(this.indexingBuffer.bytes() / (long)shardsCount);
        if (shardIndexingBufferSize.bytes() < this.minShardIndexBufferSize.bytes()) {
            shardIndexingBufferSize = this.minShardIndexBufferSize;
        }
        if (shardIndexingBufferSize.bytes() > this.maxShardIndexBufferSize.bytes()) {
            shardIndexingBufferSize = this.maxShardIndexBufferSize;
        }
        if ((shardTranslogBufferSize = new ByteSizeValue(this.translogBuffer.bytes() / (long)shardsCount)).bytes() < this.minShardTranslogBufferSize.bytes()) {
            shardTranslogBufferSize = this.minShardTranslogBufferSize;
        }
        if (shardTranslogBufferSize.bytes() > this.maxShardTranslogBufferSize.bytes()) {
            shardTranslogBufferSize = this.maxShardTranslogBufferSize;
        }
        this.logger.debug("recalculating shard indexing buffer (reason={}), total is [{}] with [{}] active shards, each shard set to indexing=[{}], translog=[{}]", reason, this.indexingBuffer, shardsCount, shardIndexingBufferSize, shardTranslogBufferSize);
        for (IndexService indexService : this.indicesService) {
            for (IndexShard indexShard : indexService) {
                ShardIndexingStatus status = this.shardsIndicesStatus.get(indexShard.shardId());
                if (status != null && status.inactiveIndexing) continue;
                try {
                    ((InternalIndexShard)indexShard).engine().updateIndexingBufferSize(shardIndexingBufferSize);
                    ((InternalIndexShard)indexShard).translog().updateBuffer(shardTranslogBufferSize);
                }
                catch (EngineClosedException e) {
                }
                catch (FlushNotAllowedEngineException e) {
                }
                catch (Exception e) {
                    this.logger.warn("failed to set shard [{}][{}] index buffer to [{}]", indexShard.shardId().index().name(), indexShard.shardId().id(), shardIndexingBufferSize);
                }
            }
        }
    }

    private int countShards() {
        int shardsCount = 0;
        for (IndexService indexService : this.indicesService) {
            for (IndexShard indexShard : indexService) {
                ShardIndexingStatus status = this.shardsIndicesStatus.get(indexShard.shardId());
                if (status != null && status.inactiveIndexing) continue;
                ++shardsCount;
            }
        }
        return shardsCount;
    }

    static class ShardIndexingStatus {
        long translogId = -1L;
        int translogNumberOfOperations = -1;
        boolean inactiveIndexing = false;
        long time = -1L;

        ShardIndexingStatus() {
        }
    }

    class Listener
    extends IndicesLifecycle.Listener {
        Listener() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void afterIndexShardCreated(IndexShard indexShard) {
            Object object = IndexingMemoryController.this.mutex;
            synchronized (object) {
                IndexingMemoryController.this.shardsIndicesStatus.put(indexShard.shardId(), new ShardIndexingStatus());
                IndexingMemoryController.this.shardsCreatedOrDeleted.set(true);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void afterIndexShardClosed(ShardId shardId) {
            Object object = IndexingMemoryController.this.mutex;
            synchronized (object) {
                IndexingMemoryController.this.shardsIndicesStatus.remove(shardId);
                IndexingMemoryController.this.shardsCreatedOrDeleted.set(true);
            }
        }
    }

    class ShardsIndicesStatusChecker
    implements Runnable {
        ShardsIndicesStatusChecker() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            Object object = IndexingMemoryController.this.mutex;
            synchronized (object) {
                boolean activeInactiveStatusChanges = false;
                ArrayList<IndexShard> activeToInactiveIndexingShards = Lists.newArrayList();
                ArrayList<IndexShard> inactiveToActiveIndexingShards = Lists.newArrayList();
                for (IndexService indexService : IndexingMemoryController.this.indicesService) {
                    for (IndexShard indexShard : indexService) {
                        long time = IndexingMemoryController.this.threadPool.estimatedTimeInMillis();
                        Translog translog = ((InternalIndexShard)indexShard).translog();
                        ShardIndexingStatus status = (ShardIndexingStatus)IndexingMemoryController.this.shardsIndicesStatus.get(indexShard.shardId());
                        if (status == null) continue;
                        if (status.translogId == translog.currentId() && translog.estimatedNumberOfOperations() == 0) {
                            if (status.time == -1L) {
                                status.time = time;
                            }
                            if (!status.inactiveIndexing && time - status.time > IndexingMemoryController.this.inactiveTime.millis() && indexShard.mergeStats().getCurrent() == 0L) {
                                activeToInactiveIndexingShards.add(indexShard);
                                status.inactiveIndexing = true;
                                activeInactiveStatusChanges = true;
                                IndexingMemoryController.this.logger.debug("marking shard [{}][{}] as inactive (inactive_time[{}]) indexing wise, setting size to [{}]", indexShard.shardId().index().name(), indexShard.shardId().id(), IndexingMemoryController.this.inactiveTime, Engine.INACTIVE_SHARD_INDEXING_BUFFER);
                            }
                        } else {
                            if (status.inactiveIndexing) {
                                inactiveToActiveIndexingShards.add(indexShard);
                                status.inactiveIndexing = false;
                                activeInactiveStatusChanges = true;
                                IndexingMemoryController.this.logger.debug("marking shard [{}][{}] as active indexing wise", indexShard.shardId().index().name(), indexShard.shardId().id());
                            }
                            status.time = -1L;
                        }
                        status.translogId = translog.currentId();
                        status.translogNumberOfOperations = translog.estimatedNumberOfOperations();
                    }
                }
                for (IndexShard indexShard : activeToInactiveIndexingShards) {
                    try {
                        ((InternalIndexShard)indexShard).engine().updateIndexingBufferSize(Engine.INACTIVE_SHARD_INDEXING_BUFFER);
                        ((InternalIndexShard)indexShard).translog().updateBuffer(Translog.INACTIVE_SHARD_TRANSLOG_BUFFER);
                    }
                    catch (EngineClosedException e) {
                    }
                    catch (FlushNotAllowedEngineException e) {}
                }
                boolean shardsCreatedOrDeleted = IndexingMemoryController.this.shardsCreatedOrDeleted.compareAndSet(true, false);
                if (shardsCreatedOrDeleted || activeInactiveStatusChanges) {
                    IndexingMemoryController.this.calcAndSetShardBuffers("active/inactive[" + activeInactiveStatusChanges + "] created/deleted[" + shardsCreatedOrDeleted + "]");
                }
            }
        }
    }
}

