/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.linkis.server;

import org.apache.linkis.utils.LinkisSpringUtils;

import org.springframework.web.servlet.HandlerInterceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PerformanceInterceptor implements HandlerInterceptor {

  private static final Logger logger = LoggerFactory.getLogger(PerformanceInterceptor.class);

  @Override
  public boolean preHandle(
      HttpServletRequest request, HttpServletResponse response, Object handler) {
    request.setAttribute("Linkis_startTime", System.currentTimeMillis());
    return true;
  }

  @Override
  public void afterCompletion(
      HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) {
    Object startObject = request.getAttribute("Linkis_startTime");
    if (null != startObject) {
      long startTime = (Long) startObject;
      long endTime = System.currentTimeMillis();
      long executeTime = endTime - startTime;
      logger.info(
          "Request client address：{} request URL: {}  Method: {} taken: {} ms",
          LinkisSpringUtils.getClientIP(request),
          request.getRequestURI(),
          request.getMethod(),
          executeTime);
    }
  }
}
