"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const fs = require("fs");
const Observable_1 = require("rxjs/Observable");
const empty_1 = require("rxjs/observable/empty");
const from_1 = require("rxjs/observable/from");
const of_1 = require("rxjs/observable/of");
const concat_1 = require("rxjs/operators/concat");
const concatMap_1 = require("rxjs/operators/concatMap");
const ignoreElements_1 = require("rxjs/operators/ignoreElements");
const map_1 = require("rxjs/operators/map");
const mergeMap_1 = require("rxjs/operators/mergeMap");
const publish_1 = require("rxjs/operators/publish");
const refCount_1 = require("rxjs/operators/refCount");
const src_1 = require("../src");
const { FSWatcher } = require('chokidar');
function _callFs(fn, ...args) {
    return new Observable_1.Observable(obs => {
        fn(...args, (err, result) => {
            if (err) {
                obs.error(err);
            }
            else {
                obs.next(result);
                obs.complete();
            }
        });
    });
}
/**
 * An implementation of the Virtual FS using Node as the background. There are two versions; one
 * synchronous and one asynchronous.
 */
class NodeJsAsyncHost {
    _getSystemPath(path) {
        if (process.platform.startsWith('win32')) {
            return src_1.asWindowsPath(path);
        }
        else {
            return src_1.asPosixPath(path);
        }
    }
    get capabilities() {
        return { synchronous: false };
    }
    write(path, content) {
        return new Observable_1.Observable(obs => {
            // Create folders if necessary.
            const _createDir = (path) => {
                if (fs.existsSync(this._getSystemPath(path))) {
                    return;
                }
                if (src_1.dirname(path) === path) {
                    throw new Error();
                }
                _createDir(src_1.dirname(path));
                fs.mkdirSync(this._getSystemPath(path));
            };
            _createDir(src_1.dirname(path));
            _callFs(fs.writeFile, this._getSystemPath(path), new Uint8Array(content)).subscribe(obs);
        });
    }
    read(path) {
        return _callFs(fs.readFile, this._getSystemPath(path)).pipe(map_1.map(buffer => new Uint8Array(buffer).buffer));
    }
    delete(path) {
        return this.isDirectory(path).pipe(mergeMap_1.mergeMap(isDirectory => {
            if (isDirectory) {
                const allFiles = [];
                const allDirs = [];
                const _recurseList = (path) => {
                    for (const fragment of fs.readdirSync(this._getSystemPath(path))) {
                        if (fs.statSync(this._getSystemPath(src_1.join(path, fragment))).isDirectory()) {
                            _recurseList(src_1.join(path, fragment));
                            allDirs.push(src_1.join(path, fragment));
                        }
                        else {
                            allFiles.push(src_1.join(path, fragment));
                        }
                    }
                };
                _recurseList(path);
                return from_1.from(allFiles)
                    .pipe(mergeMap_1.mergeMap(p => _callFs(fs.unlink, this._getSystemPath(p))), ignoreElements_1.ignoreElements(), concat_1.concat(from_1.from(allDirs).pipe(concatMap_1.concatMap(p => _callFs(fs.rmdir, this._getSystemPath(p))))), map_1.map(() => { }));
            }
            else {
                return _callFs(fs.unlink, this._getSystemPath(path));
            }
        }));
    }
    rename(from, to) {
        return _callFs(fs.rename, this._getSystemPath(from), this._getSystemPath(to));
    }
    list(path) {
        return _callFs(fs.readdir, this._getSystemPath(path)).pipe(map_1.map(names => names.map(name => src_1.fragment(name))));
    }
    exists(path) {
        // Exists is a special case because it cannot error.
        return new Observable_1.Observable(obs => {
            fs.exists(path, exists => {
                obs.next(exists);
                obs.complete();
            });
        });
    }
    isDirectory(path) {
        return _callFs(fs.stat, this._getSystemPath(path)).pipe(map_1.map(stat => stat.isDirectory()));
    }
    isFile(path) {
        return _callFs(fs.stat, this._getSystemPath(path)).pipe(map_1.map(stat => stat.isDirectory()));
    }
    // Some hosts may not support stats.
    stats(path) {
        return _callFs(fs.stat, this._getSystemPath(path));
    }
    // Some hosts may not support watching.
    watch(path, _options) {
        return new Observable_1.Observable(obs => {
            const watcher = new FSWatcher({ persistent: true }).add(this._getSystemPath(path));
            watcher
                .on('change', path => {
                obs.next({
                    path: src_1.normalize(path),
                    time: new Date(),
                    type: 0 /* Changed */,
                });
            })
                .on('add', path => {
                obs.next({
                    path: src_1.normalize(path),
                    time: new Date(),
                    type: 1 /* Created */,
                });
            })
                .on('unlink', path => {
                obs.next({
                    path: src_1.normalize(path),
                    time: new Date(),
                    type: 2 /* Deleted */,
                });
            });
            return () => watcher.close();
        }).pipe(publish_1.publish(), refCount_1.refCount());
    }
}
exports.NodeJsAsyncHost = NodeJsAsyncHost;
/**
 * An implementation of the Virtual FS using Node as the backend, synchronously.
 */
class NodeJsSyncHost {
    _getSystemPath(path) {
        if (process.platform.startsWith('win32')) {
            return src_1.asWindowsPath(path);
        }
        else {
            return src_1.asPosixPath(path);
        }
    }
    get capabilities() {
        return { synchronous: true };
    }
    write(path, content) {
        // Create folders if necessary.
        const _createDir = (path) => {
            if (fs.existsSync(this._getSystemPath(path))) {
                return;
            }
            _createDir(src_1.dirname(path));
            fs.mkdirSync(path);
        };
        _createDir(src_1.dirname(path));
        fs.writeFileSync(this._getSystemPath(path), new Uint8Array(content));
        return empty_1.empty();
    }
    read(path) {
        const buffer = fs.readFileSync(this._getSystemPath(path));
        return of_1.of(new Uint8Array(buffer).buffer);
    }
    delete(path) {
        if (this.isDirectory(path)) {
            // Since this is synchronous, we can recurse and safely ignore the result.
            for (const name of fs.readdirSync(this._getSystemPath(path))) {
                this.delete(src_1.join(path, name));
            }
            fs.rmdirSync(this._getSystemPath(path));
        }
        else {
            fs.unlinkSync(this._getSystemPath(path));
        }
        return empty_1.empty();
    }
    rename(from, to) {
        fs.renameSync(this._getSystemPath(from), this._getSystemPath(to));
        return empty_1.empty();
    }
    list(path) {
        return of_1.of(fs.readdirSync(this._getSystemPath(path))).pipe(map_1.map(names => names.map(name => src_1.fragment(name))));
    }
    exists(path) {
        return of_1.of(fs.existsSync(this._getSystemPath(path)));
    }
    isDirectory(path) {
        // tslint:disable-next-line:non-null-operator
        return this.stats(path).pipe(map_1.map(stat => stat.isDirectory()));
    }
    isFile(path) {
        // tslint:disable-next-line:non-null-operator
        return this.stats(path).pipe(map_1.map(stat => stat.isFile()));
    }
    // Some hosts may not support stats.
    stats(path) {
        return of_1.of(fs.statSync(this._getSystemPath(path)));
    }
    // Some hosts may not support watching.
    watch(path, _options) {
        return new Observable_1.Observable(obs => {
            const opts = { persistent: false };
            const watcher = new FSWatcher(opts).add(this._getSystemPath(path));
            watcher
                .on('change', path => {
                obs.next({
                    path: src_1.normalize(path),
                    time: new Date(),
                    type: 0 /* Changed */,
                });
            })
                .on('add', path => {
                obs.next({
                    path: src_1.normalize(path),
                    time: new Date(),
                    type: 1 /* Created */,
                });
            })
                .on('unlink', path => {
                obs.next({
                    path: src_1.normalize(path),
                    time: new Date(),
                    type: 2 /* Deleted */,
                });
            });
            return () => watcher.close();
        }).pipe(publish_1.publish(), refCount_1.refCount());
    }
}
exports.NodeJsSyncHost = NodeJsSyncHost;
//# sourceMappingURL=data:application/json;base64,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