/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.image.publisher;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.image.MetadataDelta;
import org.apache.kafka.image.MetadataImage;
import org.apache.kafka.image.loader.LoaderManifest;
import org.apache.kafka.image.loader.LogDeltaManifest;
import org.apache.kafka.image.loader.SnapshotManifest;
import org.apache.kafka.image.publisher.MetadataPublisher;
import org.apache.kafka.queue.EventQueue;
import org.apache.kafka.queue.KafkaEventQueue;
import org.apache.kafka.server.fault.FaultHandler;
import org.slf4j.Logger;

public class SnapshotGenerator
implements MetadataPublisher {
    private final int nodeId;
    private final Time time;
    private final Emitter emitter;
    private final Logger log;
    private final FaultHandler faultHandler;
    private final long maxBytesSinceLastSnapshot;
    private final long maxTimeSinceLastSnapshotNs;
    private final AtomicReference<String> disabledReason;
    private final EventQueue eventQueue;
    private long bytesSinceLastSnapshot;
    private long lastSnapshotTimeNs;

    private SnapshotGenerator(int nodeId, Time time, Emitter emitter, FaultHandler faultHandler, long maxBytesSinceLastSnapshot, long maxTimeSinceLastSnapshotNs, AtomicReference<String> disabledReason, String threadNamePrefix) {
        this.nodeId = nodeId;
        this.time = time;
        this.emitter = emitter;
        this.faultHandler = faultHandler;
        this.maxBytesSinceLastSnapshot = maxBytesSinceLastSnapshot;
        this.maxTimeSinceLastSnapshotNs = maxTimeSinceLastSnapshotNs;
        LogContext logContext = new LogContext("[SnapshotGenerator id=" + nodeId + "] ");
        this.log = logContext.logger(SnapshotGenerator.class);
        this.disabledReason = disabledReason;
        this.eventQueue = new KafkaEventQueue(time, logContext, threadNamePrefix + "snapshot-generator-");
        this.resetSnapshotCounters();
        this.log.debug("Starting SnapshotGenerator.");
    }

    @Override
    public String name() {
        return "SnapshotGenerator";
    }

    void resetSnapshotCounters() {
        this.bytesSinceLastSnapshot = 0L;
        this.lastSnapshotTimeNs = this.time.nanoseconds();
    }

    @Override
    public void onMetadataUpdate(MetadataDelta delta, MetadataImage newImage, LoaderManifest manifest) {
        switch (manifest.type()) {
            case LOG_DELTA: {
                this.publishLogDelta(delta, newImage, (LogDeltaManifest)manifest);
                break;
            }
            case SNAPSHOT: {
                this.publishSnapshot(delta, newImage, (SnapshotManifest)manifest);
            }
        }
    }

    void publishSnapshot(MetadataDelta delta, MetadataImage newImage, SnapshotManifest manifest) {
        this.log.debug("Resetting the snapshot counters because we just read {}.", (Object)newImage.provenance().snapshotName());
        this.resetSnapshotCounters();
    }

    void publishLogDelta(MetadataDelta delta, MetadataImage newImage, LogDeltaManifest manifest) {
        this.bytesSinceLastSnapshot += manifest.numBytes();
        if (this.bytesSinceLastSnapshot >= this.maxBytesSinceLastSnapshot) {
            if (this.eventQueue.isEmpty()) {
                this.maybeScheduleEmit("we have replayed at least " + this.maxBytesSinceLastSnapshot + " bytes", newImage, manifest.provenance().isOffsetBatchAligned());
            } else if (this.log.isTraceEnabled()) {
                this.log.trace("Not scheduling bytes-based snapshot because event queue is not empty yet.");
            }
        } else if (this.maxTimeSinceLastSnapshotNs != 0L && this.time.nanoseconds() - this.lastSnapshotTimeNs >= this.maxTimeSinceLastSnapshotNs) {
            if (this.eventQueue.isEmpty()) {
                this.maybeScheduleEmit("we have waited at least " + TimeUnit.NANOSECONDS.toMinutes(this.maxTimeSinceLastSnapshotNs) + " minute(s)", newImage, manifest.provenance().isOffsetBatchAligned());
            } else if (this.log.isTraceEnabled()) {
                this.log.trace("Not scheduling time-based snapshot because event queue is not empty yet.");
            }
        } else if (this.log.isTraceEnabled()) {
            this.log.trace("Neither time-based nor bytes-based criteria are met; not scheduling snapshot.");
        }
    }

    void maybeScheduleEmit(String reason, MetadataImage image, boolean isOffsetBatchAligned) {
        String currentDisabledReason = this.disabledReason.get();
        if (currentDisabledReason != null) {
            this.log.error("Not emitting {} despite the fact that {} because snapshots are disabled; {}", new Object[]{image.provenance().snapshotName(), reason, currentDisabledReason});
        } else if (!isOffsetBatchAligned) {
            this.log.debug("Not emitting {} despite the fact that {} because snapshots are disabled; {}", new Object[]{image.provenance().snapshotName(), reason, "metadata image is not batch aligned"});
        } else {
            this.eventQueue.append(() -> {
                this.resetSnapshotCounters();
                this.log.info("Creating new KRaft snapshot file {} because {}.", (Object)image.provenance().snapshotName(), (Object)reason);
                try {
                    this.emitter.maybeEmit(image);
                }
                catch (Throwable e) {
                    this.faultHandler.handleFault("KRaft snapshot file generation error", e);
                }
            });
        }
    }

    public void beginShutdown() {
        this.log.debug("Beginning shutdown of SnapshotGenerator.");
        this.disabledReason.compareAndSet(null, "we are shutting down");
        this.eventQueue.beginShutdown("beginShutdown");
    }

    @Override
    public void close() throws InterruptedException {
        this.eventQueue.beginShutdown("close");
        this.log.debug("Closing SnapshotGenerator.");
        this.eventQueue.close();
    }

    public static interface Emitter {
        public void maybeEmit(MetadataImage var1);
    }

    public static class Builder {
        private final Emitter emitter;
        private int nodeId = 0;
        private Time time = Time.SYSTEM;
        private FaultHandler faultHandler = (m, e) -> null;
        private long maxBytesSinceLastSnapshot = 0x6400000L;
        private long maxTimeSinceLastSnapshotNs = TimeUnit.DAYS.toNanos(1L);
        private AtomicReference<String> disabledReason = null;
        private String threadNamePrefix = "";

        public Builder(Emitter emitter) {
            this.emitter = emitter;
        }

        public Builder setNodeId(int nodeId) {
            this.nodeId = nodeId;
            return this;
        }

        public Builder setTime(Time time) {
            this.time = time;
            return this;
        }

        public Builder setFaultHandler(FaultHandler faultHandler) {
            this.faultHandler = faultHandler;
            return this;
        }

        public Builder setMaxBytesSinceLastSnapshot(long maxBytesSinceLastSnapshot) {
            this.maxBytesSinceLastSnapshot = maxBytesSinceLastSnapshot;
            return this;
        }

        public Builder setMaxTimeSinceLastSnapshotNs(long maxTimeSinceLastSnapshotNs) {
            this.maxTimeSinceLastSnapshotNs = maxTimeSinceLastSnapshotNs;
            return this;
        }

        public Builder setDisabledReason(AtomicReference<String> disabledReason) {
            this.disabledReason = disabledReason;
            return this;
        }

        public Builder setThreadNamePrefix(String threadNamePrefix) {
            this.threadNamePrefix = threadNamePrefix;
            return this;
        }

        public SnapshotGenerator build() {
            if (this.disabledReason == null) {
                this.disabledReason = new AtomicReference();
            }
            return new SnapshotGenerator(this.nodeId, this.time, this.emitter, this.faultHandler, this.maxBytesSinceLastSnapshot, this.maxTimeSinceLastSnapshotNs, this.disabledReason, this.threadNamePrefix);
        }
    }
}

