/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.iotdb.commons.auth.entity;

import org.apache.iotdb.commons.path.PartialPath;
import org.apache.iotdb.commons.path.PathDeserializeUtil;
import org.apache.iotdb.commons.utils.AuthUtils;
import org.apache.iotdb.commons.utils.SerializeUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

/** This class represents a privilege on a specific seriesPath. */
public class PathPrivilege {
  private static final Logger LOGGER = LoggerFactory.getLogger(PathPrivilege.class);

  private static final int PATH_PRI_SIZE = PrivilegeType.getPathPriCount();
  private Set<Integer> privileges;

  // grantopt show whether the privileges can be grant to / revoke from others.
  // The privilege that can be grant to others must exist in privileges.
  // The set of grantopt must be a subset of privileges.
  private Set<Integer> grantOpts;
  private PartialPath path;

  public PathPrivilege() {
    // Empty constructor
  }

  public PathPrivilege(PartialPath path) {
    this.path = path;
    this.privileges = new HashSet<>();
    this.grantOpts = new HashSet<>();
  }

  public Set<Integer> getPrivileges() {
    return privileges;
  }

  public void setPrivileges(Set<Integer> privileges) {
    this.privileges = privileges;
  }

  public Set<Integer> getGrantOpt() {
    return grantOpts;
  }

  public void setGrantOpt(Set<Integer> grantOpts) {
    this.grantOpts = grantOpts;
  }

  public void grantPrivilege(int privilege, boolean grantOpt) {
    privileges.add(privilege);
    if (grantOpt) {
      grantOpts.add(privilege);
    }
  }

  public boolean revokePrivilege(int privilege) {
    if (!privileges.contains(privilege)) {
      LOGGER.warn("not find privilege{} on path {}", PrivilegeType.values()[privilege], path);
      return false;
    }
    privileges.remove(privilege);
    // when we revoke privilege from path, remove its grant option
    grantOpts.remove(privilege);
    return true;
  }

  public boolean revokeGrantOpt(int privilege) {
    if (!privileges.contains(privilege)) {
      LOGGER.warn("path {} dont have privilege {}", path, PrivilegeType.values()[privilege]);
      return false;
    }
    grantOpts.remove(privilege);
    return true;
  }

  public boolean checkPrivilege(int privilege) {
    if (privileges.contains(privilege)) {
      return true;
    }
    if (privilege == PrivilegeType.READ_DATA.ordinal()) {
      return privileges.contains(PrivilegeType.WRITE_DATA.ordinal());
    }
    if (privilege == PrivilegeType.READ_SCHEMA.ordinal()) {
      return privileges.contains(PrivilegeType.WRITE_SCHEMA.ordinal());
    }
    return false;
  }

  public void setAllPrivileges(int privs) {
    for (int i = 0; i < PATH_PRI_SIZE; i++) {
      if (((1 << i) & privs) != 0) {
        privileges.add(AuthUtils.pathPosToPri(i));
      }
      if ((1 << (i + 16) & privs) != 0) {
        grantOpts.add(AuthUtils.pathPosToPri(i));
      }
    }
  }

  public int getAllPrivileges() {
    int privilege = 0;
    for (Integer pri : privileges) {
      privilege |= 1 << AuthUtils.pathPriToPos(PrivilegeType.values()[pri]);
    }
    for (Integer grantOpt : grantOpts) {
      privilege |= 1 << (AuthUtils.pathPriToPos(PrivilegeType.values()[grantOpt]) + 16);
    }
    return privilege;
  }

  public PartialPath getPath() {
    return path;
  }

  public void setPath(PartialPath path) {
    this.path = path;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    PathPrivilege that = (PathPrivilege) o;
    return Objects.equals(privileges, that.privileges)
        && Objects.equals(path, that.path)
        && Objects.equals(grantOpts, this.grantOpts);
  }

  @Override
  public int hashCode() {
    return Objects.hash(privileges, grantOpts, path);
  }

  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder(path.getFullPath());
    builder.append(" :");
    for (Integer privilegeId : privileges) {
      builder.append(" ").append(PrivilegeType.values()[privilegeId]);
      if (grantOpts.contains(privilegeId)) {
        builder.append("_").append("with_grant_option");
      }
    }
    return builder.toString();
  }

  public ByteBuffer serialize() {
    ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
    DataOutputStream dataOutputStream = new DataOutputStream(byteArrayOutputStream);

    SerializeUtils.serializeIntSet(privileges, dataOutputStream);
    SerializeUtils.serializeIntSet(grantOpts, dataOutputStream);
    try {
      path.serialize(dataOutputStream);
    } catch (IOException exception) {
      LOGGER.error("Unexpected exception when serialize path", exception);
    }
    return ByteBuffer.wrap(byteArrayOutputStream.toByteArray());
  }

  public void deserialize(ByteBuffer buffer) {
    privileges = new HashSet<>();
    SerializeUtils.deserializeIntSet(privileges, buffer);
    grantOpts = new HashSet<>();
    SerializeUtils.deserializeIntSet(grantOpts, buffer);
    path = (PartialPath) PathDeserializeUtil.deserialize(buffer);
  }
}
