/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.seata.saga.engine.serializer.impl;

import org.apache.seata.saga.engine.serializer.Serializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

/**
 * Exception serializer
 *
 */
public class ExceptionSerializer implements Serializer<Exception, byte[]> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExceptionSerializer.class);

    public static byte[] serializeByObjectOutput(Object o) {

        byte[] result = null;
        if (o != null) {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            try (ObjectOutputStream oos = new ObjectOutputStream(baos)) {
                oos.writeObject(o);
                oos.flush();
                result = baos.toByteArray();
            } catch (IOException e) {
                LOGGER.error("serializer failed: {}", o.getClass(), e);
                throw new RuntimeException("IO Create Error", e);
            }
        }
        return result;
    }

    public static <T> T deserializeByObjectInputStream(byte[] bytes, Class<T> valueType) {

        if (bytes == null) {
            return null;
        }

        Object result = deserializeByObjectInputStream(bytes);
        return valueType.cast(result);
    }

    public static Object deserializeByObjectInputStream(byte[] bytes) {

        Object result = null;
        if (bytes != null) {
            ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
            try (ObjectInputStream ois = new ObjectInputStream(bais)) {
                result = ois.readObject();
            } catch (IOException e) {
                LOGGER.error("deserialize failed:", e);
                throw new RuntimeException("IO Create Error", e);
            } catch (ClassNotFoundException e) {
                LOGGER.error("deserialize failed:", e);
                throw new RuntimeException("Cannot find specified class", e);
            }
        }
        return result;
    }

    @Override
    public byte[] serialize(Exception object) {

        return serializeByObjectOutput(object);
    }

    @Override
    public Exception deserialize(byte[] bytes) {
        return deserializeByObjectInputStream(bytes, Exception.class);
    }
}
