/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gravitino.trino.connector.catalog;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.Connector;
import io.trino.spi.connector.ConnectorContext;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.lang3.NotImplementedException;
import org.apache.gravitino.Catalog;
import org.apache.gravitino.client.GravitinoAdminClient;
import org.apache.gravitino.client.GravitinoMetalake;
import org.apache.gravitino.exceptions.NoSuchMetalakeException;
import org.apache.gravitino.trino.connector.GravitinoConfig;
import org.apache.gravitino.trino.connector.GravitinoErrorCode;
import org.apache.gravitino.trino.connector.catalog.CatalogConnectorContext;
import org.apache.gravitino.trino.connector.catalog.CatalogConnectorFactory;
import org.apache.gravitino.trino.connector.catalog.CatalogRegister;
import org.apache.gravitino.trino.connector.metadata.GravitinoCatalog;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CatalogConnectorManager {
    private static final Logger LOG = LoggerFactory.getLogger(CatalogConnectorManager.class);
    private static final int CATALOG_LOAD_FREQUENCY_SECOND = 10;
    private static final int NUMBER_EXECUTOR_THREAD = 1;
    private static final int LOAD_METALAKE_TIMEOUT = 60;
    private final ScheduledExecutorService executorService;
    private final CatalogRegister catalogRegister;
    private final CatalogConnectorFactory catalogConnectorFactory;
    private final ConcurrentHashMap<String, CatalogConnectorContext> catalogConnectors = new ConcurrentHashMap();
    private final Set<String> usedMetalakes = new HashSet<String>();
    private final Map<String, GravitinoMetalake> metalakes = new ConcurrentHashMap<String, GravitinoMetalake>();
    private GravitinoAdminClient gravitinoClient;
    private GravitinoConfig config;

    public CatalogConnectorManager(CatalogRegister catalogRegister, CatalogConnectorFactory catalogFactory) {
        this.catalogRegister = catalogRegister;
        this.catalogConnectorFactory = catalogFactory;
        this.executorService = CatalogConnectorManager.createScheduledThreadPoolExecutor();
    }

    private static ScheduledThreadPoolExecutor createScheduledThreadPoolExecutor() {
        return new ScheduledThreadPoolExecutor(1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("gravitino-connector-schedule-%d").setUncaughtExceptionHandler((thread, throwable) -> LOG.warn("{} uncaught exception:", (Object)thread.getName(), (Object)throwable)).build());
    }

    public void config(GravitinoConfig config, GravitinoAdminClient client) {
        this.config = (GravitinoConfig)Preconditions.checkNotNull((Object)config, (Object)"config is not null");
        this.gravitinoClient = client == null ? GravitinoAdminClient.builder((String)config.getURI()).build() : client;
    }

    public void start(ConnectorContext context) throws Exception {
        this.catalogRegister.init(context, this.config);
        if (this.catalogRegister.isCoordinator()) {
            this.executorService.scheduleWithFixedDelay(this::loadMetalake, 10L, 10L, TimeUnit.SECONDS);
        }
        LOG.info("Gravitino CatalogConnectorManager started.");
    }

    private void loadMetalake() {
        try {
            if (!this.catalogRegister.isTrinoStarted()) {
                LOG.info("Waiting for the Trino started.");
                return;
            }
            for (String usedMetalake : this.usedMetalakes) {
                try {
                    GravitinoMetalake metalake = this.metalakes.computeIfAbsent(usedMetalake, this::retrieveMetalake);
                    LOG.debug("Load metalake: {}", (Object)usedMetalake);
                    this.loadCatalogs(metalake);
                }
                catch (Exception e) {
                    LOG.error("Load Metalake {} failed.", (Object)usedMetalake, (Object)e);
                }
            }
        }
        catch (Exception e) {
            LOG.error("Error when loading metalake", (Throwable)e);
            System.exit(-1);
        }
    }

    private GravitinoMetalake retrieveMetalake(String metalakeName) {
        try {
            return this.gravitinoClient.loadMetalake(metalakeName);
        }
        catch (NoSuchMetalakeException e) {
            throw new TrinoException((ErrorCodeSupplier)GravitinoErrorCode.GRAVITINO_METALAKE_NOT_EXISTS, "Metalake " + metalakeName + " not exists.");
        }
    }

    private void loadCatalogs(GravitinoMetalake metalake) {
        Object[] catalogNames;
        try {
            catalogNames = metalake.listCatalogs();
        }
        catch (Exception e) {
            LOG.error("Failed to list catalogs in metalake {}.", (Object)metalake.name(), (Object)e);
            return;
        }
        LOG.debug("Load metalake {}'s catalogs. catalogs: {}.", (Object)metalake.name(), (Object)Arrays.toString(catalogNames));
        Set catalogNameStrings = Arrays.stream(catalogNames).map(id -> this.config.simplifyCatalogNames() ? id : this.getTrinoCatalogName(metalake.name(), (String)id)).collect(Collectors.toSet());
        for (Map.Entry<String, CatalogConnectorContext> entry : this.catalogConnectors.entrySet()) {
            if (catalogNameStrings.contains(entry.getKey()) || !entry.getValue().getMetalake().name().equals(metalake.name())) continue;
            try {
                this.unloadCatalog(entry.getValue().getCatalog());
            }
            catch (Exception e) {
                LOG.error("Failed to remove catalog {}.", (Object)entry.getKey(), (Object)e);
            }
        }
        Arrays.stream(catalogNames).forEach(catalogName -> {
            try {
                Catalog catalog = metalake.loadCatalog(catalogName);
                GravitinoCatalog gravitinoCatalog = new GravitinoCatalog(metalake.name(), catalog);
                if (this.catalogConnectors.containsKey(this.getTrinoCatalogName(gravitinoCatalog))) {
                    this.reloadCatalog(gravitinoCatalog);
                } else if (catalog.type() == Catalog.Type.RELATIONAL) {
                    this.loadCatalog(gravitinoCatalog);
                }
            }
            catch (Exception e) {
                LOG.error("Failed to load metalake {}'s catalog {}.", new Object[]{metalake.name(), catalogName, e});
            }
        });
    }

    private void reloadCatalog(GravitinoCatalog catalog) {
        String catalogFullName = this.getTrinoCatalogName(catalog);
        GravitinoCatalog oldCatalog = this.catalogConnectors.get(catalogFullName).getCatalog();
        if (catalog.getLastModifiedTime() <= oldCatalog.getLastModifiedTime()) {
            return;
        }
        this.catalogRegister.unregisterCatalog(catalogFullName);
        this.catalogConnectors.remove(catalogFullName);
        this.loadCatalogImpl(catalog);
        LOG.info("Update catalog '{}' in metalake {} successfully.", (Object)catalog, (Object)catalog.getMetalake());
    }

    private void loadCatalog(GravitinoCatalog catalog) {
        this.loadCatalogImpl(catalog);
        LOG.info("Load catalog {} in metalake {} successfully.", (Object)catalog, (Object)catalog.getMetalake());
    }

    private void loadCatalogImpl(GravitinoCatalog catalog) {
        try {
            this.catalogRegister.registerCatalog(this.getTrinoCatalogName(catalog), catalog);
        }
        catch (Exception e) {
            String message = String.format("Failed to create internal catalog connector. The catalog is: %s", catalog);
            LOG.error(message, (Throwable)e);
            throw new TrinoException((ErrorCodeSupplier)GravitinoErrorCode.GRAVITINO_CREATE_INTERNAL_CONNECTOR_ERROR, message, (Throwable)e);
        }
    }

    private void unloadCatalog(GravitinoCatalog catalog) {
        String catalogFullName = this.getTrinoCatalogName(catalog);
        this.catalogRegister.unregisterCatalog(catalogFullName);
        this.catalogConnectors.remove(catalogFullName);
        LOG.info("Remove catalog '{}' in metalake {} successfully.", (Object)catalog.getName(), (Object)catalog.getMetalake());
    }

    public CatalogConnectorContext getCatalogConnector(String catalogName) {
        return this.catalogConnectors.get(catalogName);
    }

    public boolean catalogConnectorExist(String catalogName) {
        return this.catalogConnectors.containsKey(catalogName);
    }

    public List<GravitinoCatalog> getCatalogs() {
        return this.catalogConnectors.values().stream().map(CatalogConnectorContext::getCatalog).toList();
    }

    public void shutdown() {
        LOG.info("Gravitino CatalogConnectorManager shutdown.");
        throw new NotImplementedException();
    }

    public String getTrinoCatalogName(String metalake, String catalog) {
        return this.config.simplifyCatalogNames() ? catalog : String.format("\"%s.%s\"", metalake, catalog);
    }

    public String getTrinoCatalogName(GravitinoCatalog catalog) {
        return this.getTrinoCatalogName(catalog.getMetalake(), catalog.getName());
    }

    public void addMetalake(String metalake) {
        if (this.config.simplifyCatalogNames() && this.usedMetalakes.size() > 1) {
            throw new TrinoException((ErrorCodeSupplier)GravitinoErrorCode.GRAVITINO_MISSING_CONFIG, "Multiple metalakes are not supported when setting gravitino.simplify-catalog-names = true");
        }
        this.usedMetalakes.add(metalake);
    }

    public Set<String> getUsedMetalakes() {
        return this.usedMetalakes;
    }

    public Connector createConnector(String connectorName, GravitinoConfig config, ConnectorContext context) {
        try {
            String catalogConfig = config.getCatalogConfig();
            GravitinoCatalog catalog = GravitinoCatalog.fromJson(catalogConfig);
            CatalogConnectorContext.Builder builder = this.catalogConnectorFactory.createCatalogConnectorContextBuilder(catalog);
            builder.withMetalake(this.metalakes.computeIfAbsent(catalog.getMetalake(), this::retrieveMetalake)).withContext(context);
            CatalogConnectorContext connectorContext = builder.build();
            this.catalogConnectors.put(connectorName, connectorContext);
            LOG.info("Create connector {} successful", (Object)connectorName);
            return connectorContext.getConnector();
        }
        catch (Exception e) {
            LOG.error("Failed to create connector: {}", (Object)connectorName, (Object)e);
            throw new TrinoException((ErrorCodeSupplier)GravitinoErrorCode.GRAVITINO_OPERATION_FAILED, "Failed to create connector: " + connectorName, (Throwable)e);
        }
    }

    public void loadMetalakeSync() throws Exception {
        Future<?> future = this.executorService.submit(this::loadMetalake);
        future.get(60L, TimeUnit.SECONDS);
    }

    public GravitinoMetalake getMetalake(String metalake) {
        if (!this.usedMetalakes.contains(metalake)) {
            throw new TrinoException((ErrorCodeSupplier)GravitinoErrorCode.GRAVITINO_OPERATION_FAILED, "This connector does not allowed to access metalake " + metalake);
        }
        return this.metalakes.computeIfAbsent(metalake, this::retrieveMetalake);
    }
}

