/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.puffin;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.iceberg.Metrics;
import org.apache.iceberg.io.FileAppender;
import org.apache.iceberg.io.IOUtil;
import org.apache.iceberg.io.OutputFile;
import org.apache.iceberg.io.PositionOutputStream;
import org.apache.iceberg.puffin.Blob;
import org.apache.iceberg.puffin.BlobMetadata;
import org.apache.iceberg.puffin.FileMetadata;
import org.apache.iceberg.puffin.FileMetadataParser;
import org.apache.iceberg.puffin.PuffinCompressionCodec;
import org.apache.iceberg.puffin.PuffinFormat;
import org.apache.iceberg.relocated.com.google.common.base.MoreObjects;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMap;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;

public class PuffinWriter
implements FileAppender<Blob> {
    private static final byte[] MAGIC = PuffinFormat.getMagic();
    private final PositionOutputStream outputStream;
    private final Map<String, String> properties;
    private final PuffinCompressionCodec footerCompression;
    private final PuffinCompressionCodec defaultBlobCompression;
    private final List<BlobMetadata> writtenBlobsMetadata = Lists.newArrayList();
    private boolean headerWritten;
    private boolean finished;
    private Optional<Integer> footerSize = Optional.empty();
    private Optional<Long> fileSize = Optional.empty();

    PuffinWriter(OutputFile outputFile, Map<String, String> properties, boolean compressFooter, PuffinCompressionCodec defaultBlobCompression) {
        Preconditions.checkNotNull((Object)outputFile, (Object)"outputFile is null");
        Preconditions.checkNotNull(properties, (Object)"properties is null");
        Preconditions.checkNotNull((Object)((Object)defaultBlobCompression), (Object)"defaultBlobCompression is null");
        this.outputStream = outputFile.create();
        this.properties = ImmutableMap.copyOf(properties);
        this.footerCompression = compressFooter ? PuffinFormat.FOOTER_COMPRESSION_CODEC : PuffinCompressionCodec.NONE;
        this.defaultBlobCompression = defaultBlobCompression;
    }

    public void add(Blob blob) {
        Preconditions.checkNotNull((Object)blob, (Object)"blob is null");
        this.checkNotFinished();
        try {
            this.writeHeaderIfNeeded();
            long fileOffset = this.outputStream.getPos();
            PuffinCompressionCodec codec = (PuffinCompressionCodec)((Object)MoreObjects.firstNonNull((Object)((Object)blob.requestedCompression()), (Object)((Object)this.defaultBlobCompression)));
            ByteBuffer rawData = PuffinFormat.compress(codec, blob.blobData());
            int length = rawData.remaining();
            IOUtil.writeFully((OutputStream)this.outputStream, rawData);
            this.writtenBlobsMetadata.add(new BlobMetadata(blob.type(), blob.inputFields(), blob.snapshotId(), blob.sequenceNumber(), fileOffset, length, codec.codecName(), blob.properties()));
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public Metrics metrics() {
        return new Metrics();
    }

    public long length() {
        return this.fileSize();
    }

    public void close() throws IOException {
        if (!this.finished) {
            this.finish();
        }
        this.outputStream.close();
    }

    private void writeHeaderIfNeeded() throws IOException {
        if (this.headerWritten) {
            return;
        }
        this.outputStream.write(MAGIC);
        this.headerWritten = true;
    }

    public void finish() throws IOException {
        this.checkNotFinished();
        this.writeHeaderIfNeeded();
        Preconditions.checkState((!this.footerSize.isPresent() ? 1 : 0) != 0, (Object)"footerSize already set");
        long footerOffset = this.outputStream.getPos();
        this.writeFooter();
        this.footerSize = Optional.of(Math.toIntExact(this.outputStream.getPos() - footerOffset));
        this.fileSize = Optional.of(this.outputStream.getPos());
        this.finished = true;
    }

    private void writeFooter() throws IOException {
        FileMetadata fileMetadata = new FileMetadata(this.writtenBlobsMetadata, this.properties);
        ByteBuffer footerJson = ByteBuffer.wrap(FileMetadataParser.toJson(fileMetadata, false).getBytes(StandardCharsets.UTF_8));
        ByteBuffer footerPayload = PuffinFormat.compress(this.footerCompression, footerJson);
        this.outputStream.write(MAGIC);
        int footerPayloadLength = footerPayload.remaining();
        IOUtil.writeFully((OutputStream)this.outputStream, footerPayload);
        PuffinFormat.writeIntegerLittleEndian((OutputStream)this.outputStream, footerPayloadLength);
        this.writeFlags();
        this.outputStream.write(MAGIC);
    }

    private void writeFlags() throws IOException {
        Map<Integer, List<PuffinFormat.Flag>> flagsByByteNumber = this.fileFlags().stream().collect(Collectors.groupingBy(PuffinFormat.Flag::byteNumber));
        for (int byteNumber = 0; byteNumber < 4; ++byteNumber) {
            int byteFlag = 0;
            for (PuffinFormat.Flag flag : flagsByByteNumber.getOrDefault(byteNumber, (List<PuffinFormat.Flag>)ImmutableList.of())) {
                byteFlag |= 1 << flag.bitNumber();
            }
            this.outputStream.write(byteFlag);
        }
    }

    public long footerSize() {
        return this.footerSize.orElseThrow(() -> new IllegalStateException("Footer not written yet")).intValue();
    }

    public long fileSize() {
        return this.fileSize.orElseThrow(() -> new IllegalStateException("File not written yet"));
    }

    public List<BlobMetadata> writtenBlobsMetadata() {
        return ImmutableList.copyOf(this.writtenBlobsMetadata);
    }

    private Set<PuffinFormat.Flag> fileFlags() {
        EnumSet<PuffinFormat.Flag> flags = EnumSet.noneOf(PuffinFormat.Flag.class);
        if (this.footerCompression != PuffinCompressionCodec.NONE) {
            flags.add(PuffinFormat.Flag.FOOTER_PAYLOAD_COMPRESSED);
        }
        return flags;
    }

    private void checkNotFinished() {
        Preconditions.checkState((!this.finished ? 1 : 0) != 0, (Object)"Writer already finished");
    }
}

