/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gravitino.authorization;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Scheduler;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.gravitino.Config;
import org.apache.gravitino.Configs;
import org.apache.gravitino.Entity;
import org.apache.gravitino.EntityAlreadyExistsException;
import org.apache.gravitino.EntityStore;
import org.apache.gravitino.NameIdentifier;
import org.apache.gravitino.authorization.AuthorizationUtils;
import org.apache.gravitino.authorization.SecurableObject;
import org.apache.gravitino.exceptions.NoSuchEntityException;
import org.apache.gravitino.exceptions.NoSuchRoleException;
import org.apache.gravitino.exceptions.RoleAlreadyExistsException;
import org.apache.gravitino.meta.AuditInfo;
import org.apache.gravitino.meta.RoleEntity;
import org.apache.gravitino.storage.IdGenerator;
import org.apache.gravitino.utils.PrincipalUtils;
import org.glassfish.jersey.internal.guava.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RoleManager {
    private static final Logger LOG = LoggerFactory.getLogger(RoleManager.class);
    private final EntityStore store;
    private final IdGenerator idGenerator;
    private final Cache<NameIdentifier, RoleEntity> cache;

    RoleManager(EntityStore store, IdGenerator idGenerator, Config config) {
        this.store = store;
        this.idGenerator = idGenerator;
        long cacheEvictionIntervalInMs = config.get(Configs.ROLE_CACHE_EVICTION_INTERVAL_MS);
        this.cache = Caffeine.newBuilder().expireAfterAccess(cacheEvictionIntervalInMs, TimeUnit.MILLISECONDS).removalListener((k, v, c) -> LOG.info("Remove role {} from the cache.", k)).scheduler(Scheduler.forScheduledExecutorService((ScheduledExecutorService)new ScheduledThreadPoolExecutor(1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("role-cleaner-%d").build()))).build();
    }

    RoleEntity createRole(String metalake, String role, Map<String, String> properties, List<SecurableObject> securableObjects) throws RoleAlreadyExistsException {
        AuthorizationUtils.checkMetalakeExists(metalake);
        RoleEntity roleEntity = RoleEntity.builder().withId(this.idGenerator.nextId()).withName(role).withProperties(properties).withSecurableObjects(securableObjects).withNamespace(AuthorizationUtils.ofRoleNamespace(metalake)).withAuditInfo(AuditInfo.builder().withCreator(PrincipalUtils.getCurrentPrincipal().getName()).withCreateTime(Instant.now()).build()).build();
        try {
            this.store.put(roleEntity, false);
            this.cache.put((Object)roleEntity.nameIdentifier(), (Object)roleEntity);
            AuthorizationUtils.callAuthorizationPluginForSecurableObjects(metalake, roleEntity.securableObjects(), Sets.newHashSet(), authorizationPlugin -> authorizationPlugin.onRoleCreated(roleEntity));
            return roleEntity;
        }
        catch (EntityAlreadyExistsException e) {
            LOG.warn("Role {} in the metalake {} already exists", new Object[]{role, metalake, e});
            throw new RoleAlreadyExistsException("Role %s in the metalake %s already exists", new Object[]{role, metalake});
        }
        catch (IOException ioe) {
            LOG.error("Creating role {} failed in the metalake {} due to storage issues", new Object[]{role, metalake, ioe});
            throw new RuntimeException(ioe);
        }
    }

    RoleEntity getRole(String metalake, String role) throws NoSuchRoleException {
        try {
            AuthorizationUtils.checkMetalakeExists(metalake);
            return this.getRoleEntity(AuthorizationUtils.ofRole(metalake, role));
        }
        catch (NoSuchEntityException e) {
            LOG.warn("Role {} does not exist in the metalake {}", new Object[]{role, metalake, e});
            throw new NoSuchRoleException("Role %s does not exist in th metalake %s", new Object[]{role, metalake});
        }
    }

    boolean deleteRole(String metalake, String role) {
        try {
            AuthorizationUtils.checkMetalakeExists(metalake);
            NameIdentifier ident = AuthorizationUtils.ofRole(metalake, role);
            this.cache.invalidate((Object)ident);
            try {
                RoleEntity roleEntity = this.store.get(ident, Entity.EntityType.ROLE, RoleEntity.class);
                AuthorizationUtils.callAuthorizationPluginForSecurableObjects(metalake, roleEntity.securableObjects(), Sets.newHashSet(), authorizationPlugin -> authorizationPlugin.onRoleDeleted(roleEntity));
            }
            catch (NoSuchEntityException noSuchEntityException) {
                // empty catch block
            }
            return this.store.delete(ident, Entity.EntityType.ROLE);
        }
        catch (IOException ioe) {
            LOG.error("Deleting role {} in the metalake {} failed due to storage issues", new Object[]{role, metalake, ioe});
            throw new RuntimeException(ioe);
        }
    }

    private RoleEntity getRoleEntity(NameIdentifier identifier) {
        return (RoleEntity)this.cache.get((Object)identifier, id -> {
            try {
                return this.store.get(identifier, Entity.EntityType.ROLE, RoleEntity.class);
            }
            catch (IOException ioe) {
                LOG.error("Failed to get roles {} due to storage issues", (Object)identifier, (Object)ioe);
                throw new RuntimeException(ioe);
            }
        });
    }

    @VisibleForTesting
    Cache<NameIdentifier, RoleEntity> getCache() {
        return this.cache;
    }
}

