// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.impala.calcite.rel.node;

import org.apache.calcite.rel.RelNode;
import org.apache.calcite.rel.core.Aggregate;
import org.apache.calcite.rel.core.Filter;
import org.apache.calcite.rel.core.Join;
import org.apache.calcite.rel.core.Project;
import org.apache.calcite.rel.core.Sort;
import org.apache.calcite.rel.core.TableScan;
import org.apache.calcite.rel.core.Union;
import org.apache.calcite.rel.core.Values;
import org.apache.impala.common.ImpalaException;

/**
 * ImpalaPlanRel. Interface used for all Impala intermediary RelNodes
 */
public interface ImpalaPlanRel {

  /**
   * Enum representing the type of class used in the RelNode
   * Using an enum here so that Impala Plan RelNodes can be used in
   * a "case" statement rather than using multiple "instanceof" statements.
   * Also, the PROJECT enum is used for two different RelNodes, the normal
   * project and a Project containing analytical functions which needs to
   * be handled differently.
   */
  public enum RelNodeType {
    AGGREGATE,
    FILTER,
    HDFSSCAN,
    JOIN,
    PROJECT,
    SORT,
    UNION,
    VALUES
  }

  /**
   * getPlanNode returns a NodeWithExprs object, a thin structure containing
   * the plan node along with the output expressions generated by the plan node.
   */
  public NodeWithExprs getPlanNode(ParentPlanRelContext context) throws ImpalaException;

  /**
   * Return the RelNodeType enum of the implementing class of ImpalaPlanRel.
   */
  public RelNodeType relNodeType();

  public static RelNodeType getRelNodeType(RelNode relNode) {
    if (relNode instanceof Aggregate) {
      return RelNodeType.AGGREGATE;
    }
    if (relNode instanceof Filter) {
      return RelNodeType.FILTER;
    }
    if (relNode instanceof TableScan) {
      return RelNodeType.HDFSSCAN;
    }
    if (relNode instanceof Join) {
      return RelNodeType.JOIN;
    }
    if (relNode instanceof Project) {
      return RelNodeType.PROJECT;
    }
    if (relNode instanceof Sort) {
      return RelNodeType.SORT;
    }
    if (relNode instanceof Union) {
      return RelNodeType.UNION;
    }
    if (relNode instanceof Values) {
      return RelNodeType.VALUES;
    }
    throw new RuntimeException("Unknown RelNode: " +  relNode);
  }
}
