/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase;

import java.io.Closeable;
import java.io.IOException;
import java.util.Map;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.yetus.audience.InterfaceAudience;

/**
 * Get, remove and modify table descriptors.
 */
@InterfaceAudience.Private
public interface TableDescriptors extends Closeable {

  /**
   * Test whether a given table exists, i.e, has a table descriptor.
   */
  default boolean exists(TableName tableName) throws IOException {
    return get(tableName) != null;
  }

  @Override
  default void close() throws IOException {
    // do nothing by default
  }

  /** Returns TableDescriptor for tablename */
  TableDescriptor get(TableName tableName) throws IOException;

  /**
   * Get Map of all NamespaceDescriptors for a given namespace.
   * @return Map of all descriptors.
   */
  Map<String, TableDescriptor> getByNamespace(String name) throws IOException;

  /**
   * Get Map of all TableDescriptors. Populates the descriptor cache as a side effect.
   * </p>
   * Notice: the key of map is the table name which contains namespace. It was generated by
   * {@link TableName#getNameWithNamespaceInclAsString()}.
   * @return Map of all descriptors.
   */
  Map<String, TableDescriptor> getAll() throws IOException;

  /**
   * Add or update descriptor. Just call {@link #update(TableDescriptor, boolean)} with
   * {@code cacheOnly} as {@code false}.
   */
  default void update(TableDescriptor htd) throws IOException {
    update(htd, false);
  }

  /**
   * Add or update descriptor
   * @param htd       Descriptor to set into TableDescriptors
   * @param cacheOnly only add the given {@code htd} to cache, without updating the storage. For
   *                  example, when creating table, we will write the descriptor to fs when creating
   *                  the fs layout, so we do not need to update the fs again.
   */
  void update(TableDescriptor htd, boolean cacheOnly) throws IOException;

  /** Returns Instance of table descriptor or null if none found. */
  TableDescriptor remove(TableName tablename) throws IOException;
}
