/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.password;

import com.google.common.base.Charsets;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.io.Closer;
import com.google.common.io.LineReader;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.util.AbstractMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.gobblin.configuration.State;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.jasypt.util.text.BasicTextEncryptor;
import org.jasypt.util.text.StrongTextEncryptor;
import org.jasypt.util.text.TextEncryptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PasswordManager {
    private static final Logger LOG = LoggerFactory.getLogger(PasswordManager.class);
    private static final long CACHE_SIZE = 100L;
    private static final long CACHE_EXPIRATION_MIN = 10L;
    private static final Pattern PASSWORD_PATTERN = Pattern.compile("ENC\\((.*)\\)");
    private static final LoadingCache<Map.Entry<Optional<String>, Boolean>, PasswordManager> CACHED_INSTANCES = CacheBuilder.newBuilder().maximumSize(100L).expireAfterAccess(10L, TimeUnit.MINUTES).build((CacheLoader)new CacheLoader<Map.Entry<Optional<String>, Boolean>, PasswordManager>(){

        public PasswordManager load(Map.Entry<Optional<String>, Boolean> cacheKey) {
            return new PasswordManager(cacheKey.getKey(), cacheKey.getValue());
        }
    });
    private Optional<TextEncryptor> encryptor;

    private PasswordManager(Optional<String> masterPassword, boolean useStrongEncryptor) {
        if (masterPassword.isPresent()) {
            this.encryptor = useStrongEncryptor ? Optional.of((Object)new StrongTextEncryptor()) : Optional.of((Object)new BasicTextEncryptor());
            try {
                ((TextEncryptor)this.encryptor.get()).getClass().getMethod("setPassword", String.class).invoke(this.encryptor.get(), masterPassword.get());
            }
            catch (Exception e) {
                LOG.error("Failed to set master password for encryptor", (Throwable)e);
                this.encryptor = Optional.absent();
            }
        } else {
            this.encryptor = Optional.absent();
        }
    }

    public static PasswordManager getInstance() {
        try {
            Optional absent = Optional.absent();
            return (PasswordManager)CACHED_INSTANCES.get(new AbstractMap.SimpleEntry<Optional, Boolean>(absent, PasswordManager.shouldUseStrongEncryptor(new State())));
        }
        catch (ExecutionException e) {
            throw new RuntimeException("Unable to get an instance of PasswordManager", e);
        }
    }

    public static PasswordManager getInstance(State state) {
        try {
            return (PasswordManager)CACHED_INSTANCES.get(new AbstractMap.SimpleEntry<Optional<String>, Boolean>(PasswordManager.getMasterPassword(state), PasswordManager.shouldUseStrongEncryptor(state)));
        }
        catch (ExecutionException e) {
            throw new RuntimeException("Unable to get an instance of PasswordManager", e);
        }
    }

    public static PasswordManager getInstance(Properties props) {
        return PasswordManager.getInstance(new State(props));
    }

    public static PasswordManager getInstance(Path masterPwdLoc) {
        try {
            return (PasswordManager)CACHED_INSTANCES.get(new AbstractMap.SimpleEntry<Optional<String>, Boolean>(PasswordManager.getMasterPassword(masterPwdLoc), PasswordManager.shouldUseStrongEncryptor(new State())));
        }
        catch (ExecutionException e) {
            throw new RuntimeException("Unable to get an instance of PasswordManager", e);
        }
    }

    private static boolean shouldUseStrongEncryptor(State state) {
        return state.getPropAsBoolean("encrypt.use.strong.encryptor", false);
    }

    public String encryptPassword(String plain) {
        Preconditions.checkArgument((boolean)this.encryptor.isPresent(), (Object)"A master password needs to be provided for encrypting passwords.");
        try {
            return ((TextEncryptor)this.encryptor.get()).encrypt(plain);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to encrypt password", e);
        }
    }

    public String decryptPassword(String encrypted) {
        Preconditions.checkArgument((boolean)this.encryptor.isPresent(), (Object)"A master password needs to be provided for decrypting passwords.");
        try {
            return ((TextEncryptor)this.encryptor.get()).decrypt(encrypted);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to decrypt password " + encrypted, e);
        }
    }

    public String readPassword(String password) {
        if (password == null || !this.encryptor.isPresent()) {
            return password;
        }
        Matcher matcher = PASSWORD_PATTERN.matcher(password);
        if (matcher.find()) {
            return this.decryptPassword(matcher.group(1));
        }
        return password;
    }

    private static Optional<String> getMasterPassword(State state) {
        if (!state.contains("encrypt.key.loc")) {
            LOG.warn(String.format("Property %s not set. Cannot decrypt any encrypted password.", "encrypt.key.loc"));
            return Optional.absent();
        }
        try {
            if (state.contains("encrypt.key.fs.uri")) {
                FileSystem fs = FileSystem.get((URI)URI.create(state.getProp("encrypt.key.fs.uri")), (Configuration)new Configuration());
                return PasswordManager.getMasterPassword(fs, new Path(state.getProp("encrypt.key.loc")));
            }
            return PasswordManager.getMasterPassword(new Path(state.getProp("encrypt.key.loc")));
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to obtain master password from " + state.getProp("encrypt.key.loc"), e);
        }
    }

    public static Optional<String> getMasterPassword(Path masterPasswordFile) {
        try {
            FileSystem fs = masterPasswordFile.getFileSystem(new Configuration());
            return PasswordManager.getMasterPassword(fs, masterPasswordFile);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to obtain master password from " + masterPasswordFile, e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Optional<String> getMasterPassword(FileSystem fs, Path masterPasswordFile) {
        try (Closer closer = Closer.create();){
            if (!fs.exists(masterPasswordFile) || fs.getFileStatus(masterPasswordFile).isDirectory()) {
                LOG.warn(masterPasswordFile + " does not exist or is not a file. Cannot decrypt any encrypted password.");
                Optional optional = Optional.absent();
                return optional;
            }
            InputStream in = (InputStream)closer.register((Closeable)fs.open(masterPasswordFile));
            Optional optional = Optional.of((Object)new LineReader((Readable)new InputStreamReader(in, Charsets.UTF_8)).readLine());
            return optional;
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to obtain master password from " + masterPasswordFile, e);
        }
    }
}

